package carrier

import (
	"strconv"
	"strings"

	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/storage/carrier"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/logger"
)

type codeSystem string

const csIata codeSystem = "IATA"
const csIcao codeSystem = "ICAO"
const csSirena codeSystem = "SIRENA"
const csIcaoRu codeSystem = "ICAO_RU"

type ambiguousCodeKey struct {
	CodeSystem codeSystem
	Code       string
}

type ambiguousCodes map[ambiguousCodeKey][]int32

type ambiguousFlightNumberKey struct {
	ambiguousCodeKey
	Number           string
	InitialCarrierID int32
}

type ambiguousFlightNumbers map[ambiguousFlightNumberKey][]int32

type AmbiguityResponse struct {
	Codes   ambiguousCodesResponse
	Flights ambiguousFlightNumbersResponse
}

type ambiguousCodesResponse []ambiguousCodesResponseItem
type ambiguousFlightNumbersResponse []ambiguousFlightNumbersResponseItem

type ambiguousCodesResponseItem struct {
	ambiguousCodeKey
	IDs []int32
}

type ambiguousFlightNumbersResponseItem struct {
	ambiguousFlightNumberKey
	IDs []int32
}

type CarrierService struct {
	carriers      carriers
	iatacorrector iatacorrector

	ambiguousCodes         ambiguousCodes
	ambiguousFlightNumbers ambiguousFlightNumbers
}

type carriers interface {
	GetCarriersByCode(code string) []int32
	CarrierCodes() map[int32]carrier.CarrierCodes
}

type iatacorrector interface {
	FindCarrier(iataCode, sirenaCode, flightNumber, flyingCarrierIata string, designatedCarrierID int32) int32
}

func NewCarrierService(carriers carriers, iatacorrector iatacorrector) *CarrierService {
	cs := &CarrierService{
		carriers:               carriers,
		iatacorrector:          iatacorrector,
		ambiguousCodes:         make(ambiguousCodes),
		ambiguousFlightNumbers: make(ambiguousFlightNumbers),
	}
	cs.validateIataCorrectorLogic()
	return cs
}

func (c *CarrierService) GetCodeAmbiguityMap() AmbiguityResponse {
	r := AmbiguityResponse{}
	for k, v := range c.ambiguousCodes {
		r.Codes = append(r.Codes, ambiguousCodesResponseItem{
			ambiguousCodeKey: k,
			IDs:              v,
		})
	}
	for k, v := range c.ambiguousFlightNumbers {
		r.Flights = append(r.Flights, ambiguousFlightNumbersResponseItem{
			ambiguousFlightNumberKey: k,
			IDs:                      v,
		})
	}
	return r
}

func (c *CarrierService) validateIataCorrectorLogic() {
	logger.Logger().Info("Validating that iata corrector is not ambiguous")
	for carrierID, codes := range c.carriers.CarrierCodes() {
		if len(codes.Iata) > 0 {
			if ids := c.carriers.GetCarriersByCode(codes.Iata); len(ids) > 1 {
				c.ambiguousCodes[ambiguousCodeKey{csIata, codes.Iata}] = append([]int32{}, ids...)
			}
		}
		if len(codes.Sirena) > 0 {
			if ids := c.carriers.GetCarriersByCode(codes.Sirena); len(ids) > 1 {
				c.ambiguousCodes[ambiguousCodeKey{csSirena, codes.Sirena}] = append([]int32{}, ids...)
			}
		}
		if len(codes.Icao) > 0 {
			if ids := c.carriers.GetCarriersByCode(codes.Icao); len(ids) > 1 {
				c.ambiguousCodes[ambiguousCodeKey{csIcao, codes.Icao}] = append([]int32{}, ids...)
			}
		}
		if len(codes.IcaoRu) > 0 {
			if ids := c.carriers.GetCarriersByCode(codes.IcaoRu); len(ids) > 1 {
				c.ambiguousCodes[ambiguousCodeKey{csIcaoRu, codes.IcaoRu}] = append([]int32{}, ids...)
			}
		}
		for i := 100; i <= 9999; i += 99 {
			number := strconv.Itoa(i)
			if len(codes.Iata) > 0 {
				id1 := c.getCarrierByFlightNumber(codes.Iata + number).CarrierID
				id2 := c.getCarrierByFlightNumber(codes.Iata + " " + number).CarrierID
				if id1 != id2 {
					c.ambiguousFlightNumbers[ambiguousFlightNumberKey{
						ambiguousCodeKey: ambiguousCodeKey{
							CodeSystem: csIata,
							Code:       codes.Iata,
						},
						Number:           number,
						InitialCarrierID: carrierID,
					}] = []int32{id1, id2}
				}
			}
			if len(codes.Sirena) > 0 {
				id1 := c.getCarrierByFlightNumber(codes.Sirena + number).CarrierID
				id2 := c.getCarrierByFlightNumber(codes.Sirena + " " + number).CarrierID
				if id1 != id2 {
					c.ambiguousFlightNumbers[ambiguousFlightNumberKey{
						ambiguousCodeKey: ambiguousCodeKey{
							CodeSystem: csSirena,
							Code:       codes.Sirena,
						},
						Number:           number,
						InitialCarrierID: carrierID,
					}] = []int32{id1, id2}
				}
			}
			if len(codes.Icao) > 0 {
				id1 := c.getCarrierByFlightNumber(codes.Icao + number).CarrierID
				id2 := c.getCarrierByFlightNumber(codes.Icao + " " + number).CarrierID
				if id1 != id2 {
					c.ambiguousFlightNumbers[ambiguousFlightNumberKey{
						ambiguousCodeKey: ambiguousCodeKey{
							CodeSystem: csIcao,
							Code:       codes.Icao,
						},
						Number:           number,
						InitialCarrierID: carrierID,
					}] = []int32{id1, id2}
				}
			}
			if len(codes.IcaoRu) > 0 {
				id1 := c.getCarrierByFlightNumber(codes.IcaoRu + number).CarrierID
				id2 := c.getCarrierByFlightNumber(codes.IcaoRu + " " + number).CarrierID
				if id1 != id2 {
					c.ambiguousFlightNumbers[ambiguousFlightNumberKey{
						ambiguousCodeKey: ambiguousCodeKey{
							CodeSystem: csIcaoRu,
							Code:       codes.IcaoRu,
						},
						Number:           number,
						InitialCarrierID: carrierID,
					}] = []int32{id1, id2}
				}
			}
		}
	}
}

func (c *CarrierService) GetCarriersByFlightNumbers(flightNumbers []string) map[string]*Flight {
	output := make(map[string]*Flight, len(flightNumbers))
	for _, fn := range flightNumbers {
		flight := c.getCarrierByFlightNumber(fn)
		if flight.CarrierID == 0 {
			output[fn] = nil
			continue
		}
		output[fn] = &flight
	}
	return output
}

type Flight struct {
	CarrierID int32

	CarrierCode, FlightNumber string
}

func (c *CarrierService) getCarrierByFlightNumber(flightString string) Flight {
	flightString = strings.TrimSpace(flightString)
	flightString = strings.ToUpper(flightString)

	flightRune := []rune(flightString)

	if len(flightRune) <= 2 {
		return Flight{}
	}

	var carrierCode, flightNumber string

	if strings.ContainsRune(flightString, ' ') {
		flightSplit := strings.Fields(flightString)
		carrierCode, flightNumber = flightSplit[0], flightSplit[1]
		return c.getFlightByCodeAndFlightNumber(carrierCode, flightNumber)
	}
	if len(flightRune) >= 3 {
		carrierID := c.getFlightByCodeAndFlightNumber(string(flightRune[:3]), string(flightRune[3:]))
		if carrierID.CarrierID != 0 {
			return carrierID
		}
	}
	return c.getFlightByCodeAndFlightNumber(string(flightRune[:2]), string(flightRune[2:]))
}

func (c *CarrierService) GetCarrierByCodeAndFlightNumber(carrierCode, flightNumber string) int32 {
	return c.getFlightByCodeAndFlightNumber(carrierCode, flightNumber).CarrierID
}

func (c *CarrierService) getFlightByCodeAndFlightNumber(carrierCode, flightNumber string) Flight {
	correctedCarrierID := c.iatacorrector.FindCarrier(
		carrierCode, "", flightNumber, "", 0,
	)
	if correctedCarrierID > 0 {
		return Flight{correctedCarrierID, carrierCode, flightNumber}
	}

	correctedCarrierID = c.iatacorrector.FindCarrier(
		"", carrierCode, flightNumber, "", 0,
	)
	if correctedCarrierID > 0 {
		return Flight{correctedCarrierID, carrierCode, flightNumber}
	}

	carrierIDs := c.carriers.GetCarriersByCode(carrierCode)
	if len(carrierIDs) == 0 {
		return Flight{}
	}
	return Flight{carrierIDs[0], carrierCode, flightNumber}
}
