package flight

import (
	"fmt"
	"net/http"
	"sort"
	"strconv"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/appconst"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/services/flightdata"
	dto "a.yandex-team.ru/travel/avia/shared_flights/api/internal/services/storage/DTO"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/services/storage/flightstatus"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/services/storage/timezone"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/storage"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/utils"
	"a.yandex-team.ru/travel/avia/shared_flights/api/pkg/structs"
	dir "a.yandex-team.ru/travel/avia/shared_flights/lib/go/direction"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/dtutil"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/logger"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/math"
	"a.yandex-team.ru/travel/proto/shared_flights/snapshots"
)

var ErrorDate = time.Date(2000, 1, 1, 0, 0, 0, 0, time.UTC)

type FlightParam struct {
	CarrierCode   string
	FlightNumber  string
	DepartureDate string
}

type FlightService interface {
	GetFlightRange(
		carrierParam CarrierParam,
		flightNumber string,
		nationalVersion string,
		showBanned bool,
		nowDate time.Time,
		limitBefore int,
		limitAfter int,
		direction dir.Direction,
		referenceTime time.Time,
	) (response []*FlightSegment, err error)
	GetFlightRangeMulti(
		flightNumber string,
		nationalVersion string,
		showBanned bool,
		nowDate time.Time,
		limitBefore int,
		limitAfter int,
		direction dir.Direction,
		referenceTime time.Time,
	) (response map[int32][]*FlightSegment, err error)
	GetFlight(
		carrierText, flightNumber, departureDateString string,
		fromStation *snapshots.TStationWithCodes,
		nationalVersion string,
		showBanned bool,
		nowDate time.Time,
		includeLooseSegments bool,
		includeCodeshares bool,
	) (response *FlightSegment, err error)
	GetFlights(
		flights []FlightParam,
		nationalVersion string,
		showBanned bool,
		nowDate time.Time,
		includeCodeshares bool,
	) (response []*FlightSegment)
	GetFlightData(
		carrierParam CarrierParam,
		flightNumber string,
		departureDateString string,
		fromStation *snapshots.TStationWithCodes,
		nationalVersion string,
		showBanned bool,
		includeSingleSegments bool,
		includeCodeshares bool,
	) (response []*flightdata.FlightData, err error)
	GetFlightPatterns(carrierParam CarrierParam, flightNumber string) (response []*structs.FlightPattern, err error)
}

type CarrierService interface {
	GetCarrierByCodeAndFlightNumber(carrierCode, flightNumber string) int32
}

func NewFlightService(storage *storage.Storage, timeZoneUtil timezone.TimeZoneUtil, carrierService CarrierService) FlightService {
	return &flightServiceImpl{
		Storage:        storage,
		TimeZoneUtil:   timeZoneUtil,
		CarrierService: carrierService,
	}
}

type flightServiceImpl struct {
	*storage.Storage
	timezone.TimeZoneUtil
	CarrierService
}

// This is what's printed into response
type FlightSegment struct {
	CompanyIata       string                    `json:"airlineCode"`
	CompanyRaspID     int32                     `json:"airlineID"`
	Number            string                    `json:"number"`
	AirportFromIata   string                    `json:"airportFromCode"`
	AirportFromRaspID int64                     `json:"airportFromID"`
	DepartureDay      string                    `json:"departureDay"`
	DepartureTime     string                    `json:"departureTime"`
	DepartureTzName   string                    `json:"departureTimezone"`
	DepartureUTC      string                    `json:"departureUtc"`
	DepartureTerminal string                    `json:"departureTerminal"`
	AirportToIata     string                    `json:"airportToCode"`
	AirportToRaspID   int64                     `json:"airportToID"`
	ArrivalDay        string                    `json:"arrivalDay"`
	ArrivalTime       string                    `json:"arrivalTime"`
	ArrivalTzName     string                    `json:"arrivalTimezone"`
	ArrivalUTC        string                    `json:"arrivalUtc"`
	ArrivalTerminal   string                    `json:"arrivalTerminal"`
	CreatedAtUTC      string                    `json:"createdAtUtc"`
	UpdatedAtUTC      string                    `json:"updatedAtUtc"`
	Status            flightstatus.FlightStatus `json:"status,omitempty"`
	Segments          []*FlightSegment          `json:"segments,omitempty"`
	Source            string                    `json:"source"`
	Title             string                    `json:"title,omitempty"`
	Banned            string                    `json:"banned,omitempty"`
	Operating         *dto.TitledFlight         `json:"operating"`
	TransportModelID  int64                     `json:"transportModelID"`
	FlightCodeshares  []dto.TitledFlight        `json:"-"` // Only filled in if requested
}

type FlightSegmentWithCodeshares struct {
	FlightSegment
	Codeshares []dto.TitledFlight `json:"codeshares"`
}

/*
   Returns matching flight departing on the specified date.
   includeLooseSegments: for multi-segment flights, return first matching segment even if the first segment of the flight
                         departs on a different (previous) day
*/
func (service *flightServiceImpl) GetFlight(
	carrierText, flightNumber, departureDateString string,
	fromStation *snapshots.TStationWithCodes,
	nationalVersion string,
	showBanned bool,
	nowDate time.Time,
	includeLooseSegments bool,
	includeCodeshares bool,
) (response *FlightSegment, err error) {
	flightData, err := service.GetFlightData(
		NewCarrierParamByText(carrierText),
		flightNumber,
		departureDateString,
		fromStation,
		nationalVersion,
		showBanned,
		includeLooseSegments,
		includeCodeshares,
	)
	if err != nil {
		return nil, err
	}

	if len(flightData) < 2 {
		flight, err := Convert(flightData, nowDate, service.GetLastImported())
		if err != nil {
			return nil, err
		}
		return flight, nil
	}

	// Process the case when the same leg is flown from 2 different stations
	legsMap := make(map[int32]*flightdata.FlightData)
	maxLegNum := int32(0)
	for _, fd := range flightData {
		legsMap[fd.FlightBase.LegNumber] = fd
		if maxLegNum < fd.FlightBase.LegNumber {
			maxLegNum = fd.FlightBase.LegNumber
		}
	}

	currentFlight := make([]*flightdata.FlightData, 0)
	for legNum := int32(1); legNum <= maxLegNum; legNum++ {
		leg, ok := legsMap[legNum]
		if !ok || leg == nil {
			return nil, xerrors.Errorf("missing leg %v for flight %v %v", legNum, carrierText, flightNumber)
		}
		currentFlight = append(currentFlight, leg)
	}

	return Convert(currentFlight, nowDate, service.GetLastImported())
}

func (service *flightServiceImpl) GetFlights(
	flights []FlightParam,
	nationalVersion string,
	showBanned bool,
	nowDate time.Time,
	includeCodeshares bool,
) (response []*FlightSegment) {
	response = make([]*FlightSegment, 0)
	for _, flight := range flights {
		flightData, err := service.GetFlight(
			flight.CarrierCode,
			flight.FlightNumber,
			flight.DepartureDate,
			nil,
			nationalVersion,
			showBanned,
			nowDate,
			true,
			includeCodeshares,
		)

		if err != nil {
			logger.Logger().Debug("Error while retrieving flight", log.Reflect("flight", flight), log.Error(err))
			continue
		}
		response = append(response, flightData)
	}

	return
}

const EstimateMaxFlightDuration = 10

/*
  Returns list of matching flights departing on the specified date.
  includeLooseSegments: for multi-segment flights, return first matching segment even if the first segment of the flight
                        departs on a different (previous) day
*/
func (service *flightServiceImpl) GetFlightData(
	carrierParam CarrierParam,
	flightNumber string,
	departureDateString string,
	fromStation *snapshots.TStationWithCodes,
	nationalVersion string,
	showBanned bool,
	includeLooseSegments bool,
	includeCodeshares bool,
) (response []*flightdata.FlightData, err error) {
	defer func() {
		if r := recover(); r != nil {
			response = nil
			err = &utils.ErrorWithHTTPCode{
				HTTPCode:     http.StatusInternalServerError,
				ErrorMessage: xerrors.Errorf("getFlightData panicked: %v", r).Error(),
			}
			return
		}
	}()

	departureDate := dtutil.StringDate(departureDateString).ToIntDate()

	if departureDate == 0 {
		return nil, &utils.ErrorWithHTTPCode{
			HTTPCode:     http.StatusBadRequest,
			ErrorMessage: fmt.Sprintf("Unable to parse departure date %v", departureDateString),
		}
	}

	// Assumption: if there's more than one carrier with the same IATA code, it is good enough to return the first
	// matching flight. It is an extremely rare case and we don't have any clue how to choose the intended one anyway.
	flights, err := service.GetFlightPatterns(carrierParam, flightNumber)
	if err != nil {
		return nil, err
	}

	flights = patternsWithinRange(flights, departureDate, EstimateMaxFlightDuration)

	if len(flights) == 0 {
		return nil, &utils.ErrorWithHTTPCode{
			HTTPCode: http.StatusNotFound,
			ErrorMessage: fmt.Sprintf(
				"No flight %v / %v within %v days of %v",
				carrierParam.GetValue(),
				flightNumber,
				EstimateMaxFlightDuration,
				departureDate,
			),
		}
	}

	flights = assignFlightDayShift(flights)
	if len(flights) == 0 {
		return nil, xerrors.Errorf(
			"no flight %v / %v after attempt to assign flight day shift",
			carrierParam.GetValue(), flightNumber)
	}

	found, targetLegFlightDayShift, err := service.getTargetLegFlightDayShift(flights, fromStation, departureDate)
	if err != nil {
		return nil, xerrors.Errorf(
			"cannot detect shift of flight's leg that departs from %+v at %+v, flights: %+v",
			fromStation, departureDate, flights)
	}
	if !found {
		return nil, xerrors.Errorf(
			"cannot find segment of flight %v / %v that departs from %+v at %+v",
			carrierParam.GetValue(), flightNumber, fromStation, departureDate)
	}
	var baseDepartureDate = departureDate.AddDaysP(int(-targetLegFlightDayShift))

	var flightData []*flightdata.FlightData

	for _, fp := range flights {
		legDepartureDate := baseDepartureDate.AddDaysP(int(fp.FlightDayShift))
		legDepartureIndex := dtutil.DateCache.IndexOfIntDateP(legDepartureDate)

		if !flightOperatesOnDate(fp, legDepartureIndex) {
			continue
		}

		fb, err := service.FlightStorage().GetFlightBase(fp.FlightBaseID, fp.IsDop)
		if err != nil {
			return nil, xerrors.Errorf("error with flight %v / %v : %w", carrierParam.GetValue(), flightNumber, err)
		}

		flightDataElem, ok := service.buildFlightWithStatuses(fb, fp, legDepartureDate, showBanned, nationalVersion)
		if !ok {
			continue
		}

		flightData = append(flightData, flightDataElem)
	}
	if len(flightData) == 0 && includeLooseSegments {
		// for "flights" handle - return first matching segment in the case of multi-leg flight
		for _, fp := range flights {
			departureDateIndex := dtutil.DateCache.IndexOfIntDateP(baseDepartureDate)
			if !flightOperatesOnDate(fp, departureDateIndex) {
				continue
			}
			fb, err := service.FlightStorage().GetFlightBase(fp.FlightBaseID, fp.IsDop)
			if err != nil {
				return nil, xerrors.Errorf("error with flight %v / %v : %w", carrierParam.GetValue(), flightNumber, err)
			}
			flightDataElem, ok := service.buildFlightWithStatuses(fb, fp, baseDepartureDate, showBanned, nationalVersion)
			if !ok {
				continue
			}
			flightData = append(flightData, flightDataElem)
			break
		}
	}
	if len(flightData) == 0 {
		return nil, &utils.ErrorWithHTTPCode{
			HTTPCode:     http.StatusNotFound,
			ErrorMessage: fmt.Sprintf("no matching flights for %v %v %v", carrierParam.GetValue(), flightNumber, departureDate),
		}
	}

	if includeCodeshares {
		for _, flight := range flightData {
			service.fillCodeshares(flight)
		}
	}

	return flightData, nil
}

func (service *flightServiceImpl) fillCodeshares(flight *flightdata.FlightData) {
	if flight == nil || flight.FlightPattern == nil {
		return
	}
	flightKeys, ok := service.Storage.FlightStorage().GetCodeshareFlightKeys(
		flight.FlightBase.OperatingCarrier,
		flight.FlightBase.OperatingFlightNumber,
	)
	if !ok {
		return
	}
	codeshares := make(map[dto.TitledFlight]void)
	for _, key := range flightKeys {
		segments, ok := service.Storage.FlightStorage().GetFlightsByKey(key)
		if !ok {
			logger.Logger().Error(
				"Unable to retrieve flight by key",
				log.String("flightKey", key),
				log.Reflect("operatingSegment", flight.FlightPattern),
			)
			continue
		}
		for _, flightLeg := range segments {
			for _, flightPattern := range flightLeg {
				if flightPattern.LegNumber != flight.FlightPattern.LegNumber {
					continue
				}
				departureDateIndex := dtutil.DateCache.IndexOfIntDateP(flight.FlightDepartureDate)
				if !flightOperatesOnDate(flightPattern, departureDateIndex) {
					continue
				}
				if flightPattern.MarketingCarrier == flight.FlightPattern.MarketingCarrier &&
					flightPattern.MarketingFlightNumber == flight.FlightPattern.MarketingFlightNumber {
					continue
				}
				codeshares[dto.TitledFlight{
					FlightID: dto.FlightID{
						AirlineID: flightPattern.MarketingCarrier,
						Number:    flightPattern.MarketingFlightNumber,
					},
					Title: flightPattern.FlightTitle(),
				}] = none
			}
		}
	}
	flight.Codeshares = make([]dto.TitledFlight, 0)
	for codeshare := range codeshares {
		flight.Codeshares = append(flight.Codeshares, codeshare)
	}
	sort.Slice(flight.Codeshares, func(i, j int) bool {
		return flight.Codeshares[i].Title < flight.Codeshares[j].Title
	})
}

func flightOperatesOnDate(fp *structs.FlightPattern, legDepartureIndex int) bool {
	if dtutil.DateCache.IndexOfStringDateP(dtutil.StringDate(fp.OperatingFromDate)) > legDepartureIndex ||
		dtutil.DateCache.IndexOfStringDateP(dtutil.StringDate(fp.OperatingUntilDate)) < legDepartureIndex {
		return false
	}
	if !dtutil.OperatesOn(fp.OperatingOnDays, dtutil.DateCache.WeekDay(legDepartureIndex)) {
		return false
	}
	return true
}

func (service *flightServiceImpl) buildFlightWithStatuses(
	fb structs.FlightBase,
	fp *structs.FlightPattern,
	depDate dtutil.IntDate,
	showBanned bool,
	nationalVersion string) (*flightdata.FlightData, bool) {
	flightDataElem := flightdata.NewFlightData(fb, fp, nil, depDate, service.TimeZoneUtil)
	if flightDataElem.ScheduledDeparture().IsZero() && flightDataElem.ScheduledArrival().IsZero() {
		return flightDataElem, false
	}
	if service.Storage.BlacklistRuleStorage().IsBanned(fb, fp, depDate.StringDateDashed(), nationalVersion) {
		if !showBanned {
			return flightDataElem, false
		}
		flightDataElem.Banned = true
	}

	flightStatuses, hasStatuses := service.StatusStorage().GetFlightStatuses(fb.GetBucketKey())
	if hasStatuses {
		flightStatusValue, hasStatus := flightStatuses.GetStatus(depDate)
		if hasStatus {
			flightDataElem.FlightStatus = flightStatusValue
		}
	}
	return flightDataElem, true
}

type void struct{}

var none = void{}

type setOfFlightPatterns map[*structs.FlightPattern]void

func (s setOfFlightPatterns) Put(fp *structs.FlightPattern) {
	s[fp] = none
}

func (s setOfFlightPatterns) Contains(fp *structs.FlightPattern) bool {
	_, ok := s[fp]
	return ok
}

// assignFlightDayShift goes through passed flight patterns, tries to match them and calculate flightDayShift
func assignFlightDayShift(flights []*structs.FlightPattern) []*structs.FlightPattern {
	flightsByLegs, minLeg, maxLeg := separateByLegs(flights)
	processedSet := make(setOfFlightPatterns)

	for i := minLeg; i <= maxLeg-1; i++ {
		assignFlightDayShiftLegPair(flightsByLegs[i], flightsByLegs[i+1], processedSet)
	}

	output := make([]*structs.FlightPattern, 0, len(flights))

	// Leaving only legs that were successfully processed
	for _, fp := range flights {
		if fp.LegNumber != minLeg && !processedSet.Contains(fp) {
			logger.Logger().Debug("Throwing away segment", log.Reflect("segment", fp))
			continue
		}
		output = append(output, fp)
	}
	return output
}

func assignFlightDayShiftLegPair(previousLegs []*structs.FlightPattern, nextLegs []*structs.FlightPattern, processedSet setOfFlightPatterns) {
	for _, fp1 := range previousLegs {
		currentShift := fp1.FlightDayShift
		arrivalShift := fp1.ArrivalDayShift

		fp1OperatesFrom := dtutil.DateCache.IndexOfStringDateP(dtutil.StringDate(fp1.OperatingFromDate))
		fp1OperatesUntil := dtutil.DateCache.IndexOfStringDateP(dtutil.StringDate(fp1.OperatingUntilDate))

		for _, fp2 := range nextLegs {
			if processedSet.Contains(fp2) {
				continue
			}

			fp2OperatesFrom := dtutil.DateCache.IndexOfStringDateP(dtutil.StringDate(fp2.OperatingFromDate))
			fp2OperatesUntil := dtutil.DateCache.IndexOfStringDateP(dtutil.StringDate(fp2.OperatingUntilDate))

			legShift := math.Max(int(arrivalShift), int(fp2.DepartureDayShift))
			if fp2OperatesFrom-legShift > fp1OperatesUntil ||
				fp2OperatesUntil-legShift < fp1OperatesFrom {
				continue
			}

			if dtutil.OperatingDays(fp1.OperatingOnDays).
				ShiftDays(legShift).
				Intersect(dtutil.OperatingDays(fp2.OperatingOnDays)) == 0 {
				continue
			}

			fp2.FlightDayShift = currentShift + int32(legShift)

			processedSet.Put(fp2)
		}
	}
}

// separateByLegs takes passed flight patterns and groups them by leg
func separateByLegs(flights []*structs.FlightPattern) ([][]*structs.FlightPattern, int32, int32) {

	var minLeg = int32(9999)
	var maxLeg = int32(0)
	for _, fp := range flights {
		if minLeg > fp.LegNumber {
			minLeg = fp.LegNumber
		}
		if maxLeg < fp.LegNumber {
			maxLeg = fp.LegNumber
		}
	}

	sm := make([][]*structs.FlightPattern, maxLeg+1)

	for _, fp := range flights {
		sm[fp.LegNumber] = append(sm[fp.LegNumber], fp)
	}
	return sm, minLeg, maxLeg
}

func patternsWithinRange(flights []*structs.FlightPattern, date dtutil.IntDate, i int) []*structs.FlightPattern {
	dateRangeLeft := dtutil.DateCache.IndexOfIntDateP(date.AddDaysP(-i))
	dateRangeRight := dtutil.DateCache.IndexOfIntDateP(date.AddDaysP(i))

	output := make([]*structs.FlightPattern, 0, len(flights))

	for _, fp := range flights {
		if dtutil.DateCache.IndexOfStringDateP(dtutil.StringDate(fp.OperatingFromDate)) > dateRangeRight ||
			dtutil.DateCache.IndexOfStringDateP(dtutil.StringDate(fp.OperatingUntilDate)) < dateRangeLeft {
			continue
		}
		output = append(output, fp)
	}
	return output
}

func (service *flightServiceImpl) getTargetLegFlightDayShift(
	flights []*structs.FlightPattern, station *snapshots.TStationWithCodes, date dtutil.IntDate,
) (found bool, legShift int32, err error) {
	if station == nil {
		return true, 0, nil
	}

	dateIndex := dtutil.DateCache.IndexOfIntDateP(date)

	for _, fp := range flights {
		fb, err := service.FlightStorage().GetFlightBase(fp.FlightBaseID, fp.IsDop)
		if err != nil {
			return false, 0, xerrors.Errorf("no flight base for flight %+v", fp)
		}
		if fb.DepartureStation == int64(station.Station.Id) {

			if dtutil.DateCache.IndexOfStringDateP(dtutil.StringDate(fp.OperatingFromDate)) > dateIndex ||
				dtutil.DateCache.IndexOfStringDateP(dtutil.StringDate(fp.OperatingUntilDate)) < dateIndex {
				continue
			}

			if !dtutil.OperatesOn(fp.OperatingOnDays, dtutil.DateCache.WeekDay(dateIndex)) {
				continue
			}

			return true, fp.FlightDayShift, nil
		}
	}
	return false, 0, nil
}

func (service *flightServiceImpl) GetFlightPatterns(
	carrierParam CarrierParam, flightNumber string) (response []*structs.FlightPattern, err error) {
	defer func() {
		if r := recover(); r != nil {
			response = nil
			err = &utils.ErrorWithHTTPCode{
				HTTPCode:     http.StatusInternalServerError,
				ErrorMessage: xerrors.Errorf("getFlightDate panicked: %v", r).Error(),
			}
			return
		}
	}()

	var carrierID64 = int64(carrierParam.GetID())
	if carrierID64 == 0 {
		carrierID64, err = strconv.ParseInt(carrierParam.GetText(), 10, 32)
	}

	var carrierID int32
	if err != nil {
		carrierID = service.CarrierService.GetCarrierByCodeAndFlightNumber(carrierParam.GetText(), flightNumber)
		if carrierID == 0 {
			return nil, &utils.ErrorWithHTTPCode{
				HTTPCode:     http.StatusNotFound,
				ErrorMessage: fmt.Sprintf("unknown carrier %v", carrierParam.GetValue()),
			}
		}
	} else {
		carrierID = int32(carrierID64)
		if len(service.CarrierStorage().GetCarrierCodeByID(carrierID)) == 0 {
			return nil, &utils.ErrorWithHTTPCode{
				HTTPCode:     http.StatusNotFound,
				ErrorMessage: fmt.Sprintf("non-existing carrier %v", carrierParam.GetValue()),
			}
		}
	}

	if len([]rune(flightNumber)) > 5 {
		return nil, &utils.ErrorWithHTTPCode{
			HTTPCode:     http.StatusBadRequest,
			ErrorMessage: fmt.Sprintf("invalid (too long) flight number %v", flightNumber),
		}
	}

	flights, hasValue := service.FlightStorage().GetFlights(carrierID, flightNumber)
	if !hasValue {
		return nil, nil
	}
	return flights.Flatten(), nil
}

func Convert(flightData []*flightdata.FlightData, nowDate time.Time, lastImported string) (*FlightSegment, error) {
	sort.Slice(flightData, func(pos1, pos2 int) bool {
		return flightData[pos1].FlightBase.LegNumber < flightData[pos2].FlightBase.LegNumber
	})
	firstLeg := flightData[0]
	lastLeg := flightData[len(flightData)-1]
	flight, err := getFlightSegment(firstLeg, lastLeg, lastImported)
	if err != nil {
		return nil, err
	}
	flight.Status = flightstatus.GetFlightStatus(firstLeg, lastLeg, nowDate)
	if firstLeg.Codeshares != nil && len(firstLeg.Codeshares) > 0 {
		flight.FlightCodeshares = make([]dto.TitledFlight, len(firstLeg.Codeshares))
		copy(flight.FlightCodeshares, firstLeg.Codeshares)
	} else {
		flight.FlightCodeshares = []dto.TitledFlight{}
	}
	if firstLeg.FlightPattern.IsCodeshare {
		flight.Operating = &dto.TitledFlight{
			FlightID: dto.FlightID{
				AirlineID: firstLeg.FlightBase.OperatingCarrier,
				Number:    firstLeg.FlightBase.OperatingFlightNumber,
			},
			Title: fmt.Sprintf("%s %s", firstLeg.FlightBase.OperatingCarrierCode, firstLeg.FlightBase.OperatingFlightNumber),
		}
	} else {
		flight.Operating = nil
	}

	if len(flightData) > 1 {
		for _, leg := range flightData {
			var segment *FlightSegment
			if segment, err = getFlightSegment(leg, leg, lastImported); err != nil {
				return nil, err
			}
			if leg.FlightStatus != nil {
				segment.Status = flightstatus.GetFlightStatus(leg, leg, nowDate)
			} else {
				segment.Status.DepartureStatus = string(appconst.FlightStatusUnknown)
				segment.Status.ArrivalStatus = string(appconst.FlightStatusUnknown)
				segment.Status.Status = appconst.FlightStatusUnknown
			}
			if leg.FlightPattern.IsDop {
				segment.Source = dto.FLIGHTBOARD.String()
			}
			flight.Segments = append(flight.Segments, segment)
		}
	} else {
		if firstLeg.FlightPattern.IsDop {
			flight.Source = dto.FLIGHTBOARD.String()
		}
	}
	return flight, nil
}

func getFlightSegment(departureLeg *flightdata.FlightData, arrivalLeg *flightdata.FlightData, lastImported string) (fs *FlightSegment, err error) {
	bannedStr := ""
	if departureLeg.Banned || arrivalLeg.Banned {
		bannedStr = "true"
	}

	departure := departureLeg.ScheduledDeparture()
	var departureDay, departureTime string
	if !departure.IsZero() {
		departureDay = departure.Format(dtutil.IsoDate)
		departureTime = departure.Format(dtutil.IsoTime)
	}

	arrival := arrivalLeg.ScheduledArrival()
	var arrivalDay, arrivalTime string
	if !arrival.IsZero() {
		arrivalDay = arrival.Format(dtutil.IsoDate)
		arrivalTime = arrival.Format(dtutil.IsoTime)
	}

	return &FlightSegment{
		CompanyIata:       departureLeg.FlightPattern.TitleCarrierCode(),
		CompanyRaspID:     departureLeg.FlightPattern.CorrectedCarrierID(),
		Number:            departureLeg.FlightPattern.MarketingFlightNumber,
		Title:             departureLeg.FlightPattern.FlightTitle(),
		AirportFromIata:   departureLeg.FlightBase.DepartureStationCode,
		AirportFromRaspID: departureLeg.FlightBase.DepartureStation,
		DepartureDay:      departureDay,
		DepartureTime:     departureTime,
		DepartureTzName:   departureLeg.DepartureTimezone().String(),
		DepartureUTC:      dtutil.FormatDateTimeISO(departureLeg.ScheduledDeparture().In(time.UTC)),
		DepartureTerminal: departureLeg.FlightBase.DepartureTerminal,
		AirportToIata:     arrivalLeg.FlightBase.ArrivalStationCode,
		AirportToRaspID:   arrivalLeg.FlightBase.ArrivalStation,
		ArrivalDay:        arrivalDay,
		ArrivalTime:       arrivalTime,
		ArrivalTzName:     arrivalLeg.ArrivalTimezone().String(),
		ArrivalUTC:        dtutil.FormatDateTimeISO(arrivalLeg.ScheduledArrival().In(time.UTC)),
		ArrivalTerminal:   arrivalLeg.FlightBase.ArrivalTerminal,
		TransportModelID:  departureLeg.FlightBase.AircraftTypeID,
		CreatedAtUTC:      lastImported, // TODO(u-jeen): replace with real value when implemented
		UpdatedAtUTC:      lastImported, // TODO(u-jeen): replace with real value when implemented
		Segments:          make([]*FlightSegment, 0),
		Banned:            bannedStr,
	}, nil
}
