package flight

import (
	"net/http"
	"sort"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/services/flightdata"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/utils"
	"a.yandex-team.ru/travel/avia/shared_flights/api/pkg/structs"
	dir "a.yandex-team.ru/travel/avia/shared_flights/lib/go/direction"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/dtutil"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/logger"
)

type tSearchDirection bool

const ToPast = tSearchDirection(true)
const ToFuture = tSearchDirection(false)

func (service *flightServiceImpl) GetFlightRange(
	carrierParam CarrierParam,
	flightNumber string,
	nationalVersion string,
	showBanned bool,
	nowDate time.Time,
	limitBefore int,
	limitAfter int,
	direction dir.Direction,
	referenceDateTime time.Time,
) (response []*FlightSegment, err error) {

	var flightDatesBefore, flightDatesAfter, flightDates []dtutil.IntDate
	var flights []*FlightSegment
	var flightPatterns []*structs.FlightPattern

	flightPatterns, err = service.GetFlightPatterns(carrierParam, flightNumber)
	if err != nil {
		return nil, err
	}

	if flightPatterns == nil {
		return nil, &utils.ErrorWithHTTPCode{
			HTTPCode:     http.StatusNotFound,
			ErrorMessage: xerrors.Errorf("cannot find flights: %v/%v", carrierParam.GetValue(), flightNumber).Error(),
		}
	}

	flightDatesBefore, err = getFlightDatesFuzzy(flightPatterns, referenceDateTime, limitBefore, ToPast, direction)
	if err != nil {
		return nil, xerrors.Errorf("cannot get flights before point: %w", err)
	}
	flightDatesAfter, err = getFlightDatesFuzzy(flightPatterns, referenceDateTime, limitAfter, ToFuture, direction)
	if err != nil {
		return nil, xerrors.Errorf("cannot get flights after point: %w", err)
	}

	flightDates = sortDates(flightDatesBefore, flightDatesAfter)

	flights, err = service.flightsForDates(
		referenceDateTime, direction, limitBefore, limitAfter,
		flightDates, carrierParam, flightNumber, nationalVersion, showBanned, nowDate)
	if err != nil {
		return nil, xerrors.Errorf("cannot get flights for dates: %w", err)
	}
	if flights == nil {
		return nil, &utils.ErrorWithHTTPCode{
			HTTPCode:     http.StatusNotFound,
			ErrorMessage: xerrors.Errorf("cannot find flights: %v/%v", carrierParam.GetValue(), flightNumber).Error(),
		}
	}
	return flights, nil
}

func (service *flightServiceImpl) flightsForDates(
	referenceDateTime time.Time,
	flightStage dir.Direction,
	limitBefore int,
	limitAfter int,
	dates []dtutil.IntDate,
	carrierParam CarrierParam,
	flightNumber string,
	nationalVersion string,
	showBanned bool,
	pseudoNow time.Time,
) ([]*FlightSegment, error) {
	// make segment groups (aka flights)
	var flightDataList [][]*flightdata.FlightData
	for _, d := range dates {
		flightData, err := service.GetFlightData(
			carrierParam, flightNumber, string(d.StringDate()), nil, nationalVersion, showBanned, false, false)
		if err != nil {
			httpErr, ok := err.(*utils.ErrorWithHTTPCode)
			if !ok || httpErr.HTTPCode != http.StatusNotFound {
				logger.Logger().Error(
					"Cannot get flight data for date",
					log.Int("date", int(d)),
					log.Error(err),
				)
			}
			continue
		}
		flightDataList = append(flightDataList, flightData)
	}
	if len(flightDataList) == 0 {
		return nil, nil
	}

	flightDataList = dropCorruptedFlights(flightDataList, flightStage)
	if len(flightDataList) == 0 {
		return nil, nil
	}

	flightDataList = cropFlightByLimit(flightDataList, flightStage, referenceDateTime, limitBefore, limitAfter)

	// make response for each flight and return
	result := make([]*FlightSegment, 0, len(dates))
	for _, flightData := range flightDataList {
		flight, err := Convert(flightData, pseudoNow, service.GetLastImported())
		if err != nil {
			return nil, err
		}
		result = append(result, flight)
	}
	return result, nil
}

func dropCorruptedFlights(flightDataList [][]*flightdata.FlightData, flightStage dir.Direction) [][]*flightdata.FlightData {
	// drop corrupted flights
	{
		var j = 0
		for i := range flightDataList {
			dt := getFlightDateTimeForFlightStage(flightDataList[i], flightStage)
			if dt.IsZero() {
				continue
			}
			if containsHalfFlights(flightDataList[i]) {
				continue
			}
			flightDataList[j] = flightDataList[i]
			j++
		}
		if j == 0 {
			return nil
		}
		flightDataList = flightDataList[:j]
	}
	return flightDataList
}

func cropFlightByLimit(flightDataList [][]*flightdata.FlightData, flightStage dir.Direction, referenceDateTime time.Time, limitBefore int, limitAfter int) [][]*flightdata.FlightData {
	// count flights to the left and to the right of reference point
	var countBeforeReferenceDateTime, countAfterReferenceDateTime int

	for _, flight := range flightDataList {
		dt := getFlightDateTimeForFlightStage(flight, flightStage)
		if dt.Before(referenceDateTime) {
			countBeforeReferenceDateTime++
		} else {
			countAfterReferenceDateTime++
		}
	}

	cropFromLeft := countBeforeReferenceDateTime - limitBefore
	if cropFromLeft < 0 {
		cropFromLeft = 0
	}

	cropFromRight := countAfterReferenceDateTime - limitAfter
	if cropFromRight < 0 {
		cropFromRight = 0
	}

	flightDataList = flightDataList[cropFromLeft : len(flightDataList)-cropFromRight]
	return flightDataList
}

func getFlightDateTimeForFlightStage(segments []*flightdata.FlightData, stage dir.Direction) time.Time {
	if len(segments) == 0 {
		return time.Time{}
	}
	if stage == dir.DEPARTURE {
		return segments[0].ScheduledDeparture()
	} else {
		return segments[len(segments)-1].ScheduledArrival()
	}
}

func containsHalfFlights(segments []*flightdata.FlightData) bool {
	for _, segment := range segments {
		if segment.FlightPattern != nil && segment.FlightPattern.IsHalfFlight {
			return true
		}
	}
	return false
}

func sortDates(before []dtutil.IntDate, after []dtutil.IntDate) []dtutil.IntDate {
	dates := make([]dtutil.IntDate, 0, len(before)+len(after))

	dates = append(dates, before...)
	dates = append(dates, after...)

	sort.SliceStable(dates, func(i, j int) bool { return dates[i] < dates[j] })

	var index = 1

	for i := 1; i < len(dates); i++ {
		if dates[i] == dates[i-1] {
			continue
		}
		dates[index] = dates[i]
		index++
	}
	if index < len(dates) {
		dates = dates[:index]
	}
	return dates
}

// getFlightDatesFuzzy returns departure dates of first segment of flight
func getFlightDatesFuzzy(flights []*structs.FlightPattern, timeUtc time.Time, limit int, searchDirection tSearchDirection, direction dir.Direction) ([]dtutil.IntDate, error) {

	// We cannot be sure about dates and times of flight as a whole, here we only have info about segments
	// Let just return more departure dates than needed and filter out later, when we combine segments into a single flight

	// referenceDate is not a strict limit. We are not even sure about timezone here
	var referenceDate = dtutil.DateToInt(timeUtc)

	// lets search for a couple more days just to have more info
	const referenceDateBoundaryBlur = 2
	// and increase the limit, again to get more dates
	const limitBlur = 5

	filter, err := flightPatternFilterForDateDirectionTimeArrow(referenceDate, direction, searchDirection, referenceDateBoundaryBlur)
	if err != nil {
		return nil, xerrors.Errorf("cannot get flight dates: %w", err)
	}

	var firstLegFlightPatternsInRange []*structs.FlightPattern

	firstLegFlightPatternsInRange = make([]*structs.FlightPattern, 0, len(flights))

	for _, flightPattern := range flights {
		if flightPattern.LegNumber != 1 || flightPattern.IsHalfFlight {
			continue
		}
		if !filter(flightPattern) {
			continue
		}
		firstLegFlightPatternsInRange = append(firstLegFlightPatternsInRange, flightPattern)
	}

	var datesList = getDatesListOfFirstLeg(firstLegFlightPatternsInRange, referenceDate, searchDirection, limit+limitBlur)

	return datesList, nil
}

func getDatesListOfFirstLeg(
	firstLegFlightPatterns []*structs.FlightPattern, referenceDate dtutil.IntDate, searchDirection tSearchDirection, limit int,
) []dtutil.IntDate {
	type IntDateSet map[dtutil.IntDate]void

	dateSet := make(IntDateSet)

	for _, flightPattern := range firstLegFlightPatterns {
		var increment int
		rangeFrom := dtutil.StringDate(flightPattern.OperatingFromDate).ToIntDate()
		rangeUntil := dtutil.StringDate(flightPattern.OperatingUntilDate).ToIntDate()
		if searchDirection == ToPast {
			if referenceDate < rangeUntil {
				rangeUntil = referenceDate
			}
			increment = -1
		} else if searchDirection == ToFuture {
			if referenceDate > rangeFrom {
				rangeFrom = referenceDate
			}
			increment = 1
		}

		currentCount := 0
		var date dtutil.IntDate
		if searchDirection == ToPast {
			date = rangeUntil
		} else if searchDirection == ToFuture {
			date = rangeFrom
		}
		for ; rangeFrom <= date && date <= rangeUntil; date = date.AddDaysP(increment) {
			if dtutil.OperatesOn(flightPattern.OperatingOnDays, dtutil.DateCache.WeekDay(dtutil.DateCache.IndexOfIntDateP(date))) {
				dateSet[date] = none
				currentCount++
				if currentCount >= limit {
					break
				}
			}
		}
	}

	var dateList []dtutil.IntDate
	for date := range dateSet {
		dateList = append(dateList, date)
	}
	if searchDirection {
		sort.SliceStable(dateList, func(i, j int) bool { return dateList[i] > dateList[j] })
	} else {
		sort.SliceStable(dateList, func(i, j int) bool { return dateList[i] < dateList[j] })
	}

	if limit > len(dateList) {
		limit = len(dateList)
	}
	return dateList[:limit]
}

// flightPatternFilterForDateDirectionTimeArrow
func flightPatternFilterForDateDirectionTimeArrow(
	referenceDate dtutil.IntDate, direction dir.Direction, searchDirection tSearchDirection, blur int,
) (func(pattern *structs.FlightPattern) bool, error) {

	switch direction {
	case dir.DEPARTURE:
		{
			if searchDirection == ToPast {
				bluredReferenceDate := referenceDate.AddDaysP(blur)
				return func(pattern *structs.FlightPattern) bool {
					return dtutil.StringDate(pattern.OperatingFromDate).ToIntDate() <= bluredReferenceDate
				}, nil
			} else {
				bluredReferenceDate := referenceDate.AddDaysP(-blur)
				return func(pattern *structs.FlightPattern) bool {
					return dtutil.StringDate(pattern.OperatingUntilDate).ToIntDate() >= bluredReferenceDate
				}, nil
			}
		}
	case dir.ARRIVAL:
		{
			if searchDirection == ToPast {
				futureReferenceDate := referenceDate.AddDaysP(blur)
				return func(pattern *structs.FlightPattern) bool {
					return dtutil.StringDate(pattern.OperatingFromDate).
						ToIntDate().
						AddDaysP(int(pattern.ArrivalDayShift)) <= futureReferenceDate
				}, nil
			} else {
				pastReferenceDate := referenceDate.AddDaysP(-blur)
				return func(pattern *structs.FlightPattern) bool {
					return dtutil.StringDate(pattern.OperatingUntilDate).
						ToIntDate().
						AddDaysP(int(pattern.ArrivalDayShift)) >= pastReferenceDate
				}, nil
			}
		}
	default:
		return nil, xerrors.Errorf("cannot create flight filter: unknown direction: %v", direction)
	}
}
