package flight

import (
	"net/http"
	"time"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/utils"
	dir "a.yandex-team.ru/travel/avia/shared_flights/lib/go/direction"
)

const MaxCarriersPerFlight = 5

func (service *flightServiceImpl) GetFlightRangeMulti(
	flightNumber string,
	nationalVersion string,
	showBanned bool,
	nowDate time.Time,
	limitBefore int,
	limitAfter int,
	direction dir.Direction,
	referenceDateTime time.Time,
) (response map[int32][]*FlightSegment, err error) {
	response = make(map[int32][]*FlightSegment)
	carriers := service.CarriersPopularityScores().GetCarriers(flightNumber, nationalVersion)
	if len(carriers) == 0 {
		return nil, &utils.ErrorWithHTTPCode{
			HTTPCode:     http.StatusNotFound,
			ErrorMessage: xerrors.Errorf("no carriers for the flight: %v", flightNumber).Error(),
		}
	}

	carriersCount := 0
	for _, carrier := range carriers {
		carrierFlights, err := service.GetFlightRange(
			NewCarrierParamByID(carrier),
			flightNumber,
			nationalVersion,
			showBanned,
			nowDate,
			limitBefore,
			limitAfter,
			direction,
			referenceDateTime,
		)
		if err != nil || len(carrierFlights) == 0 {
			continue
		}
		response[carrier] = carrierFlights
		carriersCount++
		if carriersCount >= MaxCarriersPerFlight {
			break
		}
	}
	if carriersCount == 0 {
		return nil, &utils.ErrorWithHTTPCode{
			HTTPCode:     http.StatusNotFound,
			ErrorMessage: xerrors.Errorf("cannot find flights for: %v / %v", carriers, flightNumber).Error(),
		}
	}
	return response, nil
}
