package flightp2p

import (
	"sort"
	"strconv"
	"time"

	"a.yandex-team.ru/library/go/core/xerrors"
	dto "a.yandex-team.ru/travel/avia/shared_flights/api/internal/services/storage/DTO"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/services/storage/flight_p2p/format"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/services/storage/timezone"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/storage"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/storage/flight"
	"a.yandex-team.ru/travel/avia/shared_flights/api/pkg/structs"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/dtutil"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/math"
	"a.yandex-team.ru/travel/proto/shared_flights/snapshots"
)

// It is assumed that absolutely no flight (even a multi-segment one) can be "in flight" for more than this number of days, ever.
const MaxDaysInFlight = 8

// It is assumed that absolutely no flight (even a multi-segment one) can have more than this number of segments, ever.
const MaxSegmentsInFlight = 12

type FlightP2PService interface {
	GetFlightsP2P(
		departFrom []*snapshots.TStationWithCodes,
		arriveTo []*snapshots.TStationWithCodes,
		from TimeParam,
		until TimeParam,
		nationalVersion string,
		showBanned bool,
		flightKeys []FlightKey,
	) (format.Response, error)

	GetFlightsP2PSchedule(
		departFrom []*snapshots.TStationWithCodes,
		arriveTo []*snapshots.TStationWithCodes,
		nationalVersion string,
		showBanned bool,
		startScheduleDate dtutil.IntDate,
	) (format.ScheduleResponse, error)

	GetFlightsP2PSummary(debug bool) (format.SummaryResponse, error)

	GetFlightP2PSegmentInfo(
		fromStations,
		toStations []*snapshots.TStationWithCodes,
		nationalVersion string,
		showBanned bool,
		debug bool,
	) (format.SegmentInfoResponse, error)

	GetFlightsP2PByDate(
		departFrom []*snapshots.TStationWithCodes,
		arriveTo []*snapshots.TStationWithCodes,
		flightDate TimeParam,
		daysCount int,
		nationalVersion string,
		showBanned bool,
		debug bool,
	) (format.DirectDatesAndFlights, error)
}

func NewFlightP2PService(
	storage *storage.Storage,
	timeZoneUtil timezone.TimeZoneUtil,
) FlightP2PService {
	return &FlightP2PServiceImpl{
		Storage:      storage,
		TimeZoneUtil: timeZoneUtil,
	}
}

type FlightKey struct {
	MarketingCarrier int32
	FlightNumber     string
}

type ResponseFlightKey struct {
	OperatingCarrier  int32
	FlightNumber      string
	DepartureStation  int32
	ArrivalStation    int32
	DepartureDatetime string
}

type FlightP2PServiceImpl struct {
	*storage.Storage
	timezone.TimeZoneUtil
}

type FlightEntryKey struct {
	departureDateIndex int
	flightID           int32
}

type FlightEntryValue struct {
	operating        map[int32]*FlightPatternBaseAndDate
	codeshares       map[dto.FlightID]dto.TitledFlight
	departureStation int32
	arrivalStation   int32
}

type FlightPatternBaseAndDate struct {
	fp                 *structs.FlightPattern
	fb                 structs.FlightBase
	departureDateIndex int
	arrivalDayShift    int32
	isBanned           bool
	departFromTz       *time.Location
	arriveToTz         *time.Location
}

type TimeParam struct {
	Value   time.Time // in UTC
	IsLocal bool
}

func (t *TimeParam) In(loc *time.Location) time.Time {
	if t.IsLocal {
		return time.Date(t.Value.Year(), t.Value.Month(), t.Value.Day(), t.Value.Hour(), t.Value.Minute(), t.Value.Second(), 0, loc)
	} else {
		return t.Value.In(loc)
	}
}

func (t *TimeParam) After(other TimeParam) bool {
	return t.Value.After(other.Value)
}

// For tests
func NewUtcTime(timeString string) TimeParam {
	value, _ := time.ParseInLocation(dtutil.IsoDateTime, timeString, time.UTC)
	return TimeParam{
		Value:   value,
		IsLocal: false,
	}
}

type void struct{}

var none = void{}

func (service *FlightP2PServiceImpl) GetFlightsP2P(
	departFrom []*snapshots.TStationWithCodes,
	arriveTo []*snapshots.TStationWithCodes,
	from TimeParam,
	until TimeParam,
	nationalVersion string,
	showBanned bool,
	flightKeys []FlightKey,
) (format.Response, error) {
	type FlightPatternAndBase struct {
		fp *structs.FlightPattern
		fb structs.FlightBase
	}
	departureStations := make([]int32, 0, len(departFrom))
	for _, station := range departFrom {
		departureStations = append(departureStations, station.Station.Id)
	}
	arrivalStations := make([]int32, 0, len(arriveTo))
	for _, station := range arriveTo {
		arrivalStations = append(arrivalStations, station.Station.Id)
	}
	response := format.Response{
		DepartureStations: departureStations,
		ArrivalStations:   arrivalStations,
		Flights:           make([]format.Flight, 0),
	}
	if from.IsLocal != until.IsLocal {
		return response, xerrors.Errorf("after/before params should be both local or both utc: (%+v) - (%+v)", from, until)
	}
	if from.After(until) {
		return response, xerrors.Errorf("invalid date range: (%+v) - (%+v)", from.Value, until.Value)
	}

	timeZoneMap := make(map[int32]*time.Location)
	for _, station := range departFrom {
		if _, alreadyProcessed := timeZoneMap[station.Station.Id]; alreadyProcessed {
			continue
		}
		tz := service.GetTimeZone(station.Station)
		if tz == nil {
			return response, xerrors.Errorf("cannot load timezone for station  %+v", station.Station.Id)
		}
		timeZoneMap[station.Station.Id] = tz
	}
	for _, station := range arriveTo {
		if _, alreadyProcessed := timeZoneMap[station.Station.Id]; alreadyProcessed {
			continue
		}
		tz := service.GetTimeZone(station.Station)
		if tz == nil {
			return response, xerrors.Errorf("cannot load timezone for station  %+v", station.Station.Id)
		}
		timeZoneMap[station.Station.Id] = tz
	}

	flightsCache := make(map[FlightEntryKey]FlightEntryValue)

	for _, departureStation := range departFrom {
		for _, arrivalStation := range arriveTo {
			flightKeys, ok := service.Storage.FlightStorage().GetFlightsP2P(
				int64(departureStation.Station.Id),
				int64(arrivalStation.Station.Id),
			)
			if !ok {
				continue
			}

			departFromTz := timeZoneMap[departureStation.Station.Id]

			fromAsTime := from.In(departFromTz)
			untilAsTime := until.In(departFromTz)
			fromAsInt := dtutil.DateToInt(fromAsTime)
			untilAsInt := dtutil.DateToInt(untilAsTime)
			fromDateIndex := dtutil.DateCache.IndexOfIntDateP(fromAsInt)
			untilDateIndex := dtutil.DateCache.IndexOfIntDateP(untilAsInt)

			for _, flightKey := range flightKeys {
				flightPatterns, ok := service.Storage.FlightStorage().GetFlightsByKey(flightKey)
				if !ok {
					continue
				}

				// find departure segment since we filter by the departure time
				departureSegments := []FlightPatternAndBase{}
				for _, legFlights := range flightPatterns {
					for _, flightPattern := range legFlights {
						flightBase, err := service.Storage.FlightStorage().GetFlightBase(flightPattern.FlightBaseID, flightPattern.IsDop)
						if err != nil {
							return response, err
						}
						// skip half-flights in the point-to-point response - they could not be shown anyway
						if !dtutil.IntTime(flightBase.DepartureTimeScheduled).IsValid() ||
							!dtutil.IntTime(flightBase.ArrivalTimeScheduled).IsValid() {
							continue
						}
						if flightBase.DepartureStation == int64(departureStation.Station.Id) {
							departureSegments = append(
								departureSegments,
								FlightPatternAndBase{
									fp: flightPattern,
									fb: flightBase,
								},
							)
						}
					}
				}
				if len(departureSegments) == 0 {
					// happens for dop flights
					continue
				}
				for departureDateIndex := fromDateIndex; departureDateIndex <= untilDateIndex; departureDateIndex++ {
					var departureSegment *FlightPatternAndBase
					departureSegment = nil
					isBanned := false
					for _, segment := range departureSegments {
						if !operatesOn(segment.fp, departureDateIndex) {
							continue
						}
						// filter by departure time
						departureTime := dtutil.LocalTime(departureDateIndex, segment.fb.DepartureTimeScheduled, departFromTz)
						if departureTime.Before(fromAsTime) || departureTime.After(untilAsTime) {
							continue
						}
						// Note. We assume that if the departure segment of flight is banned, the rest are banned as well;
						// and vice versa, if the departure segment is not banned, no flight segments are banned.
						if service.Storage.BlacklistRuleStorage().IsBanned(
							segment.fb, segment.fp, dtutil.FormatDateIso(departureTime), nationalVersion) {
							if !showBanned {
								continue
							} else {
								isBanned = true
							}
						}
						departureSegment = &segment
						break
					}
					if departureSegment == nil {
						continue
					}

					// prepare to cache the result
					flightEntryKey := FlightEntryKey{
						departureDateIndex: departureDateIndex,
						flightID:           departureSegment.fb.ID,
					}
					flightEntry, flightEntryOk := flightsCache[flightEntryKey]
					if !flightEntryOk {
						flightEntry = FlightEntryValue{
							operating:  make(map[int32]*FlightPatternBaseAndDate),
							codeshares: make(map[dto.FlightID]dto.TitledFlight),
						}
						flightsCache[flightEntryKey] = flightEntry
					}
					if departureSegment.fp.IsCodeshare {
						flightID := dto.FlightID{
							AirlineID: departureSegment.fp.MarketingCarrier,
							Number:    departureSegment.fp.MarketingFlightNumber,
						}
						flightEntry.codeshares[flightID] = dto.TitledFlight{
							FlightID: flightID,
							Title:    departureSegment.fp.FlightTitle(),
						}
						flightsCache[flightEntryKey] = flightEntry
						continue
					}

					// got a departing segment that flies on the proper date, now collect the other flight legs
					legToSegmentMap := make(map[int32]*FlightPatternBaseAndDate)
					legToSegmentMap[departureSegment.fp.LegNumber] = &FlightPatternBaseAndDate{
						fb:                 departureSegment.fb,
						fp:                 departureSegment.fp,
						departureDateIndex: departureDateIndex,
						arrivalDayShift:    departureSegment.fp.ArrivalDayShift,
						isBanned:           isBanned,
						departFromTz:       timeZoneMap[int32(departureSegment.fb.DepartureStation)],
						arriveToTz:         timeZoneMap[int32(departureSegment.fb.ArrivalStation)],
					}
					// routeStationsMap[i-1][departure station] == none for leg i,
					// routeStationsMap[i][arrival station] == none for leg i
					routeStationsMap := make(map[int32]map[int64]void)
					putStationToMap(routeStationsMap, departureSegment.fp.LegNumber-1, departureSegment.fb.DepartureStation)
					putStationToMap(routeStationsMap, departureSegment.fp.LegNumber, departureSegment.fb.ArrivalStation)

					departureLegNumber := departureSegment.fp.LegNumber
					flightPatternsForSort := make(flight.FlightLegs, len(flightPatterns))
					copy(flightPatternsForSort, flightPatterns)
					sortFlightPatterns(flightPatternsForSort, departureLegNumber)

					for _, legFlights := range flightPatternsForSort {
						for _, flightPattern := range legFlights {
							if flightPattern.LegNumber == departureLegNumber {
								continue
							}
							flightBase, err := service.Storage.FlightStorage().GetFlightBase(flightPattern.FlightBaseID, flightPattern.IsDop)
							if err != nil {
								return response, err
							}
							// verify arrival/departure station before adding segment to the route
							if flightPattern.LegNumber < departureLegNumber {
								expectedArrivalStations, stationOk := routeStationsMap[flightPattern.LegNumber]
								if !stationOk {
									continue
								}
								if _, ok := expectedArrivalStations[flightBase.ArrivalStation]; !ok {
									continue
								}
								putStationToMap(routeStationsMap, flightPattern.LegNumber-1, flightBase.DepartureStation)
							} else {
								expectedDepartureStations, stationOk := routeStationsMap[flightPattern.LegNumber-1]
								if !stationOk {
									continue
								}
								if _, ok := expectedDepartureStations[flightBase.DepartureStation]; !ok {
									continue
								}
								putStationToMap(routeStationsMap, flightPattern.LegNumber, flightBase.ArrivalStation)
							}

							startDateIndex := departureDateIndex
							segmentArrivalDayShift := int32(0)
							if flightPattern.LegNumber < departureLegNumber {
								dayShift := flightPattern.ArrivalDayShift
								for legNumber := flightPattern.LegNumber; legNumber < departureLegNumber; legNumber++ {
									nextSegment, hasNext := legToSegmentMap[flightPattern.LegNumber+1]
									if hasNext {
										dayShift = math.MaxInt32(dayShift, nextSegment.fp.DepartureDayShift)
									}
									segmentArrivalDayShift += dayShift
									if hasNext {
										dayShift = nextSegment.fp.ArrivalDayShift
									}
								}
							} else {
								dayShift := flightPattern.DepartureDayShift
								for legNumber := flightPattern.LegNumber; legNumber > departureLegNumber; legNumber-- {
									prevSegment, hasPrev := legToSegmentMap[flightPattern.LegNumber-1]
									if hasPrev {
										dayShift = math.MaxInt32(dayShift, prevSegment.fp.ArrivalDayShift)
									}
									segmentArrivalDayShift += dayShift
									if hasPrev {
										dayShift = prevSegment.fp.DepartureDayShift
									}
								}
							}

							currentSegment, ok := legToSegmentMap[flightPattern.LegNumber]
							if !ok {
								if flightPattern.LegNumber < departureLegNumber {
									for dateIndex := startDateIndex - int(segmentArrivalDayShift); dateIndex >= departureDateIndex-MaxDaysInFlight; dateIndex-- {
										if operatesOn(flightPattern, dateIndex) {
											legToSegmentMap[flightPattern.LegNumber] = &FlightPatternBaseAndDate{
												fb:                 flightBase,
												fp:                 flightPattern,
												departureDateIndex: dateIndex,
												arrivalDayShift:    flightPattern.ArrivalDayShift,
												departFromTz:       timeZoneMap[int32(flightBase.DepartureStation)],
												arriveToTz:         timeZoneMap[int32(flightBase.ArrivalStation)],
											}
											break
										}
									}
								} else {
									for dateIndex := startDateIndex + int(segmentArrivalDayShift); dateIndex <= departureDateIndex+MaxDaysInFlight; dateIndex++ {
										if operatesOn(flightPattern, dateIndex) {
											legToSegmentMap[flightPattern.LegNumber] = &FlightPatternBaseAndDate{
												fb:                 flightBase,
												fp:                 flightPattern,
												departureDateIndex: dateIndex,
												arrivalDayShift:    flightPattern.ArrivalDayShift,
												departFromTz:       timeZoneMap[int32(flightBase.DepartureStation)],
												arriveToTz:         timeZoneMap[int32(flightBase.ArrivalStation)],
											}
											break
										}
									}
								}
								continue
							}
							if flightPattern.LegNumber < departureLegNumber {
								// for the preceding segments take the one that departs as late as possible before our flight
								for dateIndex := startDateIndex - int(segmentArrivalDayShift); dateIndex >= departureDateIndex-MaxDaysInFlight; dateIndex-- {
									newSegmentOperates := operatesOn(flightPattern, dateIndex)
									currentSegmentOperates := operatesOn(currentSegment.fp, dateIndex)
									if currentSegmentOperates {
										break
									}
									if newSegmentOperates && !currentSegmentOperates {
										legToSegmentMap[flightPattern.LegNumber] = &FlightPatternBaseAndDate{
											fb:                 flightBase,
											fp:                 flightPattern,
											departureDateIndex: dateIndex,
											arrivalDayShift:    flightPattern.ArrivalDayShift,
											departFromTz:       timeZoneMap[int32(flightBase.DepartureStation)],
											arriveToTz:         timeZoneMap[int32(flightBase.ArrivalStation)],
										}
										break
									}
								}
							} else {
								// for the following segments take the one that departs as early as possible after our flight
								for dateIndex := startDateIndex + int(segmentArrivalDayShift); dateIndex <= departureDateIndex+MaxDaysInFlight; dateIndex++ {
									newSegmentOperates := operatesOn(flightPattern, dateIndex)
									currentSegmentOperates := operatesOn(currentSegment.fp, dateIndex)
									if currentSegmentOperates {
										break
									}
									if newSegmentOperates && !currentSegmentOperates {
										legToSegmentMap[flightPattern.LegNumber] = &FlightPatternBaseAndDate{
											fb:                 flightBase,
											fp:                 flightPattern,
											departureDateIndex: dateIndex,
											arrivalDayShift:    flightPattern.ArrivalDayShift,
											departFromTz:       timeZoneMap[int32(flightBase.DepartureStation)],
											arriveToTz:         timeZoneMap[int32(flightBase.ArrivalStation)],
										}
										break
									}
								}
							}
						}
					}

					arrivalLegNumber := int32(-1)
					for _, segment := range legToSegmentMap {
						if segment.fb.ArrivalStation == int64(arrivalStation.Station.Id) {
							arrivalLegNumber = segment.fp.LegNumber
							break
						}
					}

					if arrivalLegNumber >= departureLegNumber && !departureSegment.fp.IsCodeshare {
						flightEntry.operating = legToSegmentMap
						flightEntry.departureStation = departureStation.Station.Id
						flightEntry.arrivalStation = arrivalStation.Station.Id
						flightsCache[flightEntryKey] = flightEntry
					}
				}
			}
		}
	}

	if len(flightsCache) == 0 {
		return response, nil
	}

	err := service.CreateFlightsP2PResponse(&response, flightsCache, flightKeys)
	if err != nil {
		return response, xerrors.Errorf("cannot create response: %w", err)
	}

	return response, nil
}

func (service *FlightP2PServiceImpl) CreateFlightsP2PResponse(
	response *format.Response,
	flightsCache map[FlightEntryKey]FlightEntryValue,
	flightKeys []FlightKey,
) error {
	flights := make([]format.Flight, 0, len(flightsCache))
	filterByFlight := len(flightKeys) > 0
	flightFilter := make(map[FlightKey]bool)
	for _, flightKey := range flightKeys {
		flightFilter[flightKey] = true
	}
	// One flight may satisfy the search between more than one pair of stations,
	// this cache is to avoid duplications
	responseEntries := make(map[ResponseFlightKey]bool)

	for _, flightEntryValue := range flightsCache {
		var departureSegment *FlightPatternBaseAndDate
		var arrivalSegment *FlightPatternBaseAndDate
		if len(flightEntryValue.operating) == 0 {
			// This should never happen but better return an empty list of flights than panic
			continue
		}
		flightSegments := flightEntryValue.operating
		if len(flightSegments) == 1 {
			firstSegment := getFirstSegment(flightSegments)
			departureSegment = firstSegment
			arrivalSegment = firstSegment
		} else {
			departureSegment = nil
			arrivalSegment = nil
			for index, segment := range flightSegments {
				if int32(segment.fb.DepartureStation) != flightEntryValue.departureStation {
					continue
				}
				departureSegment = segment
				for index2, segment2 := range flightSegments {
					if index2 < index {
						continue
					}
					if int32(segment2.fb.ArrivalStation) == flightEntryValue.arrivalStation {
						arrivalSegment = segment2
						break
					}
				}
				if arrivalSegment != nil {
					break
				}
			}
		}
		if departureSegment == nil {
			return xerrors.Errorf(
				"cannot find departure segment for flight %+v %+v",
				getFirstSegment(flightSegments).fb,
				getFirstSegment(flightSegments).fp,
			)
		}
		if arrivalSegment == nil {
			return xerrors.Errorf(
				"cannot find arrival segment for flight %+v %+v",
				getFirstSegment(flightSegments).fb,
				getFirstSegment(flightSegments).fp,
			)
		}

		if departureSegment.departFromTz == nil {
			return xerrors.Errorf(
				"no timezone in map for departure station %v (%v)",
				departureSegment.fb.DepartureStation,
				departureSegment.fb.DepartureStationCode,
			)
		}
		if arrivalSegment.arriveToTz == nil {
			return xerrors.Errorf(
				"no timezone in map for arrival station %v (%v)",
				arrivalSegment.fb.ArrivalStation,
				arrivalSegment.fb.ArrivalStationCode,
			)
		}
		startSegment := getStartSegment(flightSegments)
		startSegmentTz := service.GetTimeZoneByStationID(int64(startSegment.fb.DepartureStation))
		flightStartDatetime := dtutil.FormatWithTz(
			startSegment.departureDateIndex,
			startSegment.fb.DepartureTimeScheduled,
			startSegmentTz,
		)
		departureDatetime := dtutil.FormatWithTz(
			departureSegment.departureDateIndex,
			departureSegment.fb.DepartureTimeScheduled,
			departureSegment.departFromTz,
		)
		departureTerminal := departureSegment.fb.DepartureTerminal
		arrivalDatetime := dtutil.FormatWithTz(
			arrivalSegment.departureDateIndex+int(arrivalSegment.fp.ArrivalDayShift),
			arrivalSegment.fb.ArrivalTimeScheduled,
			arrivalSegment.arriveToTz,
		)
		arrivalTerminal := arrivalSegment.fb.ArrivalTerminal

		route := threadRoute(flightSegments)

		codeshares := make([]dto.TitledFlight, 0, len(flightEntryValue.codeshares))
		for _, codeshare := range flightEntryValue.codeshares {
			codeshares = append(codeshares, codeshare)
		}

		flightTitle := departureSegment.fp.FlightTitle()
		flightPassesFilter := false
		if _, ok := flightFilter[FlightKey{departureSegment.fb.OperatingCarrier, departureSegment.fb.OperatingFlightNumber}]; ok {
			flightPassesFilter = true
		}

		flight := format.Flight{
			TitledFlight: dto.TitledFlight{
				FlightID: dto.FlightID{
					AirlineID: departureSegment.fb.OperatingCarrier,
					Number:    departureSegment.fb.OperatingFlightNumber,
				},
				Title: flightTitle,
			},
			DepartureDatetime: departureDatetime,
			DepartureTerminal: departureTerminal,
			DepartureStation:  int32(departureSegment.fb.DepartureStation),
			ArrivalDatetime:   arrivalDatetime,
			ArrivalTerminal:   arrivalTerminal,
			ArrivalStation:    int32(arrivalSegment.fb.ArrivalStation),
			StartDatetime:     flightStartDatetime,
			TransportModelID:  departureSegment.fb.AircraftTypeID,
			Route:             route,
		}
		if departureSegment.isBanned || arrivalSegment.isBanned {
			flight.Banned = strconv.FormatBool(true)
		}

		if len(codeshares) > 0 {
			flight.Codeshares = make([]format.CodeshareFlight, 0, len(codeshares))
			for _, codeshare := range codeshares {
				if filterByFlight {
					if _, ok := flightFilter[FlightKey{codeshare.AirlineID, codeshare.Number}]; ok {
						flightPassesFilter = true
					}
				}
				if codeshare.Title == flightTitle {
					continue
				}
				flightCodeshare := format.CodeshareFlight{}
				flightCodeshare.TitledFlight = codeshare
				flight.Codeshares = append(flight.Codeshares, flightCodeshare)
			}
		}

		if filterByFlight && !flightPassesFilter {
			continue
		}

		if departureSegment.fp.IsDop {
			flight.Source = dto.FLIGHTBOARD.String()
		}

		responseFlightKey := ResponseFlightKey{
			OperatingCarrier:  departureSegment.fb.OperatingCarrier,
			FlightNumber:      departureSegment.fb.OperatingFlightNumber,
			DepartureStation:  int32(departureSegment.fb.DepartureStation),
			ArrivalStation:    int32(arrivalSegment.fb.ArrivalStation),
			DepartureDatetime: departureDatetime,
		}
		if _, alreadyExists := responseEntries[responseFlightKey]; alreadyExists {
			continue
		}

		responseEntries[responseFlightKey] = true
		flights = append(flights, flight)
	}
	sortFlightInPlace(flights)
	response.Flights = mergeFlights(flights, service.FlightMergeRuleStorage())
	return nil
}

func putStationToMap(routeStationsMap map[int32]map[int64]void, segmentIndex int32, stationID int64) {
	value, ok := routeStationsMap[segmentIndex]
	if !ok {
		value = make(map[int64]void)
	}
	value[stationID] = none
	routeStationsMap[segmentIndex] = value
}

// It would be nicer to implement this using generic, but they are not available in golang at the moment
func mergeFlights(
	flights []format.Flight, flightMergeRuleStorage flight.FlightMergeRuleStorage) []format.Flight {
	if len(flights) == 0 {
		return flights
	}
	result := make([]format.Flight, 0, len(flights))
	currentFlights := []format.Flight{flights[0]}
	for index := 1; index < len(flights); index++ {
		nextFlight := flights[index]
		if currentFlights[0].DepartureDatetime != nextFlight.DepartureDatetime {
			result = append(result, currentFlights...)
			currentFlights = []format.Flight{nextFlight}
			continue
		}
		nextFlightHasBeenMerged := false
		for curFlightIndex, currentFlight := range currentFlights {
			currentStart, currentEnd := routeEndPoints(currentFlight)
			nextStart, nextEnd := routeEndPoints(nextFlight)
			if currentFlight.DepartureStation != nextFlight.DepartureStation ||
				currentFlight.ArrivalStation != nextFlight.ArrivalStation ||
				currentFlight.ArrivalDatetime != nextFlight.ArrivalDatetime ||
				currentStart != nextStart ||
				currentEnd != nextEnd {
				continue
			}
			currentCarrier := currentFlight.AirlineID
			currentFlightNumber := currentFlight.Number
			nextCarrier := nextFlight.AirlineID
			nextFlightNumber := nextFlight.Number
			if flightMergeRuleStorage.ShouldMerge(currentCarrier, currentFlightNumber, nextCarrier, nextFlightNumber) {
				currentFlights[curFlightIndex] = mergeSegment(currentFlight, nextFlight)
				nextFlightHasBeenMerged = true
			} else if flightMergeRuleStorage.ShouldMerge(nextCarrier, nextFlightNumber, currentCarrier, currentFlightNumber) {
				currentFlights[curFlightIndex] = mergeSegment(nextFlight, currentFlight)
				nextFlightHasBeenMerged = true
			}
		}
		if !nextFlightHasBeenMerged {
			currentFlights = append(currentFlights, nextFlight)
		}
	}
	result = append(result, currentFlights...)
	return result
}

func mergeSegment(operatingFlight, marketingFlight format.Flight) format.Flight {
	marketingCodeshareFlight := format.CodeshareFlight{
		TitledFlight: marketingFlight.TitledFlight,
	}
	operatingFlight.Codeshares = append(operatingFlight.Codeshares, marketingCodeshareFlight)
	if len(marketingFlight.Codeshares) > 0 {
		operatingFlight.Codeshares = append(operatingFlight.Codeshares, marketingFlight.Codeshares...)
	}
	return operatingFlight
}

func routeEndPoints(flight format.Flight) (startPoint, endPoint int64) {
	if len(flight.Route) == 0 {
		return 0, 0
	}
	return flight.Route[0], flight.Route[len(flight.Route)-1]
}

func sortFlightInPlace(flights []format.Flight) {
	sort.Slice(flights, func(i, j int) bool {
		return flights[i].DepartureDatetime < flights[j].DepartureDatetime
	})
}

func operatesOn(flightPattern *structs.FlightPattern, dateIndex int) bool {
	if dtutil.StringDate(flightPattern.OperatingFromDate).ToIntDate() > dtutil.DateCache.Date(dateIndex) {
		return false
	}
	if dtutil.StringDate(flightPattern.OperatingUntilDate).ToIntDate() < dtutil.DateCache.Date(dateIndex) {
		return false
	}
	if !dtutil.OperatesOn(flightPattern.OperatingOnDays, dtutil.DateCache.WeekDay(dateIndex)) {
		return false
	}
	return true
}

func threadRoute(flightSegments map[int32]*FlightPatternBaseAndDate) []int64 {
	// Instead of sorting the map, just call for the leg number in the ascending order
	result := make([]int64, 0, len(flightSegments)/2+1)
	arrivalStation := int64(0)
	for legNumber := 1; legNumber <= MaxSegmentsInFlight; legNumber++ {
		segment, ok := flightSegments[int32(legNumber)]
		if !ok {
			continue
		}
		result = append(result, segment.fb.DepartureStation)
		arrivalStation = segment.fb.ArrivalStation
	}
	result = append(result, arrivalStation)
	return result
}

func getFirstSegment(segments map[int32]*FlightPatternBaseAndDate) *FlightPatternBaseAndDate {
	for _, segment := range segments {
		return segment
	}
	return nil
}

func getStartSegment(segments map[int32]*FlightPatternBaseAndDate) *FlightPatternBaseAndDate {
	var result *FlightPatternBaseAndDate
	for _, segment := range segments {
		if result == nil || result.fp.LegNumber > segment.fp.LegNumber {
			result = segment
		}
	}
	return result
}

func sortFlightPatterns(flightPatterns flight.FlightLegs, departureLegNumber int32) {
	sort.Slice(flightPatterns, func(i, j int) bool {
		if flightPatterns[i][0].LegNumber <= departureLegNumber && flightPatterns[j][0].LegNumber <= departureLegNumber {
			return flightPatterns[i][0].LegNumber > flightPatterns[j][0].LegNumber
		}
		return flightPatterns[i][0].LegNumber < flightPatterns[j][0].LegNumber
	})
}
