package flightp2p

import (
	"sort"
	"time"

	"a.yandex-team.ru/library/go/core/xerrors"
	dto "a.yandex-team.ru/travel/avia/shared_flights/api/internal/services/storage/DTO"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/services/storage/flight_p2p/format"
	"a.yandex-team.ru/travel/avia/shared_flights/api/pkg/structs"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/dtutil"
	"a.yandex-team.ru/travel/proto/shared_flights/snapshots"
)

/*
	Возвращает список дат в заданном радиусе (диапазоне), для которых есть прямые рейсы между заданными наборами аэропортов.
	В алгоритме вычисления списка есть не соответствующее реальности допущение для маршруток: что если маршрутка вылетает
	из нужного аэропорта в нужный день и в её расписании есть нужный аэропорт прибытия, то мы считаем, что нужный нам вылет
	обязательно долетит до нужного аэропорта, рано или поздно.
	В реальности это не так, могут существовать маршрутки, летающие разными маршрутами в разные дни. В настоящее время таких
	примеров или нет, или крайне мало. Корректная реализация для таких маршруток есть в flight_p2p, но текущий алгоритм
	требует слишком много цпу, чтобы его использовать в большом количестве запросов; если текущий подход когда-нибудь
	станет проблемой, лучшим выходом будет написать кеширование ниток для маршруток, чтобы и алгоритм был простой,
	и цпу мало расходовалось. Сейчас приходится срезать этот угол.
*/
func (service *FlightP2PServiceImpl) GetFlightsP2PByDate(
	departFrom []*snapshots.TStationWithCodes,
	arriveTo []*snapshots.TStationWithCodes,
	flightDate TimeParam,
	daysCount int,
	nationalVersion string,
	showBanned bool,
	debug bool,
) (format.DirectDatesAndFlights, error) {
	type FlightPatternAndBase struct {
		fp *structs.FlightPattern
		fb structs.FlightBase
	}
	departureStations := make([]int32, 0, len(departFrom))
	for _, station := range departFrom {
		departureStations = append(departureStations, station.Station.Id)
	}
	arrivalStations := make([]int32, 0, len(arriveTo))
	for _, station := range arriveTo {
		arrivalStations = append(arrivalStations, station.Station.Id)
	}
	response := format.DirectDatesAndFlights{
		DepartureStations: departureStations,
		ArrivalStations:   arrivalStations,
		FlightDepartures:  make([]format.FlightDeparture, 0),
		Dates:             make([]string, 0),
	}
	if daysCount < 0 || daysCount > 7 {
		return response, xerrors.Errorf("invalid days count: %d", daysCount)
	}
	from := TimeParam{
		Value:   flightDate.Value.AddDate(0, 0, -daysCount),
		IsLocal: flightDate.IsLocal,
	}
	until := TimeParam{
		Value:   flightDate.Value.AddDate(0, 0, daysCount),
		IsLocal: flightDate.IsLocal,
	}

	timeZoneMap := make(map[int32]*time.Location)
	for _, station := range departFrom {
		if _, alreadyProcessed := timeZoneMap[station.Station.Id]; alreadyProcessed {
			continue
		}
		tz := service.GetTimeZone(station.Station)
		if tz == nil {
			return response, xerrors.Errorf("cannot load timezone for station  %+v", station.Station.Id)
		}
		timeZoneMap[station.Station.Id] = tz
	}
	for _, station := range arriveTo {
		if _, alreadyProcessed := timeZoneMap[station.Station.Id]; alreadyProcessed {
			continue
		}
		tz := service.GetTimeZone(station.Station)
		if tz == nil {
			return response, xerrors.Errorf("cannot load timezone for station  %+v", station.Station.Id)
		}
		timeZoneMap[station.Station.Id] = tz
	}

	datesCache := make(map[string]bool)

	for _, departureStation := range departFrom {
		for _, arrivalStation := range arriveTo {
			flightKeys, ok := service.Storage.FlightStorage().GetFlightsP2P(
				int64(departureStation.Station.Id),
				int64(arrivalStation.Station.Id),
			)
			if !ok {
				continue
			}

			departFromTz := timeZoneMap[departureStation.Station.Id]

			fromAsTime := from.In(departFromTz)
			untilAsTime := until.In(departFromTz)
			fromAsInt := dtutil.DateToInt(fromAsTime)
			untilAsInt := dtutil.DateToInt(untilAsTime)
			fromDateIndex := dtutil.DateCache.IndexOfIntDateP(fromAsInt)
			untilDateIndex := dtutil.DateCache.IndexOfIntDateP(untilAsInt)

			for _, flightKey := range flightKeys {
				flightPatterns, ok := service.Storage.FlightStorage().GetFlightsByKey(flightKey)
				if !ok {
					continue
				}

				// find departure segment since we filter by the departure time
				departureSegments := []FlightPatternAndBase{}
				for _, legFlights := range flightPatterns {
					for _, flightPattern := range legFlights {
						flightBase, err := service.Storage.FlightStorage().GetFlightBase(
							flightPattern.FlightBaseID, flightPattern.IsDop)
						if err != nil {
							return response, err
						}
						// skip half-flights in the point-to-point response - they could not be shown anyway
						if !dtutil.IntTime(flightBase.DepartureTimeScheduled).IsValid() ||
							!dtutil.IntTime(flightBase.ArrivalTimeScheduled).IsValid() {
							continue
						}
						if flightBase.DepartureStation == int64(departureStation.Station.Id) {
							departureSegments = append(
								departureSegments,
								FlightPatternAndBase{
									fp: flightPattern,
									fb: flightBase,
								},
							)
						}
					}
				}
				if len(departureSegments) == 0 {
					// happens for dop flights
					continue
				}
				for departureDateIndex := fromDateIndex; departureDateIndex <= untilDateIndex; departureDateIndex++ {
					var departureSegment *FlightPatternAndBase
					departureDate := string(dtutil.DateCache.Date(departureDateIndex).StringDateDashed())
					hasFlightsAlready, ok := datesCache[departureDate]
					if ok && hasFlightsAlready {
						continue
					}
					isBanned := false
					for _, segment := range departureSegments {
						if !operatesOn(segment.fp, departureDateIndex) {
							continue
						}
						// filter by departure time
						departureTime := dtutil.LocalTime(departureDateIndex, segment.fb.DepartureTimeScheduled, departFromTz)
						// Note. We assume that if the departure segment of flight is banned, the rest are banned as well;
						// and vice versa, if the departure segment is not banned, no flight segments are banned.
						if service.Storage.BlacklistRuleStorage().IsBanned(
							segment.fb, segment.fp, dtutil.FormatDateIso(departureTime), nationalVersion) {
							if !showBanned {
								continue
							} else {
								isBanned = true
							}
						}
						departureSegment = &segment
						break
					}
					if departureSegment == nil {
						continue
					}
					datesCache[departureDate] = true
					if debug {
						flight := format.FlightDeparture{
							TitledFlight: dto.TitledFlight{
								FlightID: dto.FlightID{
									AirlineID: departureSegment.fp.MarketingCarrier,
									Number:    departureSegment.fp.MarketingFlightNumber,
								},
								Title: departureSegment.fp.FlightTitle(),
							},
							DepartureDatetime: dtutil.FormatWithTz(
								departureDateIndex,
								departureSegment.fb.DepartureTimeScheduled,
								departFromTz,
							),
							DepartureTerminal: departureSegment.fb.DepartureTerminal,
							DepartureStation:  int32(departureSegment.fb.DepartureStation),
							ArrivalStation:    int32(departureSegment.fb.ArrivalStation),
							TransportModelID:  departureSegment.fb.AircraftTypeID,
							Source:            departureSegment.fb.Source,
							Banned:            isBanned,
						}
						response.FlightDepartures = append(response.FlightDepartures, flight)
					}
				}
			}
		}
	}

	for date := range datesCache {
		response.Dates = append(response.Dates, date)
	}
	sort.Slice(response.Dates, func(i, j int) bool {
		return response.Dates[i] < response.Dates[j]
	})

	return response, nil
}

// Verifies that at least one backward date does not precede at least one forward date.
// We consicely ignore the case when there's exactly one backward date that is equal to the max forward date,
// as spending time and resources to test whether departure time and arrival time would actually allow the user to travel
// does not seem not deliver much of a benefit for the current state of the Russia-touching itineraries.
func AtLeastOneBackwardDateIsEqualOrAfterForwardDate(forwardDates, backwardDates []string) bool {
	if len(forwardDates) == 0 || len(backwardDates) == 0 {
		return false
	}
	minForwardDate := forwardDates[0]
	// find min forwardDate
	for _, forwardDate := range forwardDates {
		if minForwardDate >= forwardDate {
			minForwardDate = forwardDate
		}
	}
	// see if we have at least one backward date that does not precede minForwardDate
	for _, backwardDate := range backwardDates {
		if backwardDate >= minForwardDate {
			return true
		}
	}
	return false
}
