package flightschedule

import (
	"net/http"
	"sort"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/services/storage/flight"
	flightp2p "a.yandex-team.ru/travel/avia/shared_flights/api/internal/services/storage/flight_p2p"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/services/storage/flight_schedule/format"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/services/storage/segment"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/storage"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/utils"
	"a.yandex-team.ru/travel/avia/shared_flights/api/pkg/structs"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/dtutil"
)

type FlightScheduleService interface {
	GetFlightSchedule(
		carrierCode, flightNumber, nationalVersion string,
		showBanned bool,
		startScheduleDate dtutil.IntDate) (response format.Response, err error)
}

func NewFlightScheduleService(
	storage *storage.Storage,
	flightService flight.FlightService,
) FlightScheduleService {
	return &flightScheduleServiceImpl{
		Storage:       storage,
		FlightService: flightService,
	}
}

type flightScheduleServiceImpl struct {
	*storage.Storage
	flight.FlightService
}

func (service *flightScheduleServiceImpl) GetFlightSchedule(
	carrierCode, flightNumber, nationalVersion string,
	showBanned bool,
	startScheduleDate dtutil.IntDate) (response format.Response, err error) {
	type ScheduleVariationKey struct {
		routeHash        string
		transportModelID int32
	}
	response = format.Response{}
	defer func() {
		if r := recover(); r != nil {
			response = format.Response{}
			err = &utils.ErrorWithHTTPCode{
				HTTPCode:     http.StatusInternalServerError,
				ErrorMessage: xerrors.Errorf("getFlightSchedule panicked: %v", r).Error(),
			}
			return
		}
	}()

	flightPatterns, err := service.FlightService.GetFlightPatterns(flight.NewCarrierParamByText(carrierCode), flightNumber)
	if err != nil {
		return response, err
	}

	filteredFlightPatterns := make([]*structs.FlightPattern, 0, len(flightPatterns))
	for _, flightPattern := range flightPatterns {
		if flightPattern.IsDop {
			continue
		}
		filteredFlightPatterns = append(filteredFlightPatterns, flightPattern)
	}

	if len(filteredFlightPatterns) == 0 {
		err = &utils.ErrorWithHTTPCode{
			HTTPCode:     http.StatusNotFound,
			ErrorMessage: xerrors.Errorf("schedule for the flight %v %v is not found", carrierCode, flightNumber).Error(),
		}
		return
	}

	var segments segment.SegmentGroup
	if segments, err = service.AttachFlightBases(filteredFlightPatterns, showBanned, true, nationalVersion); err != nil {
		return response, xerrors.Errorf("cannot attach flight bases: %w", err)
	}

	response.Title = filteredFlightPatterns[0].FlightTitle()
	response.CarrierID = filteredFlightPatterns[0].MarketingCarrier

	responseSchedules := make([]format.Schedule, 0)
	schedulesMap := make(map[ScheduleVariationKey]segment.RouteSchedule)
	for schedule := range segment.GenerateSchedules(segments) {
		key := ScheduleVariationKey{
			routeHash:        schedule.Route.Hash(),
			transportModelID: schedule.TransportModelID,
		}
		currentValue, hasValue := schedulesMap[key]
		if !hasValue {
			schedulesMap[key] = schedule
		} else {
			currentValue.Mask.AddMask(schedule.Mask)
		}
	}
	for _, schedule := range schedulesMap {
		responseSchedule := format.Schedule{}
		responseSchedule.Route = make([]format.StopPoint, 0, len(schedule.Route))
		for _, point := range schedule.Route {
			stopPoint := format.StopPoint{
				AirportID:         point.AirportID,
				AirportCode:       point.AirportCode,
				ArrivalTime:       point.ArrivalTime,
				ArrivalDayShift:   int(point.ArrivalDayShift),
				ArrivalTerminal:   point.ArrivalTerminal,
				DepartureTime:     point.DepartureTime,
				DepartureDayShift: int(point.DepartureDayShift),
				DepartureTerminal: point.DepartureTerminal,
			}
			responseSchedule.Route = append(responseSchedule.Route, stopPoint)
		}
		responseSchedule.TransportModelID = schedule.TransportModelID
		responseSchedule.Masks = flightp2p.GenerateMasks(schedule.Mask, startScheduleDate)
		if len(responseSchedule.Masks) > 0 {
			responseSchedules = append(responseSchedules, responseSchedule)
		}
	}
	sort.Slice(responseSchedules, func(i, j int) bool {
		return responseSchedules[i].Masks[0].From < responseSchedules[j].Masks[0].From
	})
	response.Schedules = responseSchedules
	return response, nil
}
