package flightboard

import (
	"sort"
	"time"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/services/flightdata"
	dto "a.yandex-team.ru/travel/avia/shared_flights/api/internal/services/storage/DTO"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/services/storage/flight"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/services/storage/flightboard/format"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/services/storage/flightstatus"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/services/storage/threads"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/services/storage/timezone"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/storage"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/storage/station"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/storage/status"
	"a.yandex-team.ru/travel/avia/shared_flights/api/pkg/structs"
	dir "a.yandex-team.ru/travel/avia/shared_flights/lib/go/direction"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/dtutil"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/math"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/timeutil"
	"a.yandex-team.ru/travel/proto/shared_flights/snapshots"
)

type DateTimeFilterResult int8

const (
	DoesNotFit DateTimeFilterResult = 1 + iota
	FitsByActualTime
	FitsByScheduledTime
	end
)

type FlightBoardService interface {
	GetFlightBoard(
		station *snapshots.TStationWithCodes,
		from time.Time,
		until time.Time,
		limit int,
		sortAscending bool,
		direction dir.Direction,
		terminal string,
		nationalVersion string,
		flightFilter string,
		now time.Time,
	) (format.Response, error)

	GetStationFlights(
		station *snapshots.TStationWithCodes,
		from time.Time,
		until time.Time,
		limit int,
		sortAscending bool,
		direction dir.Direction,
		terminal string,
		nationalVersion string,
		now time.Time,
	) (format.FlightStationResponse, error)
}

func NewFlightBoardService(
	storage *storage.Storage, timeZoneUtil timezone.TimeZoneUtil, threadRoute threads.ThreadRouteService) FlightBoardService {
	return &FlightBoardServiceImpl{
		Storage:            storage,
		TimeZoneUtil:       timeZoneUtil,
		ThreadRouteService: threadRoute,
	}
}

type FlightBoardServiceImpl struct {
	*storage.Storage
	timezone.TimeZoneUtil
	threads.ThreadRouteService
}

type FlightBaseInfo struct {
	FlightBase structs.FlightBase
	Operating  []*structs.FlightPattern
	Codeshares []*structs.FlightPattern
	Statuses   *status.SynchronizedFlightStatusMap
}

type FlightOnDateWithCodeshares struct {
	*flightdata.FlightData
	Codeshares []*structs.FlightPattern
}

type FlightPatternAndSortKey struct {
	flightPattern  *structs.FlightPattern
	operatingFrom  dtutil.IntDate
	operatingUntil dtutil.IntDate
	sortKey        int32 // Departure or arrival date and time
}

type tFlightBaseID = int32

func (service *FlightBoardServiceImpl) GetFlightBoard(
	station *snapshots.TStationWithCodes,
	from time.Time,
	until time.Time,
	limit int,
	sortAscending bool,
	direction dir.Direction,
	terminal string,
	nationalVersion string,
	flightFilter string,
	now time.Time,
) (format.Response, error) {
	response := format.Response{
		Station:   station.Station.Id,
		Direction: direction.String(),
		Flights:   make([]format.Flight, 0),
	}

	segments, err := service.getFlightBoardInternal(station, from, until, limit, sortAscending, direction, terminal, nationalVersion, now)
	if err != nil {
		return response, err
	}

	if len(segments) == 0 {
		return response, nil
	}

	err = service.CreateFlightBoardResponse(&response, segments, direction, true, flightFilter, now)
	if err != nil {
		return response, xerrors.Errorf("cannot create response: %w", err)
	}

	return response, nil
}

func (service *FlightBoardServiceImpl) GetStationFlights(
	station *snapshots.TStationWithCodes,
	from time.Time,
	until time.Time,
	limit int,
	sortAscending bool,
	direction dir.Direction,
	terminal string,
	nationalVersion string,
	now time.Time,
) (format.FlightStationResponse, error) {
	response := format.FlightStationResponse{
		Station:   station.Station.Id,
		Direction: direction.String(),
		Flights:   make([]flight.FlightSegmentWithCodeshares, 0),
	}

	segments, err := service.getFlightBoardInternal(station, from, until, limit, sortAscending, direction, terminal, nationalVersion, now)
	if err != nil {
		return response, err
	}

	if len(segments) == 0 {
		return response, nil
	}

	err = service.CreateFlightStationResponse(&response, segments, direction, true, now)
	if err != nil {
		return response, xerrors.Errorf("cannot create response: %w", err)
	}

	return response, nil
}

func (service *FlightBoardServiceImpl) getFlightBoardInternal(
	station *snapshots.TStationWithCodes,
	from time.Time,
	until time.Time,
	limit int,
	sortAscending bool,
	direction dir.Direction,
	terminal string,
	nationalVersion string,
	now time.Time,
) ([]FlightOnDateWithCodeshares, error) {
	var segments []FlightOnDateWithCodeshares
	if !direction.IsValid() {
		return segments, xerrors.Errorf("invalid direction: %+v", direction)
	}

	if from.After(until) {
		return segments, xerrors.Errorf("invalid date range: (%+v) - (%+v)", from, until)
	}

	stationTimezone := service.GetTimeZoneByStationID(int64(station.Station.Id))
	if stationTimezone == nil {
		return segments, xerrors.Errorf("cannot load timezone for station  %+v", station.Station.Id)
	}

	from = from.In(stationTimezone)
	until = until.In(stationTimezone)

	// Ease restrictions to let the delayed over midnight flights from yesterday squeeze in
	searchFrom := from.AddDate(0, 0, -1)
	if direction == dir.ARRIVAL {
		// TODO(mikhailche): optimize after we calculate global day shift for arrival or start making threads
		searchFrom = searchFrom.AddDate(0, 0, -1)
	}

	allFlightPatterns, hasValue := service.FlightBoard.GetFlightPatterns(station, direction)
	if !hasValue {
		return segments, nil
	}
	flightPatternExtras, _ := service.FlightBoard.GetExtras(station)

	operatingPatterns, err := service.getMatchingFlightPatterns(
		allFlightPatterns,
		flightPatternExtras,
		searchFrom,
		until,
		direction,
		stationTimezone,
		sortAscending,
	)
	if err != nil {
		return segments, xerrors.Errorf("cannot get operating flight patterns: %w", err)
	}
	if len(operatingPatterns) == 0 {
		return segments, nil
	}

	dateTimeFilter := DateTimeFilter(direction, from, until)
	terminalFilter := TerminalFilter(direction, terminal)

	segments, err = service.expandSegments(
		allFlightPatterns,
		operatingPatterns,
		flightPatternExtras,
		searchFrom,
		until,
		dateTimeFilter,
		terminalFilter,
		service.TimeZoneUtil,
		nationalVersion,
		limit,
	)
	if err != nil {
		return segments, xerrors.Errorf("cannot expand flights: %w", err)
	}

	if len(segments) == 0 {
		return segments, nil
	}

	sortSegmentsInPlace(segments, direction, sortAscending)
	segments = service.mergeSegments(segments)

	if limit > 0 {
		segments = segments[:math.Min(limit, len(segments))]
	}

	return segments, nil
}

// TODO(mikhailche): test me
func (service *FlightBoardServiceImpl) getMatchingFlightPatterns(
	flightPatterns []*structs.FlightPattern,
	flightPatternExtras map[int32]station.FlightPatternExtras,
	searchFrom time.Time,
	until time.Time,
	direction dir.Direction,
	stationTimezone *time.Location,
	sortAscending bool,
) ([]FlightPatternAndSortKey, error) {
	// No airport in the world has more than 2500 flights a day,
	// so this nearly guarantees no more than three memory allocation calls
	expectedSelectedPatterns := 1000
	if len(flightPatterns) < expectedSelectedPatterns {
		expectedSelectedPatterns = len(flightPatterns)
	}
	selectedPatterns := make([]FlightPatternAndSortKey, 0, expectedSelectedPatterns)
	searchFromIntDate := dtutil.DateToInt(searchFrom)
	untilIntDate := dtutil.DateToInt(until)
	for _, flightPattern := range flightPatterns {
		if flightPattern.IsCodeshare {
			continue
		}
		extras, extrasOk := flightPatternExtras[flightPattern.ID]
		if !extrasOk {
			return nil, xerrors.Errorf("cannot get flight pattern extras for (%d)", flightPattern.ID)
		}
		// Requested left bound should be before operating right bound
		operatingFrom := extras.OperatingFrom // dtutil.StringDate(flightPattern.OperatingFromDate).ToIntDate()
		if operatingFrom > untilIntDate {
			continue
		}
		// Requested right bound should be after operating left bound
		operatingUntil := extras.OperatingUntil // dtutil.StringDate(flightPattern.OperatingUntilDate).ToIntDate()
		if operatingUntil < searchFromIntDate {
			continue
		}
		var sortKey int32
		if direction == dir.DEPARTURE {
			sortKey = extras.DepartureTimeScheduled
		} else {
			sortKey = extras.ArrivalTimeScheduled
		}
		selectedPatterns = append(selectedPatterns, FlightPatternAndSortKey{
			flightPattern:  flightPattern,
			operatingFrom:  operatingFrom,
			operatingUntil: operatingUntil,
			sortKey:        sortKey,
		})
	}

	if sortAscending {
		sort.Slice(selectedPatterns, func(i, j int) bool {
			return selectedPatterns[i].sortKey < selectedPatterns[j].sortKey
		})
	} else {
		sort.Slice(selectedPatterns, func(i, j int) bool {
			return selectedPatterns[i].sortKey > selectedPatterns[j].sortKey
		})
	}

	return selectedPatterns, nil
}

func (service *FlightBoardServiceImpl) expandSegments(
	flightPatterns []*structs.FlightPattern,
	operatingPatterns []FlightPatternAndSortKey,
	flightPatternExtras map[int32]station.FlightPatternExtras,
	searchFrom time.Time,
	until time.Time,
	dateTimeFilter func(*FlightOnDateWithCodeshares) DateTimeFilterResult,
	terminalFilter func(*FlightOnDateWithCodeshares) bool,
	tzProvider flightdata.TimezoneProvider,
	nationalVersion string,
	limit int,
) ([]FlightOnDateWithCodeshares, error) {

	// Leg number is not needed for deduplication since all legs in the set either take off or land at the same station
	type DeduplicationKey struct {
		date    dtutil.IntDate
		carrier int32
		number  string
	}
	deduplicatedFlights := make(map[DeduplicationKey]FlightOnDateWithCodeshares)
	flightData := make(map[tFlightBaseID]*FlightBaseInfo)
	fromIntDate := dtutil.DateToInt(searchFrom)
	untilIntDate := dtutil.DateToInt(until)
	// first, iterate over operating flight patterns, then over codeshares

	foundFlightsCount := 0
	for _, scheduledDepartureDate := range dtutil.IterDate(fromIntDate, untilIntDate) {
		for _, operatingPattern := range operatingPatterns {
			flightPattern := operatingPattern.flightPattern
			flightBaseInfo, ok := flightData[flightPattern.FlightBaseID]
			if !ok {
				flightBaseInfo = &FlightBaseInfo{}
				flightBase, err := service.FlightStorage().GetFlightBase(flightPattern.FlightBaseID, flightPattern.IsDop)
				if err != nil {
					return nil, xerrors.Errorf("cannot get flight base (%d): %w", flightPattern.FlightBaseID, err)
				}
				flightBaseInfo.Statuses, _ = service.StatusStorage().GetFlightStatuses(flightBase.GetBucketKey())
				flightBaseInfo.FlightBase = flightBase
			}
			flightBase := flightBaseInfo.FlightBase
			flightDataBase := flightdata.NewFlightDataBase(
				flightBase,
				flightPattern,
				nil,
				tzProvider,
			)

			if !dtutil.OperatesOn(flightPattern.OperatingOnDays, scheduledDepartureDate.Weekday) {
				continue
			}
			if operatingPattern.operatingFrom > scheduledDepartureDate.IntDate {
				continue
			}
			if operatingPattern.operatingUntil < scheduledDepartureDate.IntDate {
				continue
			}
			scheduledDepartureDateISO := scheduledDepartureDate.IntDate.StringDateDashed()
			if service.Storage.BlacklistRuleStorage().IsBanned(flightBase, flightPattern, scheduledDepartureDateISO, nationalVersion) {
				continue
			}
			flightData := flightDataBase.ForDate(scheduledDepartureDate.IntDate)
			flightData.FlightStatus, _ = flightBaseInfo.Statuses.GetStatus(scheduledDepartureDate.IntDate)

			flightOnDate := FlightOnDateWithCodeshares{
				&flightData,
				make([]*structs.FlightPattern, 0),
			}
			deduplicationKey := DeduplicationKey{
				date:    scheduledDepartureDate.IntDate,
				carrier: flightPattern.MarketingCarrier,
				number:  flightPattern.MarketingFlightNumber,
			}
			if _, ok := deduplicatedFlights[deduplicationKey]; ok {
				continue
			}

			if terminalFilter != nil && !terminalFilter(&flightOnDate) {
				continue
			}

			flightFitsByScheduledTime := true
			if dateTimeFilter != nil {
				dateTimeFilterResult := dateTimeFilter(&flightOnDate)
				if dateTimeFilterResult == DoesNotFit {
					continue
				}
				if dateTimeFilterResult != FitsByScheduledTime {
					flightFitsByScheduledTime = false
				}
			}

			if _, knownFlight := deduplicatedFlights[deduplicationKey]; !knownFlight {
				if flightPattern.ArrivalDayShift > 0 {
					flightFitsByScheduledTime = false
				}
			}
			if flightFitsByScheduledTime {
				foundFlightsCount++
			}
			deduplicatedFlights[deduplicationKey] = flightOnDate
			if limit > 0 && foundFlightsCount >= limit {
				break
			}
		}
		if limit > 0 && foundFlightsCount >= limit {
			break
		}
	}

	// now iterate over codeshares
	for _, flightPattern := range flightPatterns {
		if !flightPattern.IsCodeshare {
			continue
		}
		// Requested left bound should be before operating right bound
		var operatingFrom, operatingUntil dtutil.IntDate
		extras, extrasOk := flightPatternExtras[flightPattern.ID]
		if extrasOk {
			operatingFrom = extras.OperatingFrom
			operatingUntil = extras.OperatingUntil
		} else {
			operatingFrom = dtutil.StringDate(flightPattern.OperatingFromDate).ToIntDate()
			operatingUntil = dtutil.StringDate(flightPattern.OperatingUntilDate).ToIntDate()
		}
		if operatingFrom > untilIntDate {
			continue
		}
		// Requested right bound should be after operating left bound
		if operatingUntil < fromIntDate {
			continue
		}
		flightBaseInfo, ok := flightData[flightPattern.FlightBaseID]
		if !ok {
			flightBaseInfo = &FlightBaseInfo{}
			flightBase, err := service.FlightStorage().GetFlightBase(flightPattern.FlightBaseID, flightPattern.IsDop)
			if err != nil {
				return nil, xerrors.Errorf("cannot get flight base (%d): %w", flightPattern.FlightBaseID, err)
			}
			flightBaseInfo.FlightBase = flightBase
		}

		for _, scheduledDepartureDate := range dtutil.IterDate(fromIntDate, untilIntDate) {
			if !dtutil.OperatesOn(flightPattern.OperatingOnDays, scheduledDepartureDate.Weekday) {
				continue
			}
			if operatingFrom > scheduledDepartureDate.IntDate {
				continue
			}
			if operatingUntil < scheduledDepartureDate.IntDate {
				continue
			}
			scheduledDepartureDateStr := scheduledDepartureDate.IntDate.StringDateDashed()
			flightBase := flightBaseInfo.FlightBase
			if service.Storage.BlacklistRuleStorage().IsBanned(flightBase, flightPattern, scheduledDepartureDateStr, nationalVersion) {
				continue
			}

			deduplicationKey := DeduplicationKey{
				date:    scheduledDepartureDate.IntDate,
				carrier: flightBase.OperatingCarrier,
				number:  flightBase.OperatingFlightNumber,
			}
			flightOnDate, ok := deduplicatedFlights[deduplicationKey]
			if !ok {
				continue
			}

			if flightPattern.OperatingFlightPatternID != 0 && flightPattern.OperatingFlightPatternID != flightOnDate.FlightPattern.ID {
				continue
			}
			if !dtutil.OperatesOn(flightOnDate.FlightPattern.OperatingOnDays, scheduledDepartureDate.Weekday) {
				continue
			}
			if service.Storage.BlacklistRuleStorage().IsBanned(
				flightOnDate.FlightBase, flightPattern, scheduledDepartureDateStr, nationalVersion) {
				continue
			}
			flightOnDate.Codeshares = append(flightOnDate.Codeshares, flightPattern)

			deduplicatedFlights[deduplicationKey] = flightOnDate
		}
	}

	flights := make([]FlightOnDateWithCodeshares, 0, len(deduplicatedFlights))
	for _, v := range deduplicatedFlights {
		flights = append(flights, v)
	}
	return flights, nil
}

func (service *FlightBoardServiceImpl) mergeSegments(flights []FlightOnDateWithCodeshares) []FlightOnDateWithCodeshares {
	if len(flights) == 0 {
		return flights
	}
	result := make([]FlightOnDateWithCodeshares, 0, len(flights))
	flightMergeRuleStorage := service.Storage.FlightMergeRuleStorage()
	currentFlights := []FlightOnDateWithCodeshares{flights[0]}
	for index := 1; index < len(flights); index++ {
		nextFlight := flights[index]
		if currentFlights[0].FlightDepartureDate != nextFlight.FlightDepartureDate ||
			currentFlights[0].FlightBase.DepartureTimeScheduled != nextFlight.FlightBase.DepartureTimeScheduled {
			result = append(result, currentFlights...)
			currentFlights = []FlightOnDateWithCodeshares{nextFlight}
			continue
		}
		nextFlightHasBeenMerged := false
		for curFlightIndex, currentFlight := range currentFlights {
			if currentFlight.FlightBase.DepartureStation != nextFlight.FlightBase.DepartureStation ||
				currentFlight.FlightBase.ArrivalStation != nextFlight.FlightBase.ArrivalStation ||
				currentFlight.FlightBase.ArrivalTimeScheduled != nextFlight.FlightBase.ArrivalTimeScheduled {
				continue
			}
			currentCarrier := currentFlight.FlightPattern.MarketingCarrier
			currentFlightNumber := currentFlight.FlightPattern.MarketingFlightNumber
			nextCarrier := nextFlight.FlightPattern.MarketingCarrier
			nextFlightNumber := nextFlight.FlightPattern.MarketingFlightNumber
			if flightMergeRuleStorage.ShouldMerge(currentCarrier, currentFlightNumber, nextCarrier, nextFlightNumber) {
				currentFlights[curFlightIndex] = mergeSegment(currentFlight, nextFlight)
				nextFlightHasBeenMerged = true
			} else if flightMergeRuleStorage.ShouldMerge(nextCarrier, nextFlightNumber, currentCarrier, currentFlightNumber) {
				currentFlights[curFlightIndex] = mergeSegment(nextFlight, currentFlight)
				nextFlightHasBeenMerged = true
			}
		}
		if !nextFlightHasBeenMerged {
			currentFlights = append(currentFlights, nextFlight)
		}
	}
	result = append(result, currentFlights...)
	return result
}

func mergeSegment(operatingFlight, marketingFlight FlightOnDateWithCodeshares) FlightOnDateWithCodeshares {
	operatingFlight.Codeshares = append(operatingFlight.Codeshares, marketingFlight.FlightPattern)
	if len(marketingFlight.Codeshares) > 0 {
		operatingFlight.Codeshares = append(operatingFlight.Codeshares, marketingFlight.Codeshares...)
	}
	return operatingFlight
}

func sortSegmentsInPlace(flights []FlightOnDateWithCodeshares, direction dir.Direction, sortAscending bool) {
	if direction == dir.DEPARTURE {
		if sortAscending {
			sort.Slice(flights, func(i, j int) bool {
				timeI := timeutil.Coalesce(flights[i].ActualDeparture(), flights[i].ScheduledDeparture())
				timeJ := timeutil.Coalesce(flights[j].ActualDeparture(), flights[j].ScheduledDeparture())
				return timeI.Before(timeJ)
			})
		} else {
			sort.Slice(flights, func(i, j int) bool {
				timeI := timeutil.Coalesce(flights[i].ActualDeparture(), flights[i].ScheduledDeparture())
				timeJ := timeutil.Coalesce(flights[j].ActualDeparture(), flights[j].ScheduledDeparture())
				return timeI.After(timeJ)
			})
		}
	} else {
		if sortAscending {
			sort.Slice(flights, func(i, j int) bool {
				timeI := timeutil.Coalesce(flights[i].ActualArrival(), flights[i].ScheduledArrival())
				timeJ := timeutil.Coalesce(flights[j].ActualArrival(), flights[j].ScheduledArrival())
				return timeI.Before(timeJ)
			})
		} else {
			sort.Slice(flights, func(i, j int) bool {
				timeI := timeutil.Coalesce(flights[i].ActualArrival(), flights[i].ScheduledArrival())
				timeJ := timeutil.Coalesce(flights[j].ActualArrival(), flights[j].ScheduledArrival())
				return timeI.After(timeJ)
			})
		}
	}
}

func (service *FlightBoardServiceImpl) CreateFlightBoardResponse(
	response *format.Response,
	segments []FlightOnDateWithCodeshares,
	direction dir.Direction,
	needStatus bool,
	flightFilter string,
	now time.Time,
) error {
	flights := make([]format.Flight, 0, len(segments))
	flightFilterIsEmpty := flightFilter == ""

	for _, segment := range segments {
		datetime := scheduledDateForDirection(segment.FlightData, direction)
		terminal := scheduledTerminalForDirection(segment.FlightData, direction)
		route, ok, err := service.ThreadRoute(segment.FlightDataBase)
		if err != nil {
			return xerrors.Errorf("cannot assemble route for segment [%v]/[%v]: %w", segment.FlightBase, segment.FlightPattern, err)
		}
		if !ok {
			continue
		}

		flightTitle := segment.FlightPattern.FlightTitle()
		if !flightFilterIsEmpty && flightTitle != flightFilter {
			continue
		}

		flight := format.Flight{
			TitledFlight: dto.TitledFlight{
				FlightID: dto.FlightID{
					AirlineID: segment.FlightBase.OperatingCarrier,
					Number:    segment.FlightBase.OperatingFlightNumber,
				},
				Title: flightTitle,
			},
			Datetime:         datetime,
			Terminal:         terminal,
			Codeshares:       uniqCodeshares(segment.Codeshares, flightTitle),
			Status:           flightstatus.GetFlightStatus(segment.FlightData, segment.FlightData, now),
			Route:            route.StationsList(),
			StationFrom:      int32(segment.FlightBase.DepartureStation),
			StationTo:        int32(segment.FlightBase.ArrivalStation),
			TransportModelID: segment.FlightBase.AircraftTypeID,
		}

		flightStartDate := segment.FlightData.ScheduledDeparture()
		timezone := service.GetTimeZoneByStationID(route[0].StationID)
		if !flightStartDate.IsZero() && timezone != nil {
			flightStartDate = flightStartDate.AddDate(0, 0, route[0].FlightDepartureShift)
			flight.StartDatetime = time.Date(
				flightStartDate.Year(),
				flightStartDate.Month(),
				flightStartDate.Day(),
				int(route[0].DepartureTime/100),
				int(route[0].DepartureTime%100),
				0,
				0,
				timezone,
			).Format(time.RFC3339)
		}

		if segment.FlightData.FlightPattern.IsDop {
			flight.Source = dto.FLIGHTBOARD.String()
		}

		flights = append(flights, flight)
	}
	response.Flights = flights
	return nil
}

func (service *FlightBoardServiceImpl) CreateFlightStationResponse(
	response *format.FlightStationResponse,
	segments []FlightOnDateWithCodeshares,
	direction dir.Direction,
	needStatus bool,
	now time.Time,
) error {
	flights := make([]flight.FlightSegmentWithCodeshares, 0, len(segments))

	for _, segment := range segments {
		// TODO(u-jeen): implement and set CreatedAtUTC, UpdatedAtUTC fields
		flightTitle := segment.FlightPattern.FlightTitle()
		flight := flight.FlightSegmentWithCodeshares{
			FlightSegment: flight.FlightSegment{
				CompanyIata:       segment.FlightBase.OperatingCarrierCode,
				CompanyRaspID:     segment.FlightBase.OperatingCarrier,
				Number:            segment.FlightBase.OperatingFlightNumber,
				AirportFromIata:   segment.FlightBase.DepartureStationCode,
				AirportFromRaspID: segment.FlightBase.DepartureStation,
				DepartureTzName:   service.GetTimeZoneByStationID(segment.FlightBase.DepartureStation).String(),
				DepartureTerminal: segment.FlightBase.DepartureTerminal,
				AirportToIata:     segment.FlightBase.ArrivalStationCode,
				AirportToRaspID:   segment.FlightBase.ArrivalStation,
				ArrivalTzName:     service.GetTimeZoneByStationID(segment.FlightBase.ArrivalStation).String(),
				ArrivalTerminal:   segment.FlightBase.ArrivalTerminal,
				Status:            flightstatus.GetFlightStatus(segment.FlightData, segment.FlightData, now),
				Title:             flightTitle,
				TransportModelID:  segment.FlightBase.AircraftTypeID,
			},
			Codeshares: uniqCodeshares(segment.Codeshares, flightTitle),
		}
		scheduledArrival := segment.ScheduledArrival()
		if !scheduledArrival.IsZero() {
			flight.ArrivalDay = scheduledArrival.Format(dtutil.IsoDate)
			flight.ArrivalTime = scheduledArrival.Format(dtutil.IsoTime)
			flight.ArrivalUTC = dtutil.FormatDateTimeISO(scheduledArrival.In(time.UTC))
		}
		scheduledDeparture := segment.ScheduledDeparture()
		if !scheduledDeparture.IsZero() {
			flight.DepartureDay = scheduledDeparture.Format(dtutil.IsoDate)
			flight.DepartureTime = scheduledDeparture.Format(dtutil.IsoTime)
			flight.DepartureUTC = dtutil.FormatDateTimeISO(scheduledDeparture.In(time.UTC))
		}

		if segment.FlightData.FlightPattern.IsDop {
			flight.Source = dto.FLIGHTBOARD.String()
		}

		flights = append(flights, flight)
	}
	response.Flights = flights
	return nil
}

func scheduledTerminalForDirection(data *flightdata.FlightData, direction dir.Direction) string {
	if direction == dir.ARRIVAL {
		return data.FlightBase.ArrivalTerminal
	}
	return data.FlightBase.DepartureTerminal
}

func scheduledDateForDirection(data *flightdata.FlightData, direction dir.Direction) string {
	if direction == dir.ARRIVAL {
		if data.ScheduledArrival().IsZero() {
			return ""
		}
		return data.ScheduledArrival().Format(time.RFC3339)
	}
	if data.ScheduledDeparture().IsZero() {
		return ""
	}
	return data.ScheduledDeparture().Format(time.RFC3339)
}

func uniqCodeshares(codeshares []*structs.FlightPattern, flightTitle string) []dto.TitledFlight {
	if len(codeshares) == 0 {
		return nil
	}

	uniq := make(map[dto.FlightID]*structs.FlightPattern)
	for _, codeshare := range codeshares {
		if flightTitle == codeshare.FlightTitle() {
			continue
		}
		uniq[dto.FlightID{AirlineID: codeshare.MarketingCarrier, Number: codeshare.MarketingFlightNumber}] = codeshare
	}

	slice := make([]dto.TitledFlight, 0, len(uniq))
	for codeshare, info := range uniq {
		slice = append(slice, dto.TitledFlight{
			FlightID: codeshare,
			Title:    info.FlightTitle(),
		})
	}
	return slice
}

func DateTimeFilter(
	direction dir.Direction, from time.Time, until time.Time) func(flight *FlightOnDateWithCodeshares) DateTimeFilterResult {
	switch direction {
	case dir.ARRIVAL:
		return func(flight *FlightOnDateWithCodeshares) DateTimeFilterResult {
			scheduledArrivalTime := flight.ScheduledArrival()
			arrivalTime := timeutil.Coalesce(flight.ActualArrival(), scheduledArrivalTime)
			if !arrivalTime.IsZero() && !arrivalTime.Before(from) && !arrivalTime.After(until) {
				if scheduledArrivalTime.Equal(arrivalTime) {
					return FitsByScheduledTime
				}
				return FitsByActualTime
			}
			if !scheduledArrivalTime.IsZero() && !scheduledArrivalTime.Before(from) && !scheduledArrivalTime.After(until) {
				return FitsByScheduledTime
			}
			return DoesNotFit
		}
	case dir.DEPARTURE:
		return func(flight *FlightOnDateWithCodeshares) DateTimeFilterResult {
			scheduledDepartureTime := flight.ScheduledDeparture()
			departureTime := timeutil.Coalesce(flight.ActualDeparture(), scheduledDepartureTime)
			if !departureTime.IsZero() && !departureTime.Before(from) && !departureTime.After(until) {
				if scheduledDepartureTime.Equal(departureTime) {
					return FitsByScheduledTime
				}
				return FitsByActualTime
			}
			if !scheduledDepartureTime.IsZero() && !scheduledDepartureTime.Before(from) && !scheduledDepartureTime.After(until) {
				return FitsByScheduledTime
			}
			return DoesNotFit
		}
	default:
		return func(flight *FlightOnDateWithCodeshares) DateTimeFilterResult {
			return DoesNotFit
		}
	}
}

func TerminalFilter(direction dir.Direction, terminal string) func(tz *FlightOnDateWithCodeshares) bool {

	if len(terminal) == 0 {
		return func(*FlightOnDateWithCodeshares) bool { return true }
	}

	// Assume that base is not nil, but status might be nil
	return func(fd *FlightOnDateWithCodeshares) bool {
		var baseTerminal, statusTerminal string
		if direction == dir.DEPARTURE {
			baseTerminal = fd.FlightBase.DepartureTerminal
			if fd.FlightStatus != nil {
				statusTerminal = fd.FlightStatus.DepartureTerminal
			}
		} else if direction == dir.ARRIVAL {
			baseTerminal = fd.FlightBase.ArrivalTerminal
			if fd.FlightStatus != nil {
				statusTerminal = fd.FlightStatus.ArrivalTerminal
			}
		} else {
			return false
		}
		if len(statusTerminal) > 0 {
			return statusTerminal == terminal
		}
		if len(baseTerminal) > 0 {
			return baseTerminal == terminal
		}
		return false
	}

}
