package flightsnetwork

import (
	"net/http"
	"strconv"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/services/storage/flights_network/format"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/storage"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/utils"
	"a.yandex-team.ru/travel/avia/shared_flights/api/pkg/structs"
)

type FlightsNetworkService interface {
	GetFlightsNetwork(carrierCodes []string) (response format.Response, err error)
}

func NewFlightsNetworkService(storage *storage.Storage) FlightsNetworkService {
	return &flightsNetworkServiceImpl{
		Storage: storage,
	}
}

type flightsNetworkServiceImpl struct {
	*storage.Storage
}

const (
	NoBannedFlights             = false
	NoCodeshares                = false
	NotSpecifiedNationalVersion = ""
)

func (service *flightsNetworkServiceImpl) GetFlightsNetwork(carrierCodes []string) (response format.Response, err error) {
	response = format.Response{}
	defer func() {
		if r := recover(); r != nil {
			response = format.Response{}
			err = &utils.ErrorWithHTTPCode{
				HTTPCode:     http.StatusInternalServerError,
				ErrorMessage: xerrors.Errorf("getFlightsNetwork panicked: %v", r).Error(),
			}
			return
		}
	}()

	carriers := make(map[int32]bool)
	for _, carrierCode := range carrierCodes {
		carrier, err := strconv.ParseInt(carrierCode, 10, 32)
		if err != nil {
			for _, c := range service.CarrierStorage().GetCarriersByCode(carrierCode) {
				carriers[c] = true
				response.Carriers = append(response.Carriers, c)
			}
		} else {
			carriers[int32(carrier)] = true
			response.Carriers = append(response.Carriers, int32(carrier))
		}
	}

	if len(response.Carriers) == 0 {
		err = &utils.ErrorWithHTTPCode{
			HTTPCode:     http.StatusNotFound,
			ErrorMessage: xerrors.Errorf("carrier not found: %v", carrierCodes).Error(),
		}
		return
	}

	flightPatternsMap := service.FlightStorage().GetFlightPatterns()
	var flightPatterns []*structs.FlightPattern
	for _, fp := range flightPatternsMap {
		if fp.IsCodeshare {
			continue
		}
		if _, ok := carriers[fp.MarketingCarrier]; !ok {
			continue
		}
		flightPatterns = append(flightPatterns, fp)
	}
	stations, err := service.FetchStations(flightPatterns, NoBannedFlights, NoCodeshares, NotSpecifiedNationalVersion)
	if err != nil {
		return response, err
	}
	for _, station := range stations {
		response.Stations = append(
			response.Stations,
			format.Station{
				AirportID:    station.Station.Id,
				SettlementID: station.Station.SettlementId,
				IataCode:     station.IataCode,
				SirenaCode:   station.SirenaCode,
				IcaoCode:     station.IcaoCode,
			},
		)
	}

	return response, nil
}
