package flightstatus

import (
	"fmt"
	"strings"
	"time"

	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/appconst"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/services/flightdata"
	"a.yandex-team.ru/travel/avia/shared_flights/api/pkg/structs"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/direction"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/dtutil"
)

const MinimumDelayTime = 15 * time.Minute

type FlightStatus struct {
	Status                appconst.FlightStatusText `json:"status"`
	DepartureDT           string                    `json:"departure"`
	ArrivalDT             string                    `json:"arrival"`
	DepartureStatus       string                    `json:"departureStatus"`
	ArrivalStatus         string                    `json:"arrivalStatus"`
	ArrivalGate           string                    `json:"arrivalGate"`
	ArrivalTerminal       string                    `json:"arrivalTerminal"`
	DepartureGate         string                    `json:"departureGate"`
	DepartureTerminal     string                    `json:"departureTerminal"`
	DepartureSource       string                    `json:"departureSource"`
	ArrivalSource         string                    `json:"arrivalSource"`
	CheckInDesks          string                    `json:"checkInDesks"`
	BaggageCarousels      string                    `json:"baggageCarousels"`
	CreatedAtUTC          string                    `json:"createdAtUtc"`
	UpdatedAtUTC          string                    `json:"updatedAtUtc"`
	DepartureUpdatedAtUTC string                    `json:"departureUpdatedAtUtc"`
	ArrivalUpdatedAtUTC   string                    `json:"arrivalUpdatedAtUtc"`
	Diverted              bool                      `json:"diverted"`
	DivertedAirportCode   string                    `json:"divertedAirportCode"`
	DivertedAirportID     int32                     `json:"divertedAirportID"`
}

func GetFlightStatus(departureLeg *flightdata.FlightData, arrivalLeg *flightdata.FlightData, now time.Time) FlightStatus {
	departureFS := departureLeg.FlightStatus
	if departureFS == nil {
		departureFS = &structs.FlightStatus{DepartureStatus: string(appconst.FlightStatusUnknown)}
	}

	arrivalFS := arrivalLeg.FlightStatus
	if arrivalFS == nil {
		arrivalFS = &structs.FlightStatus{ArrivalStatus: string(appconst.FlightStatusUnknown)}
	}

	departureUpdatedUTC := departureFS.DepartureUpdatedAtUtc
	arrivalUpdatedUTC := arrivalFS.ArrivalUpdatedAtUtc
	createdAtUTC := departureFS.CreatedAtUtc
	if arrivalFS.CreatedAtUtc > createdAtUTC {
		createdAtUTC = arrivalFS.CreatedAtUtc
	}
	updatedAtUTC := departureFS.UpdatedAtUtc
	if arrivalFS.UpdatedAtUtc > updatedAtUTC {
		updatedAtUTC = arrivalFS.UpdatedAtUtc
	}

	divertedAirportCode := ""
	divertedAirportID := int32(0)
	if departureFS.DepartureDiverted {
		divertedAirportCode = departureFS.DepartureDivertedAirportCode
		divertedAirportID = departureFS.DepartureDivertedAirportID
	}
	if arrivalFS.ArrivalDiverted {
		divertedAirportCode = arrivalFS.ArrivalDivertedAirportCode
		divertedAirportID = arrivalFS.ArrivalDivertedAirportID
	}

	departureStatus, arrivalStatus, globalStatus := getFlightStatusText(departureLeg, arrivalLeg, now)

	return FlightStatus{
		Status:                globalStatus,
		DepartureDT:           dtutil.FormatDateTimeISO(departureLeg.ActualDeparture()),
		ArrivalDT:             dtutil.FormatDateTimeISO(arrivalLeg.ActualArrival()),
		DepartureStatus:       string(departureStatus),
		ArrivalStatus:         string(arrivalStatus),
		ArrivalGate:           arrivalFS.ArrivalGate,
		ArrivalTerminal:       arrivalFS.ArrivalTerminal,
		DepartureGate:         departureFS.DepartureGate,
		DepartureTerminal:     departureFS.DepartureTerminal,
		DepartureSource:       fmt.Sprintf("%v", departureFS.DepartureSourceID),
		ArrivalSource:         fmt.Sprintf("%v", arrivalFS.ArrivalSourceID),
		CheckInDesks:          departureFS.CheckInDesks,
		BaggageCarousels:      arrivalFS.BaggageCarousels,
		CreatedAtUTC:          createdAtUTC,
		UpdatedAtUTC:          updatedAtUTC,
		DepartureUpdatedAtUTC: departureUpdatedUTC,
		ArrivalUpdatedAtUTC:   arrivalUpdatedUTC,
		Diverted:              departureFS.DepartureDiverted || arrivalFS.ArrivalDiverted,
		DivertedAirportCode:   divertedAirportCode,
		DivertedAirportID:     divertedAirportID,
	}
}

func getFlightStatusText(
	departureLeg *flightdata.FlightData, arrivalLeg *flightdata.FlightData, now time.Time,
) (departureStatus, arrivalStatus, globalStatus appconst.FlightStatusText) {
	var (
		scheduledDeparture, actualDeparture, scheduledArrival, actualArrival time.Time
	)

	departureStatus = getDepartureStatusText(departureLeg, now)
	arrivalStatus = getArrivalStatusText(arrivalLeg, now)
	if departureStatus == appconst.FlightStatusUnknown && arrivalStatus == appconst.FlightStatusUnknown {
		globalStatus = appconst.FlightStatusUnknown
		return
	}
	if (departureStatus == appconst.FlightStatusCancelled &&
		departureLeg.DepartureStatusIsFromTrustedSource()) ||
		(arrivalStatus == appconst.FlightStatusCancelled &&
			arrivalLeg.ArrivalStatusIsFromTrustedSource()) {
		departureStatus = appconst.FlightStatusCancelled
		arrivalStatus = appconst.FlightStatusCancelled
		globalStatus = appconst.FlightStatusCancelled
		return
	}
	if departureLeg != nil {
		scheduledDeparture = departureLeg.ScheduledDeparture()
		actualDeparture = departureLeg.ActualDeparture()
	}
	if arrivalLeg != nil {
		scheduledArrival = arrivalLeg.ScheduledArrival()
		actualArrival = arrivalLeg.ActualArrival()
	}
	globalStatus = calculateGlobalFlightStatus(
		departureStatus,
		arrivalStatus,
		now,
		scheduledDeparture,
		actualDeparture,
		scheduledArrival,
		actualArrival,
	)
	return
}

// TODO(mikhailche): добавить тест когда известно время по расписанию, но нет актуального времени и наоборот
func getHalfStatusText(dir direction.Direction, leg *flightdata.FlightData, now time.Time) appconst.FlightStatusText {
	if leg == nil || leg.FlightStatus == nil || !dir.IsValid() {
		return appconst.FlightStatusUnknown
	}
	var (
		rawStatusString string
		scheduledTime   time.Time
		actualTime      time.Time
	)
	switch dir {
	case direction.DEPARTURE:
		{
			rawStatusString = leg.FlightStatus.DepartureStatus
			scheduledTime = leg.ScheduledDeparture()
			actualTime = leg.ActualDeparture()
		}
	case direction.ARRIVAL:
		{
			rawStatusString = leg.FlightStatus.ArrivalStatus
			scheduledTime = leg.ScheduledArrival()
			actualTime = leg.ActualArrival()
		}
	}
	status := calculateHalfOfFlightStatus(scheduledTime, actualTime, now, IsCancelled(rawStatusString))
	if status == appconst.FlightStatusFinal {
		switch dir {
		case direction.DEPARTURE:
			return appconst.FlightStatusDeparted
		case direction.ARRIVAL:
			return appconst.FlightStatusArrived
		}
		return appconst.FlightStatusUnknown
	}
	return status
}

func getDepartureStatusText(leg *flightdata.FlightData, now time.Time) appconst.FlightStatusText {
	return getHalfStatusText(direction.DEPARTURE, leg, now)
}

func getArrivalStatusText(leg *flightdata.FlightData, now time.Time) appconst.FlightStatusText {
	return getHalfStatusText(direction.ARRIVAL, leg, now)
}

func IsCancelled(rawStatusText string) bool {
	if len(rawStatusText) == 0 {
		return false
	}
	rawStatusText = strings.ToLower(rawStatusText)
	return strings.Contains(rawStatusText, "отмен") || strings.Contains(rawStatusText, "cancel")
}

func calculateHalfOfFlightStatus(scheduled, actual, now time.Time, cancelled bool) appconst.FlightStatusText {
	if cancelled {
		return appconst.FlightStatusCancelled
	}
	if scheduled.IsZero() || actual.IsZero() || now.IsZero() {
		return appconst.FlightStatusUnknown
	}
	if now.Before(actual) {
		if actual.Before(scheduled.Add(-MinimumDelayTime)) {
			return appconst.FlightStatusEarly
		}
		if !actual.After(scheduled.Add(MinimumDelayTime)) {
			return appconst.FlightStatusOnTime
		}
		return appconst.FlightStatusDelayed
	}
	return appconst.FlightStatusFinal
}

func calculateGlobalFlightStatus(
	departureStatus, arrivalStatus appconst.FlightStatusText,
	now, scheduledDeparture, actualDeparture, scheduledArrival, actualArrival time.Time,
) appconst.FlightStatusText {
	if departureStatus == appconst.FlightStatusCancelled || arrivalStatus == appconst.FlightStatusCancelled {
		return appconst.FlightStatusCancelled
	}
	if now.IsZero() {
		return appconst.FlightStatusUnknown
	}
	if !(scheduledDeparture.IsZero() || actualDeparture.IsZero()) {
		if now.Before(actualDeparture) {
			if actualDeparture.After(scheduledDeparture.Add(MinimumDelayTime)) {
				return appconst.FlightStatusDelayed
			}
			if actualDeparture.After(scheduledDeparture.Add(-MinimumDelayTime)) {
				return appconst.FlightStatusOnTime
			}
			return appconst.FlightStatusEarly
		}
	}
	if !(scheduledArrival.IsZero() || actualArrival.IsZero()) {
		if now.Before(actualArrival) {
			if actualArrival.After(scheduledArrival.Add(MinimumDelayTime)) {
				return appconst.FlightStatusDelayed
			}
			if actualArrival.After(scheduledArrival.Add(-MinimumDelayTime)) {
				return appconst.FlightStatusOnTime
			}
			return appconst.FlightStatusOnTime // Should have been FlightStatusEarly, but that would break wizard
		} else {
			return appconst.FlightStatusArrived
		}
	}
	return appconst.FlightStatusUnknown

}
