package segment

import (
	"fmt"
	"strings"
	"time"

	"a.yandex-team.ru/travel/avia/shared_flights/api/pkg/structs"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/dtutil"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/logger"
	mathint "a.yandex-team.ru/travel/avia/shared_flights/lib/go/math"
)

type RoutePoint struct {
	AirportID         int64
	AirportCode       string
	ArrivalTime       string // "23:59:59" in local timezone
	ArrivalDayShift   int
	DepartureTime     string
	DepartureDayShift int
	DepartureTerminal string
	ArrivalTerminal   string
}

type Route []RoutePoint

func (r Route) Hash() string {
	builder := strings.Builder{}
	for _, point := range r {
		builder.WriteString(fmt.Sprint(point.AirportID))
		builder.WriteRune('/')
		builder.WriteString(fmt.Sprint(point.ArrivalDayShift))
		builder.WriteRune('/')
		builder.WriteString(point.ArrivalTime)
		builder.WriteRune('/')
		builder.WriteString(fmt.Sprint(point.DepartureDayShift))
		builder.WriteRune('/')
		builder.WriteString(point.DepartureTime)
		builder.WriteRune('/')
		builder.WriteString(point.DepartureTerminal)
		builder.WriteRune('/')
		builder.WriteString(point.ArrivalTerminal)
		builder.WriteRune(',')
	}
	return builder.String()
}

type RouteSchedule struct {
	Route            Route
	Mask             dtutil.DateMask
	TransportModelID int32
}

type Segment struct {
	FlightBase    structs.FlightBase
	FlightPattern structs.FlightPattern
}

type SegmentGroup []Segment

func GenerateSchedules(segments SegmentGroup) chan RouteSchedule {
	ch := make(chan RouteSchedule)
	go func() {
		defer close(ch)
		for _, patternGroup := range GenerateRoutes(segments) {
			ch <- patternGroup.schedule()
		}
	}()
	return ch
}

func GenerateRoutes(segments SegmentGroup) []SegmentGroup {
	defer func() {
		if r := recover(); r != nil {
			logger.Logger().Error("GenerateRoutes panicked")
			return
		}
	}()
	if len(segments) == 0 {
		return []SegmentGroup{}
	}
	patternsMapByLegSequence := make(map[int32]SegmentGroup)
	var minLegSeq, maxLegSeq int32
	minLegSeq = segments[0].FlightPattern.LegNumber
	maxLegSeq = segments[0].FlightPattern.LegNumber
	for _, segment := range segments {
		patternsMapByLegSequence[segment.FlightPattern.LegNumber] = append(patternsMapByLegSequence[segment.FlightPattern.LegNumber], segment)
		if minLegSeq > segment.FlightPattern.LegNumber {
			minLegSeq = segment.FlightPattern.LegNumber
		}
		if maxLegSeq < segment.FlightPattern.LegNumber {
			maxLegSeq = segment.FlightPattern.LegNumber
		}
	}

	var patternsListsOrderedByLegSequence []SegmentGroup
	for i := minLegSeq; i <= maxLegSeq; i++ {
		patternsListsOrderedByLegSequence = append(patternsListsOrderedByLegSequence, patternsMapByLegSequence[i])
	}

	return generateCombinations(patternsListsOrderedByLegSequence)
}

func generateCombinations(sequence []SegmentGroup) []SegmentGroup {
	result, _ := generateNextCombination(make([]SegmentGroup, 0), sequence, 0, nil)
	return result
}

func generateNextCombination(
	result []SegmentGroup, sequence []SegmentGroup, layer int, currentGroup SegmentGroup) ([]SegmentGroup, bool) {
	if layer >= len(sequence) {
		return result, false
	}
	segmentOnCurrentLayer := sequence[layer]
	allBad := true
	for _, segment := range segmentOnCurrentLayer {
		newGroup := append(currentGroup, segment)
		if !newGroup.valid() {
			continue
		}
		allBad = false
		currentIsGood := false
		result, currentIsGood = generateNextCombination(result, sequence, layer+1, newGroup)
		if !currentIsGood {
			newGroupCopy := make(SegmentGroup, len(newGroup))
			copy(newGroupCopy, newGroup)
			result = append(result, newGroupCopy)
		}
	}
	return result, !allBad
}

func (g SegmentGroup) route() Route {
	var route = make(Route, len(g)+1)

	for i, segment := range g {
		route[i].AirportID = segment.FlightBase.DepartureStation
		route[i+1].AirportID = segment.FlightBase.ArrivalStation

		route[i].AirportCode = segment.FlightBase.DepartureStationCode
		route[i+1].AirportCode = segment.FlightBase.ArrivalStationCode

		route[i].DepartureTime = dtutil.IntTime(segment.FlightBase.DepartureTimeScheduled).String()
		route[i+1].ArrivalTime = dtutil.IntTime(segment.FlightBase.ArrivalTimeScheduled).String()

		route[i].DepartureDayShift = int(segment.FlightPattern.FlightDayShift)
		route[i+1].ArrivalDayShift = int(segment.FlightPattern.FlightDayShift + segment.FlightPattern.ArrivalDayShift)

		route[i].DepartureTerminal = segment.FlightBase.DepartureTerminal
		route[i+1].ArrivalTerminal = segment.FlightBase.ArrivalTerminal
	}
	return route
}

func (g SegmentGroup) transportModelID() int32 {
	if len(g) == 0 {
		return 0
	}
	return int32(g[0].FlightBase.AircraftTypeID)
}

var startDateIndex = dtutil.DateCache.IndexOfStringDateP(dtutil.FormatDateIso(time.Now().AddDate(0, 0, -32)))

// For tests only
func SetGlobalStartDateIndex(dateIndex int) {
	startDateIndex = dateIndex
}

func (g SegmentGroup) mask() dtutil.DateMask {
	var masks []dtutil.DateMask

	for i := 1; i < len(g); i++ {
		overnightShift := mathint.Max(int(g[i-1].FlightPattern.ArrivalDayShift), int(g[i].FlightPattern.DepartureDayShift))
		g[i].FlightPattern.FlightDayShift = g[i-1].FlightPattern.FlightDayShift + int32(overnightShift)
	}

	for _, segment := range g {
		mask := dtutil.NewDateMask(startDateIndex, dtutil.MaxDaysInSchedule)
		mask.AddRange(
			dtutil.StringDate(segment.FlightPattern.OperatingFromDate),
			dtutil.StringDate(segment.FlightPattern.OperatingUntilDate),
			segment.FlightPattern.OperatingOnDays,
		)
		mask.ShiftDays(-int(segment.FlightPattern.FlightDayShift))

		masks = append(masks, mask)
	}
	if len(masks) == 0 {
		return dtutil.NewDateMask(startDateIndex, dtutil.MaxDaysInSchedule)
	}
	baseMask := dtutil.CloneDateMask(masks[0])
	for _, mask := range masks {
		baseMask.IntersectWithMask(mask)
		if baseMask.IsEmpty() {
			return baseMask
		}
	}
	return baseMask
}

func SumUpMasks(schedules []RouteSchedule) dtutil.DateMask {
	mask := dtutil.NewDateMask(startDateIndex, dtutil.MaxDaysInSchedule)

	for _, schedule := range schedules {
		mask.AddMask(schedule.Mask)
	}

	return mask
}

func (g SegmentGroup) schedule() RouteSchedule {
	return RouteSchedule{
		Route:            g.route(),
		Mask:             g.mask(),
		TransportModelID: g.transportModelID(),
	}
}

func (g SegmentGroup) valid() bool {
	for i := 1; i < len(g); i++ {
		if g[i].FlightBase.DepartureStation != g[i-1].FlightBase.ArrivalStation {
			return false
		}
	}
	return !g.mask().IsEmpty()
}
