package stationschedule

import (
	"net/http"
	"sort"
	"time"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/services/flightdata"
	dto "a.yandex-team.ru/travel/avia/shared_flights/api/internal/services/storage/DTO"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/services/storage/segment"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/services/storage/stationschedule/format"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/services/storage/threads"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/services/storage/timezone"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/storage"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/utils"
	"a.yandex-team.ru/travel/avia/shared_flights/api/pkg/structs"
	dir "a.yandex-team.ru/travel/avia/shared_flights/lib/go/direction"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/dtutil"
	"a.yandex-team.ru/travel/proto/shared_flights/snapshots"
)

type StationScheduleService interface {
	GetStationSchedule(
		station *snapshots.TStationWithCodes, direction dir.Direction, nationalVersion string, now time.Time,
	) (response format.Response, err error)
}

func NewStationScheduleService(storage *storage.Storage, timeZoneUtil timezone.TimeZoneUtil, threadRoute threads.ThreadRouteService) StationScheduleService {
	return &stationScheduleServiceImpl{
		Storage:            storage,
		TimeZoneUtil:       timeZoneUtil,
		ThreadRouteService: threadRoute,
	}
}

type stationScheduleServiceImpl struct {
	*storage.Storage
	timezone.TimeZoneUtil
	threads.ThreadRouteService
}

type tFlightBaseID = int32

type timeTerminal struct {
	Time             dtutil.IntTime
	Terminal         string
	Departure        int64
	Arrival          int64
	StartTime        dtutil.IntTime
	StartDayShift    int
	TransportModelID int32
}

type RouteMask struct {
	Route dto.Route
	Masks format.Mask
}

type FlightBaseGroup struct {
	structs.FlightBase
	FlightTitle string
	Map         map[timeTerminal][]RouteMask
}

func (service *stationScheduleServiceImpl) GetStationSchedule(
	station *snapshots.TStationWithCodes, direction dir.Direction, nationalVersion string, now time.Time,
) (response format.Response, err error) {
	if !direction.IsValid() {
		return response, xerrors.Errorf("invalid direction: %+v", direction)
	}

	flightPatterns, hasValue := service.FlightBoard.GetFlightPatterns(station, direction)
	if !hasValue {
		return response, &utils.ErrorWithHTTPCode{
			HTTPCode:     http.StatusNotFound,
			ErrorMessage: "no flight patterns has been found for the specified station",
		}
	}

	filteredFlightPatterns := make([]*structs.FlightPattern, 0, len(flightPatterns))
	for _, flightPattern := range flightPatterns {
		if flightPattern.IsDop {
			continue
		}
		filteredFlightPatterns = append(filteredFlightPatterns, flightPattern)
	}

	var segments []segment.Segment
	if segments, err = service.AttachFlightBases(filteredFlightPatterns, false, false, nationalVersion); err != nil {
		return response, xerrors.Errorf("cannot attach flight bases: %w", err)
	}

	flightBases := make([]*flightdata.FlightDataBase, 0, len(segments))
	for _, segment := range segments {
		temp := segment.FlightPattern
		flightBases = append(
			flightBases,
			flightdata.NewFlightDataBase(segment.FlightBase, &temp, nil, service.TimeZoneUtil),
		)
	}

	// Group by airline id, flight number
	// Group by arrival/departure time, arrival/departure terminal, route
	flightBaseGroups := make(map[dto.FlightID]FlightBaseGroup)
	if err = service.flightIDGroup(flightBases, flightBaseGroups, direction, now); err != nil {
		return response, xerrors.Errorf("cannot group flights: %w", err)
	}

	response = format.Response{
		Station:   station.Station.Id,
		Direction: direction.String(),
		Flights:   flightsFromGroup(flightBaseGroups),
	}

	return response, nil
}

func (service *stationScheduleServiceImpl) flightIDGroup(
	bases []*flightdata.FlightDataBase,
	groups map[dto.FlightID]FlightBaseGroup,
	direction dir.Direction,
	now time.Time) error {
	today := dtutil.IntDate(-1)
	if !now.IsZero() {
		if len(bases) > 0 && bases[0] != nil {
			var tz *time.Location
			if direction == dir.DEPARTURE {
				tz = service.TimeZoneUtil.GetTimeZoneByStationID(bases[0].FlightBase.DepartureStation)
			} else if direction == dir.ARRIVAL {
				tz = service.TimeZoneUtil.GetTimeZoneByStationID(bases[0].FlightBase.ArrivalStation)
			}
			if tz != nil {
				now = now.In(tz)
			}
		}
		today = dtutil.DateToInt(now)
	}
	for _, base := range bases {
		flightID := dto.FlightID{
			AirlineID: base.FlightBase.OperatingCarrier,
			Number:    base.FlightBase.OperatingFlightNumber,
		}
		var timeTermGroup FlightBaseGroup
		var ttExists bool
		if timeTermGroup, ttExists = groups[flightID]; !ttExists {
			timeTermGroup.FlightBase = base.FlightBase
			timeTermGroup.FlightTitle = base.FlightPattern.FlightTitle()
			timeTermGroup.Map = make(map[timeTerminal][]RouteMask)
		}
		route, ok, err := service.ThreadRoute(base)
		if err != nil {
			return xerrors.Errorf("cannot get thread route: %w", err)
		}
		if !ok {
			continue
		}
		operatingFrom := dtutil.StringDate(base.FlightPattern.OperatingFromDate)
		operatingUntil := dtutil.StringDate(base.FlightPattern.OperatingUntilDate)
		operatingOnDays := base.FlightPattern.OperatingOnDays
		if today >= 0 {
			dm := dtutil.NewDateMask(dtutil.DateCache.IndexOfIntDateP(today), dtutil.MaxDaysInSchedule)
			dm.AddRange(operatingFrom, operatingUntil, operatingOnDays)
			if dm.IsEmpty() {
				continue
			}
			operatingFrom, operatingUntil, operatingOnDays = dm.GetSingleRange()
		}

		timeTerm := timeTerminal{
			Time:             scheduledTimeForDirection(&base.FlightBase, direction),
			Terminal:         scheduledTerminalForDirection(&base.FlightBase, direction),
			Departure:        base.FlightBase.DepartureStation,
			Arrival:          base.FlightBase.ArrivalStation,
			StartTime:        dtutil.IntTime(route[0].DepartureTime),
			StartDayShift:    route[0].FlightDepartureShift,
			TransportModelID: int32(base.FlightBase.AircraftTypeID),
		}

		mask := format.Mask{
			From:  operatingFrom,
			Until: operatingUntil,
			On:    operatingOnDays,
		}
		if direction == dir.ARRIVAL {
			arrivalShift := int(base.FlightPattern.ArrivalDayShift)
			timeTerm.StartDayShift = timeTerm.StartDayShift - arrivalShift

			if arrivalShift != 0 {
				mask = format.Mask{
					From:  operatingFrom.AddDaysP(arrivalShift),
					Until: operatingUntil.AddDaysP(arrivalShift),
					On:    int32(dtutil.OperatingDays(operatingOnDays).ShiftDays(arrivalShift)),
				}
			}
		}

		routeMask := RouteMask{
			Route: route.StationsList(),
			Masks: mask,
		}

		timeTermGroup.Map[timeTerm] = append(timeTermGroup.Map[timeTerm], routeMask)
		groups[flightID] = timeTermGroup
	}
	return nil
}

func (service *stationScheduleServiceImpl) attachFlightBases(
	patterns []*structs.FlightPattern, nationalVersion string) (bases []*flightdata.FlightDataBase, err error) {
	for _, pattern := range patterns {
		if pattern.IsCodeshare {
			continue
		}
		var flightBase structs.FlightBase
		if flightBase, err = service.FlightStorage().GetFlightBase(pattern.FlightBaseID, pattern.IsDop); err != nil {
			return nil, xerrors.Errorf("cannot get flight base: %w", err)
		}
		bannedFrom, bannedUntil, isBanned := service.Storage.BlacklistRuleStorage().GetBannedDates(flightBase, pattern, nationalVersion)

		if isBanned {
			if bannedFrom == "" {
				// flight is banned regardless of dates
				continue
			}
			intervals := dtutil.Intersect(
				dtutil.NewStringDateInterval(pattern.OperatingFromDate, pattern.OperatingUntilDate),
				dtutil.NewStringDateInterval(bannedFrom, bannedUntil),
			)
			for _, interval := range intervals {
				startDate := interval.From.ToTime(time.UTC)
				endDate := interval.To.ToTime(time.UTC)
				intervalHasOperatingDay := false
				for flightDate := startDate; flightDate.Before(endDate) || flightDate.Equal(endDate); flightDate = flightDate.AddDate(0, 0, 1) {
					if dtutil.OperatesOn(pattern.OperatingOnDays, flightDate.Weekday()) {
						intervalHasOperatingDay = true
						break
					}
				}
				if intervalHasOperatingDay {
					intervalFlightPattern := *pattern
					intervalFlightPattern.OperatingFromDate = string(interval.From)
					intervalFlightPattern.OperatingUntilDate = string(interval.To)
					bases = append(bases, flightdata.NewFlightDataBase(flightBase, &intervalFlightPattern, nil, service.TimeZoneUtil))
					continue
				}
			}
		} else {
			bases = append(bases, flightdata.NewFlightDataBase(flightBase, pattern, nil, service.TimeZoneUtil))
		}
	}
	return
}

func flightsFromGroup(groups map[dto.FlightID]FlightBaseGroup) (flights []format.Flight) {
	flights = make([]format.Flight, 0, len(groups))
	for _, group := range groups {
		flights = append(flights,
			format.Flight{
				TitledFlight: dto.TitledFlight{
					FlightID: dto.FlightID{
						AirlineID: group.OperatingCarrier,
						Number:    group.OperatingFlightNumber,
					},
					Title: group.FlightTitle,
				},
				Schedules: schedules(group.Map),
			})
	}
	return
}

func schedules(group map[timeTerminal][]RouteMask) (schedules []format.Schedule) {
	var lastRoute dto.Route
	var schedule format.Schedule
	var lastTimeTerminal timeTerminal

	keysSlice := timeTerminalKeysSlice(group)
	sort.Stable(keysSlice)
	for _, timeTerm := range keysSlice {
		routemasks := group[timeTerm]
		for _, routemask := range routemasks {
			if lastRoute == nil || !lastRoute.Equals(routemask.Route) || lastTimeTerminal != timeTerm {
				if lastRoute != nil {
					schedules = append(schedules, schedule)
				}
				lastRoute = routemask.Route
				lastTimeTerminal = timeTerm
				schedule = format.Schedule{
					Time:             timeTerm.Time.StringTime(),
					Terminal:         timeTerm.Terminal,
					StartTime:        timeTerm.StartTime.StringTime(),
					StartDayShift:    int32(timeTerm.StartDayShift),
					TransportModelID: timeTerm.TransportModelID,
					Route:            lastRoute,
					Masks:            nil,
				}
			}
			schedule.Masks = append(schedule.Masks, routemask.Masks)
		}
	}
	if schedule.Masks != nil {
		schedules = append(schedules, schedule)
	}
	return schedules
}

type timeTerminalSlice []timeTerminal

func (t timeTerminalSlice) Len() int {
	return len(t)
}

func (t timeTerminalSlice) Less(i, j int) bool {
	if t[i].Time < t[j].Time {
		return true
	}
	if t[i].Time == t[j].Time {
		if t[i].Terminal < t[j].Terminal {
			return true
		}
		if t[i].Terminal == t[j].Terminal {
			if t[i].Departure < t[j].Departure {
				return true
			}
			if t[i].Departure == t[j].Departure {
				if t[i].Arrival < t[j].Arrival {
					return true
				}
			}
		}
	}
	return false
}

func (t timeTerminalSlice) Swap(i, j int) {
	t[i], t[j] = t[j], t[i]
}

func timeTerminalKeysSlice(group map[timeTerminal][]RouteMask) (tt timeTerminalSlice) {
	for key := range group {
		tt = append(tt, key)
	}
	return tt
}

func scheduledTerminalForDirection(base *structs.FlightBase, direction dir.Direction) string {
	if direction == dir.ARRIVAL {
		return base.ArrivalTerminal
	}
	return base.DepartureTerminal
}

func scheduledTimeForDirection(base *structs.FlightBase, direction dir.Direction) dtutil.IntTime {
	if direction == dir.ARRIVAL {
		return dtutil.IntTime(base.ArrivalTimeScheduled)
	}
	return dtutil.IntTime(base.DepartureTimeScheduled)
}
