package threads

import (
	"encoding/json"
	"math"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/services/flightdata"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/services/storage/timezone"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/storage"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/dtutil"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/logger"
	mathint "a.yandex-team.ru/travel/avia/shared_flights/lib/go/math"
)

type RouteElem struct {
	StationID            int64
	DepartureTime        int32
	FlightDepartureShift int // Shift in days between the initial take-off and the departure of the requested segment
}

type RouteElems []RouteElem

func (r RouteElems) StationsList() []int64 {
	result := make([]int64, len(r))
	for index, elem := range r {
		result[index] = elem.StationID
	}
	return result
}

type ThreadRouteService interface {
	// !hasData means "skip this itinerary", err != nil means "something terrible happened, return httpCode=500 right now"
	ThreadRoute(current *flightdata.FlightDataBase) (route RouteElems, hasData bool, err error)
}

func NewThreadRouteService(storage *storage.Storage, tzutil timezone.TimeZoneUtil) ThreadRouteService {
	return &threadRouteServiceImpl{Storage: storage, TimeZoneUtil: tzutil}
}

type threadRouteServiceImpl struct {
	*storage.Storage
	timezone.TimeZoneUtil
}

func (service *threadRouteServiceImpl) ThreadRoute(
	current *flightdata.FlightDataBase,
) (route RouteElems, hasData bool, err error) {
	/*
		Для поиска подходящего списка сегментов берём все flight pattern которые попадают в окно +-10 дней
		от даты искомого сегмента. Раскладываем их по номеру lega. Пока что не обращаем внимания на день недели.
		Получаем укороченный список, с которым можно работать тяжелыми функциями работы с датами.

		Окно в 10 дней можно обосновать текущим известным маскимальным числом сегментов и предположением что
		не может один рейс-маршрутка лететь больше 10 дней.

		Для составления нитки от текущего сегмента, чей leg number мы знаем,
		начинаем искать совпадаюшие предыдущие и следующие сегменты

		Для поиска предыдущих сегментов:
		убеждаемся что мы нашли flight pattern который летает в правильную дату, а именно
		(дата прилёта текущего сегмента - разница в датах предыдущего сегмента) должна попадать в диапазон
		оперирующих дат для предыдущего сегмента (с окном в -1 день на случай если рейс улетает завтра, а прилетает сегодня).

		Если рейс подходит под условия - забираем его

		Для поиска следующих сегментов:
		убеждаемся, что мы нашли flight pattern который летает в правильную дату, а именно
		(дата прилёта текущего сегмента + разница в датах для текущего сегмента) должна попадать в диапазон
		оперирующих дат для предыдущего сегмента (с окном в +1 день на случай если рейс прилетает сегодня, а улетает завтра)
	*/
	const EstimateMaxLegs = 10
	const EstimateFlightDateWindow = 10

	carrier := current.FlightBase.OperatingCarrier
	flightNumber := current.FlightBase.OperatingFlightNumber
	flights, hasValue := service.FlightStorage().GetFlights(carrier, flightNumber)
	if !hasValue || len(flights) == 0 {
		return nil, false, xerrors.Errorf("No flights found. %v %v", carrier, flightNumber)
	}

	route = make(RouteElems, EstimateMaxLegs)

	variantsByLeg := make([][]*flightdata.FlightDataBase, EstimateMaxLegs)
	maxLeg := int32(-1)
	minLeg := int32(math.MaxInt32)
	for _, legFlights := range flights {
		for _, segment := range legFlights {
			if segment.IsCodeshare {
				continue
			}
			var currentSegmentOperateFromDateIndex, currentSegmentOperateUntilDateIndex, flightDepartureDateFromIndex, flightDepartureDateUntilIndex int
			var ok bool
			if currentSegmentOperateFromDateIndex, ok = dtutil.DateCache.IndexOfStringDate(
				dtutil.StringDate(segment.OperatingFromDate)); !ok {
				return nil, false, xerrors.Errorf(
					"cannot find date %v in date cache for segment %v",
					segment.OperatingFromDate,
					segment,
				)
			}
			if currentSegmentOperateUntilDateIndex, ok = dtutil.DateCache.IndexOfStringDate(
				dtutil.StringDate(segment.OperatingUntilDate)); !ok {
				return nil, false, xerrors.Errorf(
					"cannot find date %v in date cache for segment %v",
					segment.OperatingUntilDate,
					segment,
				)
			}
			if flightDepartureDateFromIndex, ok = dtutil.DateCache.IndexOfStringDate(dtutil.StringDate(current.FlightPattern.OperatingFromDate)); !ok {
				return nil, false, xerrors.Errorf(
					"cannot find date %v in date cache for flight pattern %v",
					current.FlightPattern.OperatingFromDate,
					current.FlightPattern,
				)
			}
			if flightDepartureDateUntilIndex, ok = dtutil.DateCache.IndexOfStringDate(dtutil.StringDate(current.FlightPattern.OperatingUntilDate)); !ok {
				return nil, false, xerrors.Errorf(
					"cannot find date %v in date cache for flight pattern %v",
					current.FlightPattern.OperatingUntilDate,
					current.FlightPattern,
				)
			}
			if flightDepartureDateUntilIndex+EstimateFlightDateWindow < currentSegmentOperateFromDateIndex {
				continue
			}
			if flightDepartureDateFromIndex-EstimateFlightDateWindow > currentSegmentOperateUntilDateIndex {
				continue
			}

			fb, err := service.FlightStorage().GetFlightBase(segment.FlightBaseID, segment.IsDop)
			if err != nil {
				return nil, false, xerrors.Errorf(
					"unexpected error while looking for a flight base %v; flight %v %v; error: %v",
					segment.FlightBaseID,
					carrier,
					flightNumber,
					err,
				)
			}
			fd := flightdata.NewFlightDataBase(fb, segment, nil, service)
			if maxLeg < fb.LegNumber {
				maxLeg = fb.LegNumber
				if int(maxLeg) > cap(variantsByLeg) {
					newSlice := make([][]*flightdata.FlightDataBase, maxLeg*2)
					copy(newSlice, variantsByLeg)
					variantsByLeg = newSlice
				}
			}
			if minLeg > fb.LegNumber {
				minLeg = fb.LegNumber
			}
			variantsByLeg[fb.LegNumber] = append(variantsByLeg[fb.LegNumber], fd)
		}
	}

	if minLeg > maxLeg {
		return nil, false, xerrors.Errorf("could not extract route from flights. %v %v %#v", carrier, flightNumber, flights)
	}

	/*
		надо положить в route 3 показателя вместо 1:
		- DepStation
		- StartTime
		- DepartureFlightShift
		DepStation считается как сейчас
		StartTime считается как current.FlightBase.ScheduledDepartureTime
		DepartureFlightShift считается как сумма -arrivalShift по route [0..current.FlightBase.LegNumber-1]

		в schedule это можно прямо выводить
		в табло надо вычитать руками от даты вылета (если табло по дате прилёта, надо найти дату вылета,
		  вычитая ArrivalDayShift из даты прилёта заданной)
	*/
	route[current.FlightBase.LegNumber-1] = RouteElem{
		StationID:     current.FlightBase.DepartureStation,
		DepartureTime: current.FlightBase.DepartureTimeScheduled,
	}

	err = backwardSegmentSearch(route, int(current.FlightBase.LegNumber-2), current, minLeg, variantsByLeg)
	if err != nil {
		// flight's data is incorrect but it's more preferable to skip a flight than lose the whole flight board
		return nil, false, nil
	}

	route[current.FlightBase.LegNumber] = RouteElem{
		StationID: current.FlightBase.ArrivalStation,
	}
	maxLeg, err = forwardSegmentSearch(route, int(current.FlightBase.LegNumber+1), current, maxLeg, variantsByLeg)
	if err != nil {
		// dopFlights may not match the scheduled flights, nevertheless they should be displayed on a flight-board
		if current.FlightPattern.IsDop {
			currentLeg := int(current.FlightBase.LegNumber)
			return route[currentLeg-1 : currentLeg], true, nil
		}
		// flight's data is incorrect but it's more preferable to skip a flight than lose the whole flight board
		return nil, false, nil
	}

	return route[minLeg-1 : maxLeg+1], true, nil
}

func backwardSegmentSearch(route []RouteElem, routeIndex int, referenceSegment *flightdata.FlightDataBase, minLeg int32, variantsByLeg [][]*flightdata.FlightDataBase) error {
	nextSegment := referenceSegment
	nextSegmentDepartureDayFromIndex, ok := dtutil.DateCache.IndexOfStringDate(dtutil.StringDate(nextSegment.FlightPattern.OperatingFromDate))
	if !ok {
		return xerrors.Errorf("cannot get date index for current segment %v", nextSegment)
	}
	nextSegmentDepartureDayUntilIndex, ok := dtutil.DateCache.IndexOfStringDate(dtutil.StringDate(nextSegment.FlightPattern.OperatingUntilDate))
	if !ok {
		return xerrors.Errorf("cannot get date index for current segment %v", nextSegment)
	}

	expectedArrivalStation := referenceSegment.FlightBase.DepartureStation
	for i := referenceSegment.FlightBase.LegNumber - 1; i >= minLeg; i-- {
		found := false
		for _, arrivalShiftCorrection := range []int{0, 1, -1} {

			for _, currentSegment := range variantsByLeg[i] {
				if currentSegment.FlightBase.ArrivalStation != expectedArrivalStation {
					continue
				}
				// calculate date difference for previous arrival segment
				var currentSegmentOperateFromDateIndex, currentSegmentOperateUntilDateIndex,
					flightDepartureDateFromIndex, flightDepartureDateUntilIndex int
				if currentSegmentOperateFromDateIndex, ok = dtutil.DateCache.IndexOfStringDate(
					dtutil.StringDate(currentSegment.FlightPattern.OperatingFromDate)); !ok {
					return xerrors.Errorf(
						"cannot find date %v in date cache for segment %v",
						currentSegment.FlightPattern.OperatingFromDate,
						currentSegment,
					)
				}
				if currentSegmentOperateUntilDateIndex, ok = dtutil.DateCache.IndexOfStringDate(
					dtutil.StringDate(currentSegment.FlightPattern.OperatingUntilDate)); !ok {
					return xerrors.Errorf(
						"cannot find date %v in date cache for segment %v",
						currentSegment.FlightPattern.OperatingUntilDate,
						currentSegment,
					)
				}

				overnightShift := mathint.Max(currentSegment.ArrivalDayDifference(), nextSegment.DepartureDayDifference())
				arrivalDiff := overnightShift + arrivalShiftCorrection
				flightDepartureDateFromIndex = nextSegmentDepartureDayFromIndex - arrivalDiff
				flightDepartureDateUntilIndex = nextSegmentDepartureDayUntilIndex - arrivalDiff
				if flightDepartureDateUntilIndex < currentSegmentOperateFromDateIndex {
					continue
				}
				if flightDepartureDateFromIndex > currentSegmentOperateUntilDateIndex {
					continue
				}

				var currentOperating, nextOperatingShifted dtutil.OperatingDays
				currentOperating = dtutil.OperatingDays(currentSegment.FlightPattern.OperatingOnDays)
				nextOperatingShifted = dtutil.OperatingDays(nextSegment.FlightPattern.OperatingOnDays).
					ShiftDays(-arrivalDiff)

				if currentOperating.Intersect(nextOperatingShifted) == 0 {
					continue
				}
				if routeIndex >= len(route) {
					logger.Logger().L.DPanic("Going to fail")
				}
				route[routeIndex] = RouteElem{
					StationID:     currentSegment.FlightBase.DepartureStation,
					DepartureTime: currentSegment.FlightBase.DepartureTimeScheduled,
				}
				if routeIndex+1 < len(route) {
					route[routeIndex].FlightDepartureShift = route[routeIndex+1].FlightDepartureShift - arrivalDiff
				}

				routeIndex--
				expectedArrivalStation = currentSegment.FlightBase.DepartureStation

				nextSegment = currentSegment
				nextSegmentDepartureDayFromIndex = currentSegmentOperateFromDateIndex
				nextSegmentDepartureDayUntilIndex = currentSegmentOperateUntilDateIndex
				found = true
				break
			}
			if found {
				break
			}
		}
		if !found {
			return xerrors.Errorf("cannot find appropriate segment %+v", variantsByLeg)
		}

	}
	return nil
}

/* TODO(mikhailche): добавить тест, в котором проверить ситуацию, когда сегодня 3 сегмента, а завтра 2
(на первом этапе должны попасть все три, а потом третий должен отсеяться)
*/
func forwardSegmentSearch(
	route []RouteElem, routeIndex int, current *flightdata.FlightDataBase, maxLeg int32, variantsByLeg [][]*flightdata.FlightDataBase,
) (newMaxLeg int32, err error) {
	//TODO (mikhailche): make it same as backwards search
	previouseSegmentDepartureDayIndex, ok := dtutil.DateCache.IndexOfStringDate(dtutil.StringDate(current.FlightPattern.OperatingFromDate))
	if !ok {
		return maxLeg, xerrors.Errorf("cannot get date index for current segment %v", current)
	}

	previouseSegmentArrivalDayShift := current.ArrivalDayDifference()

	expectedDepartureStation := current.FlightBase.ArrivalStation
	for i := current.FlightBase.LegNumber + 1; i <= maxLeg; i++ {
		found := false
		for _, arrivalShiftCorrection := range []int{0, 1, -1} {
			for _, currentSegment := range variantsByLeg[i] {
				if currentSegment.FlightBase.DepartureStation != expectedDepartureStation {
					continue
				}
				// calculate date difference for previous arrival segment
				var currentSegmentOperateFromDateIndex, currentSegmentOperateUntilDateIndex, flightDepartureDateIndex int
				if currentSegmentOperateFromDateIndex, ok = dtutil.DateCache.IndexOfStringDate(
					dtutil.StringDate(currentSegment.FlightPattern.OperatingFromDate)); !ok {
					return i - 1, xerrors.Errorf(
						"cannot find date %v in date cache for segment %v",
						currentSegment.FlightPattern.OperatingFromDate,
						currentSegment,
					)
				}
				if currentSegmentOperateUntilDateIndex, ok = dtutil.DateCache.IndexOfStringDate(
					dtutil.StringDate(currentSegment.FlightPattern.OperatingUntilDate)); !ok {
					return i - 1, xerrors.Errorf(
						"cannot find date %v in date cache for segment %v",
						currentSegment.FlightPattern.OperatingUntilDate,
						currentSegment,
					)
				}
				flightDepartureDateIndex = previouseSegmentDepartureDayIndex + previouseSegmentArrivalDayShift + arrivalShiftCorrection
				if flightDepartureDateIndex < currentSegmentOperateFromDateIndex {
					continue
				}
				if flightDepartureDateIndex > currentSegmentOperateUntilDateIndex {
					continue
				}
				if !dtutil.OperatesOn(currentSegment.FlightPattern.OperatingOnDays, dtutil.DateCache.WeekDay(flightDepartureDateIndex)) {
					continue
				}
				// For the forward search we don't fill in the departure time and the shift from the start of the flight -
				// those are not printed in the response anyway
				route[routeIndex].StationID = currentSegment.FlightBase.ArrivalStation
				routeIndex++
				expectedDepartureStation = currentSegment.FlightBase.ArrivalStation

				previouseSegmentDepartureDayIndex = flightDepartureDateIndex
				previouseSegmentArrivalDayShift = currentSegment.ArrivalDayDifference(dtutil.DateCache.Date(flightDepartureDateIndex))
				found = true
				break
			}

			if found {
				break
			}
		}
		if !found {
			return i - 1, nil
		}
	}
	return maxLeg, nil
}

func jsonify(v interface{}) string {
	bytes, _ := json.MarshalIndent(v, "  ", "  ")
	return string(bytes)
}
