package carrier

import (
	"strings"

	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/strutil"
)

type CarrierCodes struct {
	Iata, Sirena, Icao, IcaoRu string
}

type CarrierStorage struct {
	carrierCodes map[int32]CarrierCodes
	iataByID     map[int32]string
	sirenaByID   map[int32]string
	codeByID     map[int32]string // iata, or sirena, or icao code - first one which is not empty
	idsByCode    map[string][]int32
}

func (c *CarrierStorage) CarrierCodes() map[int32]CarrierCodes {
	return c.carrierCodes
}

func (c *CarrierStorage) PutCarrier(id int32, iata, sirena, icao, icaoRu string) {
	iata = strings.ToUpper(iata)
	sirena = strings.ToUpper(sirena)
	icao = strings.ToUpper(icao)
	icaoRu = strings.ToUpper(icaoRu)

	c.carrierCodes[id] = CarrierCodes{iata, sirena, icao, icaoRu}

	if iata != "" {
		c.iataByID[id] = iata
		c.fillIDsByCode(c.idsByCode, iata, id)
	}
	if sirena != "" {
		c.sirenaByID[id] = sirena
		c.fillIDsByCode(c.idsByCode, sirena, id)

	}
	if icao != "" {
		c.fillIDsByCode(c.idsByCode, icao, id)
	}

	if icaoRu != "" {
		c.fillIDsByCode(c.idsByCode, icaoRu, id)
	}

	code := strutil.Coalesce(iata, sirena, icao, icaoRu)
	if code != "" {
		c.codeByID[id] = code
	}
}

func (c *CarrierStorage) fillIDsByCode(m map[string][]int32, code string, id int32) {
	values, exists := m[code]
	if !exists {
		m[code] = []int32{id}
		return
	}
	for _, value := range values {
		if value == id {
			return
		}
	}
	values = append(values, id)
	m[code] = values
}

func (c *CarrierStorage) GetCarrierByID(id int32) string {
	carrier, hasCarrier := c.iataByID[id]
	if !hasCarrier {
		return ""
	}
	return carrier
}

func (c *CarrierStorage) HasCarrier(id int32) bool {
	_, hasCarrier := c.iataByID[id]
	return hasCarrier
}

func (c *CarrierStorage) GetSirenaByID(id int32) string {
	carrier, hasCarrier := c.sirenaByID[id]
	if !hasCarrier {
		return ""
	}
	return carrier
}

func (c *CarrierStorage) GetCarrierCodeByID(id int32) string {
	carrier, hasCarrier := c.codeByID[id]
	if !hasCarrier {
		return ""
	}
	return carrier
}

func (c *CarrierStorage) GetCarriersByCode(code string) []int32 {
	carriers := c.idsByCode[strings.ToUpper(code)]
	return carriers
}

func NewCarrierStorage() *CarrierStorage {
	instance := CarrierStorage{
		carrierCodes: make(map[int32]CarrierCodes),
		iataByID:     make(map[int32]string),
		sirenaByID:   make(map[int32]string),
		codeByID:     make(map[int32]string),
		idsByCode:    make(map[string][]int32),
	}
	return &instance
}
