package flight

import (
	"strings"
	"time"

	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/storage/station"
	"a.yandex-team.ru/travel/avia/shared_flights/api/pkg/structs"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/dtutil"
	"a.yandex-team.ru/travel/proto/shared_flights/snapshots"
	"a.yandex-team.ru/travel/proto/shared_flights/ssim"
)

type BlacklistRuleStorage interface {
	AddRule(blacklistRule *snapshots.TBlacklistRule) bool
	// Returns false if flight is not banned, true if it's possibly banned.
	IsPossiblyBanned(flightBase structs.FlightBase, flightPattern *structs.FlightPattern) bool
	// Determines whether the flight is banned for the given day and national version
	IsBanned(flightBase structs.FlightBase, flightPattern *structs.FlightPattern, departureDate dtutil.StringDate, nationalVersion string) bool
	// Returns the range of the dates when the flight is banned, if any
	GetBannedDates(flightBase structs.FlightBase, flightPattern *structs.FlightPattern, nationalVersion string) (bannedFrom, bannedUntil string, isBanned bool)
	// To simplify testing
	RuleApplies(flightBase structs.FlightBase, flightPattern *structs.FlightPattern, rule *snapshots.TBlacklistRule) bool
	IsRuleActive(nowStr string, rule *snapshots.TBlacklistRule) bool
}

type blacklistRuleStorageImpl struct {
	Rules []*snapshots.TBlacklistRule
	station.StationStorage
}

func NewBlacklistRuleStorage(stationStorage station.StationStorage) BlacklistRuleStorage {
	return &blacklistRuleStorageImpl{
		Rules:          make([]*snapshots.TBlacklistRule, 0),
		StationStorage: stationStorage,
	}
}

func (s *blacklistRuleStorageImpl) AddRule(blacklistRule *snapshots.TBlacklistRule) bool {
	if blacklistRule == nil || blacklistRule.ForceMode == Disabled.String() {
		return false
	}
	now := time.Now().UTC()
	if blacklistRule.ForceMode == AsScheduled.String() &&
		len(blacklistRule.ActiveUntil) > 0 &&
		dtutil.FormatDateTimeISO(now) >= blacklistRule.ActiveUntil {
		return false
	}
	s.Rules = append(s.Rules, blacklistRule)
	return true
}

// Returns false when flight is not banned, true when it's possibly banned
func (s *blacklistRuleStorageImpl) IsPossiblyBanned(flightBase structs.FlightBase, flightPattern *structs.FlightPattern) bool {
	nowStr := dtutil.FormatDateTimeISO(time.Now().UTC())
	for _, rule := range s.Rules {
		if !s.IsRuleActive(nowStr, rule) {
			continue
		}
		if s.RuleApplies(flightBase, flightPattern, rule) {
			return true
		}
	}
	return false
}

func (s *blacklistRuleStorageImpl) RuleApplies(
	flightBase structs.FlightBase, flightPattern *structs.FlightPattern, rule *snapshots.TBlacklistRule) bool {
	// test the fastest conditions first
	if rule.StationFromId != 0 && int64(rule.StationFromId) != flightBase.DepartureStation {
		return false
	}
	if rule.StationToId != 0 && int64(rule.StationToId) != flightBase.ArrivalStation {
		return false
	}
	if rule.Source != ssim.EFlightBaseSource_TYPE_UNKNOWN && structs.GetSource(rule.Source) != flightBase.Source {
		return false
	}
	if rule.MarketingCarrierId != 0 || len(rule.MarketingFlightNumber) != 0 {
		if len(rule.MarketingFlightNumber) == 0 {
			// carrier only
			if rule.MarketingCarrierId != flightPattern.MarketingCarrier &&
				rule.MarketingCarrierId != flightBase.OperatingCarrier {
				return false
			}
		} else if rule.MarketingCarrierId == 0 {
			// flight number only (do we need this case?)
			if rule.MarketingFlightNumber != flightPattern.MarketingFlightNumber {
				return false
			}
		} else {
			// both carrier and flight number should match
			if (rule.MarketingCarrierId != flightPattern.MarketingCarrier ||
				rule.MarketingFlightNumber != flightPattern.MarketingFlightNumber) &&
				(rule.MarketingCarrierId != flightBase.OperatingCarrier ||
					rule.MarketingFlightNumber != flightBase.OperatingFlightNumber) {
				return false
			}
		}
	}
	if len(rule.FlightDateSince) != 0 && rule.FlightDateSince > flightPattern.OperatingUntilDate {
		return false
	}
	if len(rule.FlightDateUntil) != 0 && rule.FlightDateUntil < flightPattern.OperatingFromDate {
		return false
	}
	// settlement и country мы можем проверить здесь
	// но дату вылета и нацверсию придётся проверять уже в точке вызова
	if rule.StationFromSettlement != 0 || rule.StationFromCountry != 0 {
		stationFrom, ok := s.StationStorage.ByID(flightBase.DepartureStation)
		if !ok {
			// unknown station
			return false
		}
		if rule.StationFromSettlement != 0 && rule.StationFromSettlement != stationFrom.Station.SettlementId {
			return false
		}
		if rule.StationFromCountry != 0 && uint32(rule.StationFromCountry) != stationFrom.Station.CountryId {
			return false
		}
	}
	if rule.StationToSettlement != 0 || rule.StationToCountry != 0 {
		stationTo, ok := s.StationStorage.ByID(flightBase.ArrivalStation)
		if !ok {
			// unknown station
			return false
		}
		if rule.StationToSettlement != 0 && rule.StationToSettlement != stationTo.Station.SettlementId {
			return false
		}
		if rule.StationToCountry != 0 && uint32(rule.StationToCountry) != stationTo.Station.CountryId {
			return false
		}
	}
	return true
}

func (s *blacklistRuleStorageImpl) IsRuleActive(nowStr string, rule *snapshots.TBlacklistRule) bool {
	if rule.ForceMode == AsScheduled.String() {
		if len(rule.ActiveUntil) > 0 && nowStr >= rule.ActiveUntil {
			return false
		}
		if len(rule.ActiveSince) > 0 && nowStr <= rule.ActiveSince {
			return false
		}
	}
	// we don't check for blacklistRule.ForceMode == Disabled because such rules should be filtered out while loading the rules list
	return true
}

func (s *blacklistRuleStorageImpl) IsBanned(
	flightBase structs.FlightBase, flightPattern *structs.FlightPattern, departureDate dtutil.StringDate, nationalVersion string) bool {
	nowStr := dtutil.FormatDateTimeISO(time.Now().UTC())

	for _, rule := range s.Rules {
		if !s.IsRuleActive(nowStr, rule) {
			continue
		}
		if !s.RuleApplies(flightBase, flightPattern, rule) {
			continue
		}
		bannedByDepatureDate := false
		if len(rule.FlightDateSince) != 0 || len(rule.FlightDateUntil) != 0 {
			if len(rule.FlightDateSince) != 0 && rule.FlightDateSince > string(departureDate) {
				continue
			}
			if len(rule.FlightDateUntil) != 0 && rule.FlightDateUntil < string(departureDate) {
				continue
			}
			bannedByDepatureDate = true
		}
		bannedByNationalVersion := false
		if len(rule.NationalVersion) != 0 {
			if !strings.EqualFold(rule.NationalVersion, nationalVersion) {
				continue
			}
			bannedByNationalVersion = true
		}
		if bannedByDepatureDate || bannedByNationalVersion {
			return true
		}
		if len(rule.FlightDateSince) == 0 && len(rule.FlightDateUntil) == 0 && len(rule.NationalVersion) == 0 {
			return true
		}
	}
	return false
}

func (s *blacklistRuleStorageImpl) GetBannedDates(
	flightBase structs.FlightBase, flightPattern *structs.FlightPattern, nationalVersion string) (bannedFrom, bannedUntil string, isBanned bool) {
	nowStr := dtutil.FormatDateTimeISO(time.Now().UTC())

	isBanned = true
	for _, rule := range s.Rules {
		if !s.IsRuleActive(nowStr, rule) {
			continue
		}
		if !s.RuleApplies(flightBase, flightPattern, rule) {
			continue
		}
		if len(rule.NationalVersion) != 0 && strings.EqualFold(rule.NationalVersion, nationalVersion) {
			return
		}
		if len(rule.FlightDateSince) == 0 && len(rule.FlightDateUntil) == 0 && len(rule.NationalVersion) == 0 {
			return
		}
		if len(rule.FlightDateSince) != 0 || len(rule.FlightDateUntil) != 0 {
			bannedFrom = rule.FlightDateSince
			bannedUntil = rule.FlightDateUntil
			return
		}
	}
	isBanned = false
	return
}
