package flight

import (
	"sort"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/storage/carrier"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/utils"
	"a.yandex-team.ru/travel/avia/shared_flights/api/pkg/structs"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/dtutil"
	iatacorrector "a.yandex-team.ru/travel/avia/shared_flights/lib/go/iata_correction"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/logger"
)

type DopFlightsStorage struct {
	DopFlightBases    map[int32]structs.FlightBase
	DopFlightPatterns map[int32]*structs.FlightPattern
	DopFlights        map[string]FlightLegs
	CarrierStorage    *carrier.CarrierStorage
	// Maps (stationFrom, stationTo) pair to the list of keys for the Flights map
	FlightsP2P map[uint64]utils.StringList
	// Maps (stationFrom, stationTo) pair to the list of keys for the Flights map, but only for operating flights, no codeshares
	FlightsP2POperating map[uint64]utils.StringList
	iataCorrector       iatacorrector.IataCorrector
	nextFlightPatternID int32
	StartDate           string
}

func NewDopFlightStorage(
	iataCorrector iatacorrector.IataCorrector,
	carrierStorage *carrier.CarrierStorage,
	startDate string) *DopFlightsStorage {
	instance := DopFlightsStorage{
		DopFlightBases:      make(map[int32]structs.FlightBase),
		DopFlightPatterns:   make(map[int32]*structs.FlightPattern),
		DopFlights:          make(map[string]FlightLegs),
		FlightsP2P:          make(map[uint64]utils.StringList),
		FlightsP2POperating: make(map[uint64]utils.StringList),
		CarrierStorage:      carrierStorage,
		iataCorrector:       iataCorrector,
		nextFlightPatternID: MinIataCorrectedFlightPatternID,
		StartDate:           dtutil.GetMonthAgoString(),
	}
	if startDate != "" {
		instance.StartDate = startDate
	}
	return &instance
}

func (s *DopFlightsStorage) PutDopFlightBase(dopFlightBase structs.FlightBase) {
	s.DopFlightBases[dopFlightBase.ID] = dopFlightBase
}

func (s *DopFlightsStorage) PutDopFlightPattern(dopFlightPattern structs.FlightPattern) {
	flyingCarrierIata := ""
	if fb, ok := s.DopFlightBases[dopFlightPattern.FlightBaseID]; ok {
		flyingCarrierIata = fb.FlyingCarrierIata
	}

	carrierSirena := s.CarrierStorage.GetSirenaByID(dopFlightPattern.MarketingCarrier)
	correctedCarrierID := s.iataCorrector.FindFlightPatternCarrier(&dopFlightPattern, flyingCarrierIata, carrierSirena)
	if correctedCarrierID != 0 {
		correctedCarrierIata := s.CarrierStorage.GetCarrierCodeByID(correctedCarrierID)
		if correctedCarrierIata == dopFlightPattern.MarketingCarrierCode {
			// See RASPTICKETS-17651 for details
			dopFlightPattern.FilingCarrier = dopFlightPattern.MarketingCarrier
			dopFlightPattern.FilingCarrierCode = dopFlightPattern.MarketingCarrierCode
			dopFlightPattern.MarketingCarrier = correctedCarrierID
			if !s.updateFlightBase(dopFlightPattern.FlightBaseID, correctedCarrierID, correctedCarrierIata) {
				return
			}
		} else {
			correctedFlightPattern := dopFlightPattern
			s.nextFlightPatternID++
			correctedFlightPattern.ID = s.nextFlightPatternID
			correctedFlightPattern.MarketingCarrier = correctedCarrierID
			correctedFlightPattern.MarketingCarrierCode = s.CarrierStorage.GetCarrierCodeByID(correctedCarrierID)
			correctedFlightPattern.FilingCarrier = dopFlightPattern.MarketingCarrier
			correctedFlightPattern.FilingCarrierCode = dopFlightPattern.MarketingCarrierCode
			dopFlightPattern.CorrectedCarrier = correctedCarrierID
			if !dopFlightPattern.IsCodeshare {
				dopFlightPattern.IsCodeshare = true
				dopFlightPattern.OperatingFlightPatternID = correctedFlightPattern.ID
				if !s.updateFlightBase(dopFlightPattern.FlightBaseID, correctedCarrierID, correctedCarrierIata) {
					return
				}
			}
			s.DopFlightPatterns[correctedFlightPattern.ID] = &correctedFlightPattern
		}
	}

	s.DopFlightPatterns[dopFlightPattern.ID] = &dopFlightPattern
	s.updateDopFlights(&dopFlightPattern)
}

func (s *DopFlightsStorage) updateFlightBase(flightBaseID, carrierID int32, carrierCode string) bool {
	flightBase, ok := s.DopFlightBases[flightBaseID]
	if !ok {
		logger.Logger().Error(
			"Unknown flight base ID",
			log.Int32("id", flightBaseID),
			log.Int32("carrierID", carrierID),
			log.String("carrierCode", carrierCode),
		)
		return false
	}
	flightBase.OperatingCarrier = carrierID
	flightBase.OperatingCarrierCode = carrierCode
	return true
}

// Assumes there's no more than one thread doing updates
func (s *DopFlightsStorage) updateDopFlights(dopFlightPattern *structs.FlightPattern) {
	key := GetFlightKeyForPattern(dopFlightPattern)

	values, hasValues := s.DopFlights[key]
	if !hasValues {
		values = FlightLegs{}
	}
	legIndex := -1
	for idx, legsList := range values {
		for _, leg := range legsList {
			if leg.LegNumber == dopFlightPattern.LegNumber {
				legIndex = idx
				break
			}
		}
	}
	if legIndex >= 0 {
		values[legIndex] = append(values[legIndex], dopFlightPattern)
	} else {
		values = append(values, LegsList{dopFlightPattern})
	}
	sort.Slice(values, func(i, j int) bool {
		return values[i][0].LegNumber < values[j][0].LegNumber
	})
	s.DopFlights[key] = values
	s.updateFlightsP2PMap(key, dopFlightPattern)
}

func (s *DopFlightsStorage) updateFlightsP2PMap(flightKey string, dopFlightPattern *structs.FlightPattern) {
	if dopFlightPattern.OperatingUntilDate < s.StartDate {
		return
	}

	flightBase, ok := s.DopFlightBases[dopFlightPattern.FlightBaseID]
	if !ok {
		logger.Logger().Error(
			"Unable to get dop flight base object",
			log.Int32("dop flight base id", dopFlightPattern.FlightBaseID),
			log.Int32("dop flight pattern id", dopFlightPattern.ID),
		)
		return
	}
	p2pKey := utils.GetFlightP2PKey(flightBase.DepartureStation, flightBase.ArrivalStation)
	utils.UpdateMapToSortedStringList(s.FlightsP2P, p2pKey, flightKey)
	if !dopFlightPattern.IsCodeshare {
		utils.UpdateMapToSortedStringList(s.FlightsP2POperating, p2pKey, flightKey)
	}
}

func (s *DopFlightsStorage) GetFlightsP2POperating() map[uint64]utils.StringList {
	return s.FlightsP2POperating
}

func (s *DopFlightsStorage) GetFlightPatterns() map[int32]*structs.FlightPattern {
	return s.DopFlightPatterns
}
