package flight

import (
	"fmt"
	"sort"
	"strconv"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/config"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/storage/carrier"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/utils"
	"a.yandex-team.ru/travel/avia/shared_flights/api/pkg/structs"
	dir "a.yandex-team.ru/travel/avia/shared_flights/lib/go/direction"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/dtutil"
	iatacorrector "a.yandex-team.ru/travel/avia/shared_flights/lib/go/iata_correction"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/logger"
	"a.yandex-team.ru/travel/proto/shared_flights/snapshots"
)

type LegsList []*structs.FlightPattern
type FlightLegs []LegsList

type FlightPatternAndBase struct {
	FlightBase    structs.FlightBase
	FlightPattern structs.FlightPattern
}

func (legsList FlightLegs) Flatten() []*structs.FlightPattern {
	var result []*structs.FlightPattern
	for _, legFlights := range legsList {
		for _, fp := range legFlights {
			result = append(result, fp)
		}
	}
	return result
}

type StationAndDirection struct {
	StationID int32
	Direction dir.Direction
}
type OperatingFlightKey struct {
	Carrier      int32
	FlightNumber string
}
type FlightBaseID = int32
type FlightPatternID = int32

type flightStorageImpl struct {
	FlightBases    map[FlightBaseID]structs.FlightBase
	FlightPatterns map[FlightPatternID]*structs.FlightPattern

	// Flights by marketing carrier and flight number,
	// order of the flight patterns in the value defines their priority (bigger index => higher priority)
	Flights map[string]FlightLegs
	// Flights by filing carrier (only cached when filing carrier differs from the marketing one)
	FlightsByFilingCarrier map[string]FlightLegs
	// Maps (stationFrom, stationTo) pair to the list of keys for the Flights map
	FlightsP2P map[uint64]utils.StringList
	// Maps (stationFrom, stationTo) pair to the list of keys for the Flights map, but only for operating flights, no codeshares
	FlightsP2POperating map[uint64]utils.StringList
	// Caches codeshare flight keys for every operating flight
	CodeshareCache map[OperatingFlightKey]utils.StringList

	AeroflotFlightsCache AeroflotFlightsCache
	DopFlights           *DopFlightsStorage
	FlightCacheConfig    config.FlightCacheConfig
	StartDate            string
}

func (s *flightStorageImpl) Copy() FlightStorage {
	return &flightStorageImpl{
		FlightBases:            s.FlightBases,
		FlightPatterns:         s.FlightPatterns,
		Flights:                s.Flights,
		FlightsByFilingCarrier: s.FlightsByFilingCarrier,
		FlightsP2P:             s.FlightsP2P,
		FlightsP2POperating:    s.FlightsP2POperating,
		CodeshareCache:         s.CodeshareCache,
		AeroflotFlightsCache:   s.AeroflotFlightsCache,
		DopFlights:             s.DopFlights,
		FlightCacheConfig:      s.FlightCacheConfig,
		StartDate:              s.GetStartDate(),
	}
}

type FlightStorage interface {
	PutFlightBase(flightBase structs.FlightBase)
	GetFlightBase(id int32, isDopFlight bool) (flightBase structs.FlightBase, err error)
	PutFlightPattern(flightPattern structs.FlightPattern)
	GetFlightPatterns() map[int32]*structs.FlightPattern
	GetDopFlightPatterns() map[int32]*structs.FlightPattern
	GetFlights(marketingCarrier int32, flightNumber string) (flights FlightLegs, ok bool)
	GetFlightsByKey(flightKey string) (flights FlightLegs, ok bool)
	GetFlightsP2P(departureStation, arrivalStation int64) (flights utils.StringList, ok bool)
	GetFlightsP2POperating() (flights, dopFlights map[uint64]utils.StringList)
	GetCodeshareFlightKeys(operatingCarrier int32, flightNumber string) (flights utils.StringList, ok bool)
	SetDopFlights(newStorage *DopFlightsStorage)
	SetFlightCacheConfig(flightCacheConfig config.FlightCacheConfig)
	FindLegInfo(carrierID int32, flightNumber, flightDate string, departureStation, arrivalStation int64, useArrivalShift bool) (legInfo FlightPatternAndBase, err error)
	// for setting up tests (slow methods, added only for the sake of convenience)
	PutDopFlightBase(dopFlightBase structs.FlightBase)
	PutDopFlightPattern(dopFlightPattern structs.FlightPattern)
	Copy() FlightStorage
	GetStartDate() string
	UpdateP2PCache(entry *snapshots.TP2PCacheEntry)
	AeroflotCache() AeroflotFlightsCache
}

func NewFlightStorage(
	iataCorrector iatacorrector.IataCorrector,
	carrierStorage *carrier.CarrierStorage,
	startDate string) *flightStorageImpl {
	instance := flightStorageImpl{
		FlightBases:            make(map[int32]structs.FlightBase),
		FlightPatterns:         make(map[int32]*structs.FlightPattern),
		Flights:                make(map[string]FlightLegs),
		FlightsByFilingCarrier: make(map[string]FlightLegs),
		FlightsP2P:             make(map[uint64]utils.StringList),
		FlightsP2POperating:    make(map[uint64]utils.StringList),
		CodeshareCache:         make(map[OperatingFlightKey]utils.StringList),
		DopFlights:             NewDopFlightStorage(iataCorrector, carrierStorage, startDate),
		FlightCacheConfig: config.FlightCacheConfig{
			OnlyCarriers:    []string{},
			OnlyCarriersMap: map[string]bool{},
			MaxFlights:      0,
		},
		StartDate: dtutil.GetMonthAgoString(),
	}
	instance.AeroflotFlightsCache = NewAeroflotFlightsCache(&instance)
	if startDate != "" {
		instance.StartDate = startDate
	}
	return &instance
}

func (s *flightStorageImpl) SetFlightCacheConfig(flightCacheConfig config.FlightCacheConfig) {
	s.FlightCacheConfig = flightCacheConfig
}

func (s *flightStorageImpl) PutFlightBase(flightBase structs.FlightBase) {
	s.FlightBases[flightBase.ID] = flightBase
}

func (s *flightStorageImpl) GetFlightBase(id int32, isDopFlight bool) (flightBase structs.FlightBase, err error) {
	if isDopFlight {
		value, hasValue := s.DopFlights.DopFlightBases[id]
		if !hasValue {
			return value, xerrors.Errorf("unknown dop flight base id %v", id)
		}
		return value, nil
	}
	value, hasValue := s.FlightBases[id]
	if !hasValue {
		return value, xerrors.Errorf("unknown flight base id %v", id)
	}
	return value, nil
}

func (s *flightStorageImpl) PutFlightPattern(flightPattern structs.FlightPattern) {
	if s.FlightCacheConfig.MaxFlights > 0 && int32(len(s.FlightPatterns)) >= s.FlightCacheConfig.MaxFlights {
		if len(s.FlightCacheConfig.OnlyCarriersMap) > 0 {
			_, ok := s.FlightCacheConfig.OnlyCarriersMap[flightPattern.MarketingCarrierCode]
			if !ok {
				_, ok = s.FlightCacheConfig.OnlyCarriersMap[strconv.Itoa(int(flightPattern.MarketingCarrier))]
			}
			if !ok {
				return
			}
		}
	}
	s.FlightPatterns[flightPattern.ID] = &flightPattern
	s.updateFlights(&flightPattern)
}

// Assumes there's no more than one thread doing updates
func (s *flightStorageImpl) updateFlights(flightPattern *structs.FlightPattern) {
	key := GetFlightKeyForPattern(flightPattern)
	updateFlightsMap(s.Flights, key, flightPattern)

	if flightPattern.FilingCarrier != 0 && flightPattern.FilingCarrier != flightPattern.MarketingCarrier {
		keyByFilingCarrier := GetFlightKey(flightPattern.FilingCarrier, flightPattern.MarketingFlightNumber)
		updateFlightsMap(s.FlightsByFilingCarrier, keyByFilingCarrier, flightPattern)
	}

	if flightPattern.IsCodeshare {
		flightBase, ok := s.FlightBases[flightPattern.FlightBaseID]
		if ok {
			s.updateCodeshareCache(
				flightBase.OperatingCarrier,
				flightBase.OperatingFlightNumber,
				flightPattern.MarketingCarrier,
				flightPattern.MarketingFlightNumber,
			)
		} else {
			logger.Logger().Error(
				"No cossresponding flight base for the flight pattern",
				log.Reflect("pattern", flightPattern),
			)
		}
	}
}

func (s *flightStorageImpl) UpdateP2PCache(entry *snapshots.TP2PCacheEntry) {
	p2pKey := utils.GetFlightP2PKey(int64(entry.DepartureStationId), int64(entry.ArrivalStationId))
	for _, flight := range entry.Flights {
		flightKey := GetFlightKey(flight.MarketingCarrierId, flight.MarketingFlightNumber)
		s.updateFlightsP2PMapInternal(p2pKey, flightKey)
		if !flight.IsCodeshare {
			s.updateFlightsP2POperatingMapInternal(p2pKey, flightKey)
		}
	}
}

func (s *flightStorageImpl) updateFlightsP2PMapInternal(p2pKey uint64, flightKey string) {
	utils.UpdateMapToSortedStringList(s.FlightsP2P, p2pKey, flightKey)
}

func (s *flightStorageImpl) updateFlightsP2POperatingMapInternal(p2pKey uint64, flightKey string) {
	utils.UpdateMapToSortedStringList(s.FlightsP2POperating, p2pKey, flightKey)
}

func (s *flightStorageImpl) updateCodeshareCache(
	operatingCarrier int32, operatingFlightNumber string, marketingCarrier int32, marketingFlightNumber string) {
	operatingKey := OperatingFlightKey{operatingCarrier, operatingFlightNumber}
	marketingKey := GetFlightKey(marketingCarrier, marketingFlightNumber)
	flightKeys, ok := s.CodeshareCache[operatingKey]
	if !ok {
		s.CodeshareCache[operatingKey] = []string{marketingKey}
	} else {
		s.CodeshareCache[operatingKey] = utils.UpdateSortedStringList(flightKeys, marketingKey)
	}
}

func (s *flightStorageImpl) GetFlightPatterns() map[int32]*structs.FlightPattern {
	return s.FlightPatterns
}

func (s *flightStorageImpl) GetDopFlightPatterns() map[int32]*structs.FlightPattern {
	return s.DopFlights.DopFlightPatterns
}

func (s *flightStorageImpl) GetFlightsP2POperating() (flights, dopFlights map[uint64]utils.StringList) {
	dopFlightsMap := s.DopFlights.GetFlightsP2POperating()
	return s.FlightsP2POperating, dopFlightsMap
}

func (s *flightStorageImpl) GetFlights(marketingCarrier int32, flightNumber string) (flights FlightLegs, ok bool) {
	return s.GetFlightsByKey(GetFlightKey(marketingCarrier, flightNumber))
}

func (s *flightStorageImpl) GetFlightsByKey(flightKey string) (flights FlightLegs, ok bool) {
	scheduledFlights, scheduledOk := s.Flights[flightKey]
	dopFlights, dopOk := s.DopFlights.DopFlights[flightKey]

	if !dopOk {
		return scheduledFlights, scheduledOk
	}

	if !scheduledOk {
		return dopFlights, dopOk
	}

	return append(scheduledFlights, dopFlights...), true
}

func (s *flightStorageImpl) GetFlightsP2P(departureStation, arrivalStation int64) (flights utils.StringList, ok bool) {
	p2pKey := utils.GetFlightP2PKey(departureStation, arrivalStation)
	flights, ok = s.FlightsP2P[p2pKey]
	dopFlights, dopOk := s.DopFlights.FlightsP2P[p2pKey]
	if !ok {
		return dopFlights, dopOk
	}
	if !dopOk {
		return flights, ok
	}

	return append(flights, dopFlights...), true
}

func (s *flightStorageImpl) GetCodeshareFlightKeys(operatingCarrier int32, flightNumber string) (flights utils.StringList, ok bool) {
	flights, ok = s.CodeshareCache[OperatingFlightKey{operatingCarrier, flightNumber}]
	return
}

var dateOutOfIndexError = xerrors.New("date_out_of_index")

// First we search by marketing carrier, then (if unsuccessful) by filing carrier; this is to avoid applying IATA correction rules
// to flight statuses, as those don't have flying or designated carriers specified.
// Returns a flight-base/flight-pattern pair or nil (if not found).
func (s *flightStorageImpl) FindLegInfo(carrierID int32, flightNumber, flightDate string, departureStation, arrivalStation int64, useArrivalShift bool) (legInfo FlightPatternAndBase, err error) {
	flightDateIndex, flightDateIndexOk := dtutil.DateCache.IndexOfStringDate(dtutil.StringDate(flightDate))
	if !flightDateIndexOk {
		return FlightPatternAndBase{}, dateOutOfIndexError
	}

	flightKey := GetFlightKey(carrierID, flightNumber)
	legInfo, err = s.findLegNumberInMap(s.Flights, flightKey, flightDateIndex, departureStation, arrivalStation, useArrivalShift)
	if err == nil {
		return legInfo, nil
	}

	return s.findLegNumberInMap(s.FlightsByFilingCarrier, flightKey, flightDateIndex, departureStation, arrivalStation, useArrivalShift)
}

var flightKeyNotFoundError = xerrors.New("flight key not found")
var SegmentNotFoundError = xerrors.New("segment_not_found")

func (s *flightStorageImpl) findLegNumberInMap(flightsMap map[string]FlightLegs, flightKey string, flightDateIndex int, departureStation, arrivalStation int64, useArrivalShift bool) (FlightPatternAndBase, error) {
	flightLegs, ok := flightsMap[flightKey]
	if !ok {
		return FlightPatternAndBase{}, flightKeyNotFoundError
	}

	for _, flightLeg := range flightLegs {
		for _, flightPattern := range flightLeg {
			flightBase, flightBaseOk := s.FlightBases[flightPattern.FlightBaseID]
			if !flightBaseOk {
				continue
			}

			departureStationMatchesOrAbsent := departureStation == 0 || flightBase.DepartureStation == departureStation
			arrivalStationMatchesOrAbsent := arrivalStation == 0 || flightBase.ArrivalStation == arrivalStation

			if !(departureStationMatchesOrAbsent && arrivalStationMatchesOrAbsent) {
				continue
			}

			segmentDepartureDate := flightDateIndex
			if useArrivalShift {
				segmentDepartureDate -= int(flightPattern.ArrivalDayShift)
			}

			operatesFromDateIndex, idxOK := dtutil.DateCache.IndexOfStringDate(dtutil.StringDate(flightPattern.OperatingFromDate))
			if !idxOK {
				logger.Logger().Error(
					"Flight pattern operating from date out of date cache range",
					log.Reflect("pattern", flightPattern),
				)
				continue
			}

			operatesUntilDateIndex, idxOK := dtutil.DateCache.IndexOfStringDate(dtutil.StringDate(flightPattern.OperatingUntilDate))
			if !idxOK {
				logger.Logger().Error(
					"Flight pattern operating until date out of date cache range",
					log.Reflect("pattern", flightPattern),
				)
				continue
			}

			if operatesFromDateIndex > segmentDepartureDate {
				continue
			}
			if operatesUntilDateIndex < segmentDepartureDate {
				continue
			}
			if !dtutil.OperatesOn(flightPattern.OperatingOnDays, dtutil.DateCache.WeekDay(segmentDepartureDate)) {
				continue
			}
			return FlightPatternAndBase{
				FlightBase:    flightBase,
				FlightPattern: *flightPattern,
			}, nil
		}
	}
	return FlightPatternAndBase{}, SegmentNotFoundError
}

func (s *flightStorageImpl) SetDopFlights(newStorage *DopFlightsStorage) {
	s.DopFlights = newStorage
}

func GetFlightKey(marketingCarrier int32, flightNumber string) string {
	return fmt.Sprintf("%v/%v", marketingCarrier, flightNumber)
}

func GetFlightKeyForPattern(flightPattern *structs.FlightPattern) string {
	return GetFlightKey(flightPattern.MarketingCarrier, flightPattern.MarketingFlightNumber)
}

func updateFlightsMap(flightsMap map[string]FlightLegs, key string, flightPattern *structs.FlightPattern) {
	values, hasValues := flightsMap[key]
	if !hasValues {
		values = FlightLegs{}
	}
	legIndex := -1
	for idx, legsList := range values {
		for _, leg := range legsList {
			if leg.LegNumber == flightPattern.LegNumber {
				// Do not add duplicate legs
				if leg.ID == flightPattern.ID {
					return
				}
				legIndex = idx
			}
		}
	}
	if legIndex >= 0 {
		values[legIndex] = append(values[legIndex], flightPattern)
	} else {
		values = append(values, LegsList{flightPattern})
	}
	sort.Slice(values, func(i, j int) bool {
		return values[i][0].LegNumber < values[j][0].LegNumber
	})
	flightsMap[key] = values
}

// Slow method, don't use in the performance-sensitive code
func (s *flightStorageImpl) PutDopFlightBase(dopFlightBase structs.FlightBase) {
	s.DopFlights.PutDopFlightBase(dopFlightBase)
}

// Slow method, don't use in the performance-sensitive code
func (s *flightStorageImpl) PutDopFlightPattern(dopFlightPattern structs.FlightPattern) {
	s.DopFlights.PutDopFlightPattern(dopFlightPattern)
}

func (s *flightStorageImpl) AeroflotCache() AeroflotFlightsCache {
	return s.AeroflotFlightsCache
}

func (s *flightStorageImpl) GetAeroflotConnections(stationFrom, stationTo int64) map[string][][]FlightPatternAndBase {
	return s.AeroflotFlightsCache.GetConnectionsMap(stationFrom, stationTo)
}

func (s *flightStorageImpl) GetStartDate() string {
	return s.StartDate
}
