package flight

import (
	"regexp"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/utils"
	"a.yandex-team.ru/travel/avia/shared_flights/api/pkg/structs"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/logger"
)

type FlightMergeRuleStorage interface {
	AddRule(flightMergeRule structs.FlightMergeRule) bool
	ShouldMerge(operatingCarrier int32, operatingFlight string, marketingCarrier int32, marketingFlight string) bool
	GetRules() []structs.FlightMergeRule
	RulesCount() int
	UseLesserFlightNumberAsOperating(turnOn bool)
}

type RegexpMap map[int32]*regexp.Regexp

type void struct{}

var empty = void{}

type flightMergeRuleStorageImpl struct {
	Rules                              map[int32][]structs.FlightMergeRule
	OperatingRegexps                   RegexpMap
	MarketingRegexps                   RegexpMap
	MergeLesserFlightNumberAsOperating bool
	ExcludedCarriers                   map[int32]void
}

func NewFlightMergeRuleStorage() FlightMergeRuleStorage {
	return &flightMergeRuleStorageImpl{
		Rules:                              make(map[int32][]structs.FlightMergeRule),
		OperatingRegexps:                   make(RegexpMap),
		MarketingRegexps:                   make(RegexpMap),
		MergeLesserFlightNumberAsOperating: true,
		ExcludedCarriers:                   make(map[int32]void),
	}
}

func (s *flightMergeRuleStorageImpl) UseLesserFlightNumberAsOperating(turnOn bool) {
	s.MergeLesserFlightNumberAsOperating = turnOn
}

func (s *flightMergeRuleStorageImpl) RulesCount() int {
	rulesCount := 0
	for _, rules := range s.Rules {
		rulesCount += len(rules)
	}
	return rulesCount
}

func (s *flightMergeRuleStorageImpl) GetRules() []structs.FlightMergeRule {
	result := make([]structs.FlightMergeRule, 0)
	for _, rules := range s.Rules {
		result = append(result, rules...)
	}
	return result
}

func (s *flightMergeRuleStorageImpl) AddRule(rule structs.FlightMergeRule) bool {
	if !rule.IsActive {
		return false
	}
	if rule.ExcludedCarrier > 0 {
		s.ExcludedCarriers[rule.ExcludedCarrier] = empty
		return true
	}
	if rule.OperatingCarrier == rule.MarketingCarrier {
		return false
	}
	rulesList, ok := s.Rules[rule.MarketingCarrier]
	if !ok {
		s.Rules[rule.MarketingCarrier] = []structs.FlightMergeRule{rule}
	} else {
		s.Rules[rule.MarketingCarrier] = append(rulesList, rule)
	}
	updateRegexps(s.OperatingRegexps, rule.OperatingFlightRegexp, rule.ID)
	updateRegexps(s.MarketingRegexps, rule.MarketingFlightRegexp, rule.ID)
	return true
}

func updateRegexps(regexpMap RegexpMap, regexpText string, ruleID int32) {
	if regexpText == "" {
		return
	}
	re, err := regexp.Compile(regexpText)
	if err != nil {
		logger.Logger().Error(
			"Invalid flight number regexp in the rule",
			log.String("regexp", regexpText),
			log.Int32("ruleId", ruleID),
			log.Error(err),
		)
	} else {
		regexpMap[ruleID] = re
	}
}

func (s *flightMergeRuleStorageImpl) ShouldMerge(
	operatingCarrier int32, operatingFlight string, marketingCarrier int32, marketingFlight string) bool {
	if operatingCarrier == marketingCarrier {
		return false
	}
	if _, excludedOp := s.ExcludedCarriers[operatingCarrier]; excludedOp {
		return false
	}
	if _, excludedMkt := s.ExcludedCarriers[marketingCarrier]; excludedMkt {
		return false
	}
	rules, ok := s.Rules[marketingCarrier]
	if ok {
		for _, rule := range rules {
			if rule.OperatingCarrier != operatingCarrier {
				continue
			}
			if operatingRegexp, operatingOk := s.OperatingRegexps[rule.ID]; operatingOk {
				if !operatingRegexp.MatchString(operatingFlight) {
					continue
				}
			}
			if marketingRegexp, marketingOk := s.MarketingRegexps[rule.ID]; marketingOk {
				if !marketingRegexp.MatchString(marketingFlight) {
					continue
				}
			}
			return rule.ShouldMerge
		}
	}

	if !s.MergeLesserFlightNumberAsOperating {
		return false
	}

	// consider the flight with smaller flight number as the operating one
	flightNumber1 := utils.NumericFlightNumber(operatingFlight)
	flightNumber2 := utils.NumericFlightNumber(marketingFlight)
	return flightNumber1 < flightNumber2
}
