package flight

import (
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/avia/shared_flights/api/pkg/structs"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/logger"
)

const (
	DirectFlights = "direct"
)

type AeroflotFlightsCache interface {
	Rebuild() error
	GetConnectionsMap(stationFrom, stationTo int64) map[string][][]FlightPatternAndBase
	GetDirectFlights(stationFrom, stationTo int64) []*structs.FlightPattern
	GetConnectingFlights(stationFrom, stationTo int64) []FlightPatternsListPair
}

type SegmentsProvider interface {
	GetFlightPatterns() map[int32]*structs.FlightPattern
	GetFlightBase(int32, bool) (structs.FlightBase, error)
}

type aeroflotFlightsCacheImpl struct {
	flightPatternsProvider SegmentsProvider
	directCache            map[routeKey]FlightPatternsList
	connectionsCache       map[routeKey][]FlightPatternsListPair
}

type routeKey struct {
	stationFrom int64
	stationTo   int64
}

type FlightPatternsList []*structs.FlightPattern
type FlightPatternsListPair struct {
	FirstSegments  FlightPatternsList
	SecondSegments FlightPatternsList
}

func NewAeroflotFlightsCache(flightPatternsProvider SegmentsProvider) AeroflotFlightsCache {
	return &aeroflotFlightsCacheImpl{
		flightPatternsProvider: flightPatternsProvider,
	}
}

func (c *aeroflotFlightsCacheImpl) Rebuild() error {
	logger.Logger().Info("Start building Aeroflot cache", log.Time("start", time.Now()))
	stationsCache := make(map[int64]bool)
	directCache := make(map[routeKey]FlightPatternsList)
	for _, fp := range c.flightPatternsProvider.GetFlightPatterns() {
		if fp.IsCodeshare || !isAeroflotCarrier(fp.MarketingCarrierCode) {
			continue
		}
		fb, err := c.flightPatternsProvider.GetFlightBase(fp.FlightBaseID, false)
		if err != nil {
			continue
		}
		stationsCache[fb.DepartureStation] = true
		stationsCache[fb.ArrivalStation] = true
		route := routeKey{
			stationFrom: fb.DepartureStation,
			stationTo:   fb.ArrivalStation,
		}
		currentSegmentsList, ok := directCache[route]
		if !ok {
			currentSegmentsList = make(FlightPatternsList, 0)
		}
		currentSegmentsList = append(currentSegmentsList, fp)
		directCache[route] = currentSegmentsList
	}

	logger.Logger().Info("Start building Aeroflot connections cache", log.Time("start", time.Now()))

	connectionsCache := make(map[routeKey][]FlightPatternsListPair)
	for stationFrom := range stationsCache {
		for stationTo := range stationsCache {
			if stationFrom == stationTo {
				continue
			}
			route := routeKey{
				stationFrom: stationFrom,
				stationTo:   stationTo,
			}
			for connectingStation := range stationsCache {
				if connectingStation == stationFrom || connectingStation == stationTo {
					continue
				}
				segment1route := routeKey{
					stationFrom: stationFrom,
					stationTo:   connectingStation,
				}
				firstSegments := directCache[segment1route]
				if len(firstSegments) == 0 {
					continue
				}
				segment2route := routeKey{
					stationFrom: connectingStation,
					stationTo:   stationTo,
				}
				secondSegments := directCache[segment2route]
				if len(secondSegments) == 0 {
					continue
				}
				currentSegments, ok := connectionsCache[route]
				if !ok {
					currentSegments = make([]FlightPatternsListPair, 0)
				}
				currentSegments = append(
					currentSegments,
					FlightPatternsListPair{
						FirstSegments:  firstSegments,
						SecondSegments: secondSegments,
					},
				)
				connectionsCache[route] = currentSegments
			}
		}
	}
	c.directCache = directCache
	c.connectionsCache = connectionsCache

	logger.Logger().Info(
		"End building Aeroflot cache",
		log.Time("end", time.Now()),
		log.Int("stations", len(stationsCache)),
		log.Int(DirectFlights, len(directCache)),
		log.Int("connections", len(connectionsCache)),
	)
	return nil
}

func (c *aeroflotFlightsCacheImpl) GetDirectFlights(stationFrom, stationTo int64) []*structs.FlightPattern {
	route := routeKey{
		stationFrom: stationFrom,
		stationTo:   stationTo,
	}
	return c.directCache[route]
}

func (c *aeroflotFlightsCacheImpl) GetConnectingFlights(stationFrom, stationTo int64) []FlightPatternsListPair {
	route := routeKey{
		stationFrom: stationFrom,
		stationTo:   stationTo,
	}
	return c.connectionsCache[route]
}

func (c *aeroflotFlightsCacheImpl) GetConnectionsMap(stationFrom, stationTo int64) map[string][][]FlightPatternAndBase {
	result := make(map[string][][]FlightPatternAndBase)
	route := routeKey{
		stationFrom: stationFrom,
		stationTo:   stationTo,
	}
	directSegments := c.directCache[route]
	if len(directSegments) > 0 {
		result[DirectFlights] = [][]FlightPatternAndBase{c.convertToFlightPatternsAndBaseList(directSegments)}
	}
	connectedSegments := c.connectionsCache[route]
	if len(connectedSegments) > 0 {
		for _, segmentPairs := range connectedSegments {
			var connectingStation string
			fb, err := c.flightPatternsProvider.GetFlightBase(segmentPairs.FirstSegments[0].FlightBaseID, false)
			if err == nil {
				connectingStation = fb.ArrivalStationCode
			} else {
				connectingStation = "error"
			}
			result[connectingStation] = [][]FlightPatternAndBase{
				c.convertToFlightPatternsAndBaseList(segmentPairs.FirstSegments),
				c.convertToFlightPatternsAndBaseList(segmentPairs.SecondSegments),
			}
		}
	}
	return result
}

func (c *aeroflotFlightsCacheImpl) convertToFlightPatternsAndBaseList(flightPatterns []*structs.FlightPattern) []FlightPatternAndBase {
	value := make([]FlightPatternAndBase, 0)
	for _, fp := range flightPatterns {
		fpAndFb := FlightPatternAndBase{FlightPattern: *fp}
		fb, err := c.flightPatternsProvider.GetFlightBase(fp.FlightBaseID, false)
		if err == nil {
			fpAndFb.FlightBase = fb
		}
		value = append(value, fpAndFb)
	}
	return value
}

func isAeroflotCarrier(iataCode string) bool {
	return iataCode == "SU" || iataCode == "FV"
}
