package station

import (
	"sync"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/avia/shared_flights/api/pkg/structs"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/direction"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/dtutil"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/logger"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/strutil"
	"a.yandex-team.ru/travel/proto/shared_flights/snapshots"
)

type StationStorage interface {
	ByIcao(code string) (*snapshots.TStationWithCodes, bool)
	BySirena(code string) (*snapshots.TStationWithCodes, bool)
	ByIata(code string) (*snapshots.TStationWithCodes, bool)
	ByID(id int64) (*snapshots.TStationWithCodes, bool)
	PutStation(station *snapshots.TStationWithCodes)
	GetCode(id int64) string
}

type stationStorageImpl struct {
	byID     sync.Map
	byIata   sync.Map
	bySirena sync.Map
	byIcao   sync.Map
}

func NewStationStorage() StationStorage {
	return &stationStorageImpl{}
}

func (s *stationStorageImpl) ByIcao(code string) (*snapshots.TStationWithCodes, bool) {
	v, ok := s.byIcao.Load(code)
	if ok {
		station := v.(*snapshots.TStationWithCodes)
		return station, ok
	}
	return nil, ok
}

func (s *stationStorageImpl) BySirena(code string) (*snapshots.TStationWithCodes, bool) {
	v, ok := s.bySirena.Load(code)
	if ok {
		station := v.(*snapshots.TStationWithCodes)
		return station, ok
	}
	return nil, ok
}

func (s *stationStorageImpl) ByIata(code string) (*snapshots.TStationWithCodes, bool) {
	v, ok := s.byIata.Load(code)
	if ok {
		station := v.(*snapshots.TStationWithCodes)
		return station, ok
	}
	return nil, ok
}

func (s *stationStorageImpl) ByID(id int64) (*snapshots.TStationWithCodes, bool) {
	v, ok := s.byID.Load(id)
	if ok {
		station := v.(*snapshots.TStationWithCodes)
		return station, ok
	}

	return nil, ok
}

func (s *stationStorageImpl) PutStation(station *snapshots.TStationWithCodes) {
	if station.Station.Id != 0 {
		s.byID.Store(int64(station.Station.Id), station)
	}
	if len(station.IataCode) > 0 {
		s.byIata.Store(station.IataCode, station)
	}
	if len(station.SirenaCode) > 0 {
		s.bySirena.Store(station.SirenaCode, station)
	}
	if len(station.IcaoCode) > 0 {
		s.byIcao.Store(station.IcaoCode, station)
	}
}

func (s *stationStorageImpl) GetCode(id int64) string {
	station, ok := s.ByID(id)
	if !ok {
		return ""
	}
	return GetStationCode(station)
}

func GetStationCode(station *snapshots.TStationWithCodes) string {
	if station == nil {
		return ""
	}
	return strutil.Coalesce(station.IataCode, station.SirenaCode, station.IcaoCode)
}

type FlightPatternExtras struct {
	OperatingFrom          dtutil.IntDate
	OperatingUntil         dtutil.IntDate
	DepartureTimeScheduled int32
	ArrivalTimeScheduled   int32
}

type FlightBoardStorage interface {
	PutFlightPatterns(map[int32]*structs.FlightPattern, map[int32]*structs.FlightPattern) error
	GetFlightPatterns(*snapshots.TStationWithCodes, direction.Direction) ([]*structs.FlightPattern, bool)
	GetExtras(*snapshots.TStationWithCodes) (map[int32]FlightPatternExtras, bool)
}

type flightBaseProvider interface {
	GetFlightBase(id int32, isDopFlight bool) (flightBase structs.FlightBase, err error)
}

type flightBoardStorageImpl struct {
	flightBaseProvider flightBaseProvider
	flightBoardCache   *flightBoardCache
}
type tStationID int64

func NewFlightBoard(flightBaseProvider flightBaseProvider) FlightBoardStorage {
	return &flightBoardStorageImpl{
		flightBaseProvider: flightBaseProvider,
		flightBoardCache:   newFlightBoardCache(),
	}
}

func (f *flightBoardStorageImpl) PutFlightPatterns(patterns map[int32]*structs.FlightPattern, dopPatterns map[int32]*structs.FlightPattern) error {
	logger.Logger().Info("Start: Updating flight board")
	defer func() { logger.Logger().Info("Done: Updating flight board") }()
	// TODO(u-jeen): Improve FlightBoardCache, so it would allow swapping patterns and dopPatterns caches independently
	// (see more on this: https://a.yandex-team.ru/review/1118644/details)
	newCache := newFlightBoardCache()
	err := newCache.putFlightPatterns(patterns, f.flightBaseProvider)
	if err != nil {
		return xerrors.Errorf("error filling new cache: %w", err)
	}
	err = newCache.putFlightPatterns(dopPatterns, f.flightBaseProvider)
	if err != nil {
		return xerrors.Errorf("error filling new cache with dop patterns: %w", err)
	}
	// replace old cache with fresh one
	// TODO(mikhailche): possible memory leak: avoid holding cache in other routines
	f.flightBoardCache = newCache
	return nil
}

func (f *flightBoardStorageImpl) GetFlightPatterns(
	station *snapshots.TStationWithCodes,
	dir direction.Direction,
) ([]*structs.FlightPattern, bool) {
	return f.flightBoardCache.getFlightPatterns(station, dir)
}

func (f *flightBoardStorageImpl) GetExtras(station *snapshots.TStationWithCodes) (map[int32]FlightPatternExtras, bool) {
	extras, ok := f.flightBoardCache.extras[tStationID(station.Station.Id)]
	return extras, ok
}

// Switchable cache
type flightBoardCache struct {
	departures map[tStationID][]*structs.FlightPattern
	arrivals   map[tStationID][]*structs.FlightPattern
	extras     map[tStationID]map[int32]FlightPatternExtras
}

func newFlightBoardCache() *flightBoardCache {
	return &flightBoardCache{
		make(map[tStationID][]*structs.FlightPattern),
		make(map[tStationID][]*structs.FlightPattern),
		make(map[tStationID]map[int32]FlightPatternExtras),
	}
}

// Not thread safe, populate in isolation
func (fbc *flightBoardCache) putFlightPatterns(
	patterns map[int32]*structs.FlightPattern,
	flightBaseProvider flightBaseProvider,
) error {
	for _, flightPattern := range patterns {
		flightBase, err := flightBaseProvider.GetFlightBase(flightPattern.FlightBaseID, flightPattern.IsDop)
		if err != nil {
			return xerrors.Errorf("cannot get flight base for flight pattern: %w", err)
		}
		fbc.departures[tStationID(flightBase.DepartureStation)] = append(fbc.departures[tStationID(flightBase.DepartureStation)], flightPattern)
		fbc.arrivals[tStationID(flightBase.ArrivalStation)] = append(fbc.arrivals[tStationID(flightBase.ArrivalStation)], flightPattern)
		fbc.putExtras(tStationID(flightBase.DepartureStation), flightPattern, &flightBase)
		fbc.putExtras(tStationID(flightBase.ArrivalStation), flightPattern, &flightBase)
	}
	return nil
}

func (fbc *flightBoardCache) putExtras(stationID tStationID, flightPattern *structs.FlightPattern, flightBase *structs.FlightBase) {
	extrasMap, ok := fbc.extras[stationID]
	if !ok {
		extrasMap = make(map[int32]FlightPatternExtras)
	}
	extrasMap[flightPattern.ID] = FlightPatternExtras{
		dtutil.StringDate(flightPattern.OperatingFromDate).ToIntDate(),
		dtutil.StringDate(flightPattern.OperatingUntilDate).ToIntDate(),
		flightBase.DepartureTimeScheduled,
		flightBase.ArrivalTimeScheduled,
	}
	fbc.extras[stationID] = extrasMap
}

func (fbc *flightBoardCache) getFlightPatterns(
	station *snapshots.TStationWithCodes,
	dir direction.Direction,
) ([]*structs.FlightPattern, bool) {
	switch dir {
	case direction.ARRIVAL:
		flightPatterns, ok := fbc.arrivals[tStationID(station.Station.Id)]
		return flightPatterns, ok
	case direction.DEPARTURE:
		flightPatterns, ok := fbc.departures[tStationID(station.Station.Id)]
		return flightPatterns, ok
	default:
		return nil, false
	}
}
