package status

import (
	"fmt"
	"sort"
	"sync"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/appconst"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/services/storage/flightstatus"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/services/storage/timezone"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/storage/station"
	"a.yandex-team.ru/travel/avia/shared_flights/api/pkg/structs"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/dtutil"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/logger"
	"a.yandex-team.ru/travel/proto/shared_flights/snapshots"
)

type StatusStorage interface {
	PutFlightStatus(flightStatus structs.FlightStatus) structs.FlightStatus
	GetFlightStatuses(bucketKey string) (*SynchronizedFlightStatusMap, bool)
	PutStatusSource(flightStatusSource structs.FlightStatusSource)
	GetStatusSources() map[int32]structs.FlightStatusSource
	UpdateFlightDelays(flightStatus structs.FlightStatus, scheduledDeparture, scheduledArrival string)
	GetDelayedFlights(station int64) (map[string]emptyType, bool)
	GetCancelledFlights(station int64) (map[string]emptyType, bool)
	GetDelayedOrCancelledStations() []int64
	UpdateDivertedIDs(station.StationStorage)
}

type emptyType struct{}

var empty = emptyType{}

type statusStorageImpl struct {
	FlightStatuses    map[string]*SynchronizedFlightStatusMap
	flightStatusMutex sync.RWMutex
	StatusSources     map[int32]structs.FlightStatusSource
	ReferenceTime     time.Time
	cancelledFlights  map[int64]map[string]emptyType
	delayedFlights    map[int64]map[string]emptyType
	tzProvider        timezone.TimeZoneProvider
}

type SynchronizedFlightStatusMap struct {
	sync.RWMutex
	statuses map[dtutil.IntDate]*structs.FlightStatus
}

var zeroPrioritySource = structs.FlightStatusSource{}

func NewStatusStorage(tzProvider timezone.TimeZoneProvider) StatusStorage {
	return &statusStorageImpl{
		FlightStatuses:    make(map[string]*SynchronizedFlightStatusMap),
		flightStatusMutex: sync.RWMutex{},
		StatusSources:     make(map[int32]structs.FlightStatusSource),
		ReferenceTime:     time.Now().UTC(),
		cancelledFlights:  make(map[int64]map[string]emptyType),
		delayedFlights:    make(map[int64]map[string]emptyType),
		tzProvider:        tzProvider,
	}
}

func (s *statusStorageImpl) PutFlightStatus(flightStatus structs.FlightStatus) structs.FlightStatus {
	bucketKey := GetFlightStatusKey(&flightStatus)
	flightStatuses := s.getOrCreateFlightStatuses(bucketKey)
	return flightStatuses.putStatus(flightStatus, s.StatusSources)
}

func stationID(fn func(code string) (*snapshots.TStationWithCodes, bool)) func(code string) (int32, bool) {
	return func(code string) (int32, bool) {
		if fn == nil {
			return 0, false
		}
		Station, exists := fn(code)
		if !exists {
			return 0, false
		}

		if Station == nil || Station.Station == nil {
			return 0, false
		}
		return Station.Station.Id, true
	}
}

// UpdateDivertedIDs translates diverted airport codes to ids and updates corresponding statuses
func (s *statusStorageImpl) UpdateDivertedIDs(stations station.StationStorage) {
	for _, v1 := range s.FlightStatuses {
		for _, v2 := range v1.statuses {
			v2.FillDivertedAirportIDs(
				stationID(stations.ByIata),
				stationID(stations.BySirena),
				stationID(stations.ByIcao),
			)
		}
	}
}

func (s *statusStorageImpl) UpdateFlightDelays(flightStatus structs.FlightStatus, scheduledDeparture, scheduledArrival string) {
	s.updateCancelledFlights(flightStatus, scheduledDeparture, scheduledArrival)
	s.updateDelayedFlights(flightStatus, scheduledDeparture, scheduledArrival)
}

func (s *statusStorageImpl) updateCancelledFlights(flightStatus structs.FlightStatus, scheduledDeparture, scheduledArrival string) {
	departureCancelled := flightStatus.DepartureSourceID == appconst.AirportFlightStatusSource &&
		flightstatus.IsCancelled(flightStatus.DepartureStatus)
	departureTrusted := flightStatus.DepartureSourceIsTrusted
	arrivalCancelled := flightStatus.ArrivalSourceID == appconst.AirportFlightStatusSource &&
		flightstatus.IsCancelled(flightStatus.ArrivalStatus)
	arrivalTrusted := flightStatus.ArrivalSourceIsTrusted

	if departureCancelled || arrivalCancelled && arrivalTrusted {
		s.updateCancelledFlightsByTime(
			flightStatus.DepartureStation,
			scheduledDeparture,
			GetFlightTitleKey(&flightStatus),
		)
	}
	if arrivalCancelled || departureCancelled && departureTrusted {
		s.updateCancelledFlightsByTime(
			flightStatus.ArrivalStation,
			scheduledArrival,
			GetFlightTitleKey(&flightStatus),
		)
	}
}

func (s *statusStorageImpl) updateDelayedFlights(flightStatus structs.FlightStatus, scheduledDeparture, scheduledArrival string) {
	if flightStatus.DepartureSourceID == appconst.AirportFlightStatusSource &&
		flightStatus.DepartureStatus != string(appconst.FlightStatusCancelled) {
		s.updateDelayedFlightsByTime(
			flightStatus.DepartureStation,
			scheduledDeparture,
			flightStatus.DepartureTimeActual,
			30,
			GetFlightTitleKey(&flightStatus),
		)
	}
	if flightStatus.ArrivalSourceID == appconst.AirportFlightStatusSource &&
		flightStatus.ArrivalStatus != string(appconst.FlightStatusCancelled) {
		s.updateDelayedFlightsByTime(
			flightStatus.ArrivalStation,
			scheduledArrival,
			flightStatus.ArrivalTimeActual,
			120,
			GetFlightTitleKey(&flightStatus),
		)
	}
}

func (s *statusStorageImpl) updateDelayedFlightsByTime(
	station int64, timeScheduledString, timeActualString string, minutesInFuture int, flightTitle string) {
	if station == 0 || timeScheduledString == "" || timeActualString == "" {
		return
	}

	tzStation := s.tzProvider.GetTimeZoneByStationID(station)
	if tzStation == nil {
		return
	}
	timeScheduled, err := dtutil.ParseDateTimeISO(timeScheduledString, tzStation)
	if err != nil {
		logger.Logger().Error(
			"Cannot parse scheduled datetime from flight status",
			log.String("timeScheduled", timeScheduledString),
			log.String("flightTitle", flightTitle),
			log.Int64("station", station),
		)
		return
	}
	timeActual, err := dtutil.ParseDateTimeISO(timeActualString, tzStation)
	if err != nil {
		logger.Logger().Error(
			"Cannot parse actual datetime from flight status",
			log.String("timeActual", timeActualString),
			log.String("flightTitle", flightTitle),
			log.Int64("station", station),
		)
		return
	}
	if int(timeActual.Sub(timeScheduled)/time.Minute) <= 30 {
		return
	}

	scheduledDiff := int(s.ReferenceTime.Sub(timeScheduled.In(time.UTC)) / time.Minute)
	actualDiff := int(s.ReferenceTime.Sub(timeActual.In(time.UTC)) / time.Minute)

	// positive means "this is time in the past", negative means "in the future"
	if (scheduledDiff >= 0 && actualDiff < 0) || scheduledDiff < 0 && scheduledDiff >= -minutesInFuture {
		if _, ok := s.delayedFlights[station]; !ok {
			s.delayedFlights[station] = make(map[string]emptyType)
		}
		s.delayedFlights[station][flightTitle] = empty
	}
}

func (s *statusStorageImpl) updateCancelledFlightsByTime(station int64, timeScheduledString, flightTitle string) {
	if station == 0 || timeScheduledString == "" {
		return
	}
	tzStation := s.tzProvider.GetTimeZoneByStationID(station)
	if tzStation == nil {
		return
	}
	timeScheduled, err := dtutil.ParseDateTimeISO(timeScheduledString, tzStation)
	if err != nil {
		logger.Logger().Error(
			"Cannot parse datetime from flight status",
			log.String("timeScheduled", timeScheduledString),
			log.String("flightTitle", flightTitle),
			log.Int64("station", station),
		)
		return
	}
	timeDiff := int(s.ReferenceTime.Sub(timeScheduled.In(time.UTC)) / time.Minute)
	if timeDiff > -24*60 && timeDiff < 120 {
		if _, ok := s.cancelledFlights[station]; !ok {
			s.cancelledFlights[station] = make(map[string]emptyType)
		}
		s.cancelledFlights[station][flightTitle] = empty
	}
}

func (s *statusStorageImpl) GetDelayedFlights(station int64) (map[string]emptyType, bool) {
	result, hasData := s.delayedFlights[station]
	return result, hasData
}

func (s *statusStorageImpl) GetCancelledFlights(station int64) (map[string]emptyType, bool) {
	result, hasData := s.cancelledFlights[station]
	return result, hasData
}

func (s *statusStorageImpl) GetDelayedOrCancelledStations() []int64 {
	stationsMap := make(map[int64]bool)
	for station := range s.delayedFlights {
		stationsMap[station] = true
	}
	for station := range s.cancelledFlights {
		stationsMap[station] = true
	}
	result := make([]int64, 0)
	for station := range stationsMap {
		result = append(result, station)
	}
	sort.Slice(result, func(i, j int) bool {
		return result[i] < result[j]
	})
	return result
}

func (s *statusStorageImpl) getOrCreateFlightStatuses(bucketKey string) *SynchronizedFlightStatusMap {
	flightStatuses, hasValue := s.GetFlightStatuses(bucketKey)
	if !hasValue {
		flightStatuses = NewSynchronizedFlightStatusMap()
		s.putFlightStatuses(bucketKey, flightStatuses)
	}
	return flightStatuses
}

func (s *statusStorageImpl) GetFlightStatuses(bucketKey string) (*SynchronizedFlightStatusMap, bool) {
	s.flightStatusMutex.RLock()
	defer s.flightStatusMutex.RUnlock()
	m, ok := s.FlightStatuses[bucketKey]
	return m, ok
}

func (s *statusStorageImpl) putFlightStatuses(bucketKey string, statuses *SynchronizedFlightStatusMap) {
	s.flightStatusMutex.Lock()
	defer s.flightStatusMutex.Unlock()
	s.FlightStatuses[bucketKey] = statuses
}

func (s *statusStorageImpl) PutStatusSource(flightStatusSource structs.FlightStatusSource) {
	s.StatusSources[flightStatusSource.ID] = flightStatusSource
}

func (s *statusStorageImpl) GetStatusSources() map[int32]structs.FlightStatusSource {
	return s.StatusSources
}

func GetFlightStatusKey(flightStatus *structs.FlightStatus) string {
	return fmt.Sprintf("%v.%v.%v", flightStatus.AirlineID, flightStatus.FlightNumber, flightStatus.LegNumber)
}

// For delays/cancellations maps leg number and carrier ID do not matter
func GetFlightTitleKey(flightStatus *structs.FlightStatus) string {
	return fmt.Sprintf("%v.%v", flightStatus.CarrierCode, flightStatus.FlightNumber)
}

func NewSynchronizedFlightStatusMap() *SynchronizedFlightStatusMap {
	return &SynchronizedFlightStatusMap{
		RWMutex:  sync.RWMutex{},
		statuses: make(map[dtutil.IntDate]*structs.FlightStatus),
	}
}

func (sfsm *SynchronizedFlightStatusMap) putStatus(
	status structs.FlightStatus,
	statusSources map[int32]structs.FlightStatusSource) structs.FlightStatus {
	sfsm.Lock()
	defer sfsm.Unlock()
	statusKey := dtutil.StringDate(status.FlightDate).ToIntDate()
	currentStatus, ok := sfsm.statuses[statusKey]
	if !ok {
		sfsm.statuses[statusKey] = &status
		return status
	}
	// Merge current and new statuses
	currentDepartureSource, currentOk := statusSources[currentStatus.DepartureSourceID]
	newDepartureSource, newOk := statusSources[status.DepartureSourceID]
	if !currentOk {
		currentDepartureSource = zeroPrioritySource
	}
	if !newOk {
		newDepartureSource = zeroPrioritySource
	}
	/*
		Если у источников статусов нет равных приоритетов, то не важно, сравнивать по > или >=, в силу отсутствия случая равенства.
		Если равные приоритеты возможны, то нужно дополнительно сравнивать время апдейта соответствующей половинки статуса;
		сейчас статусы в файле упорядочены от более свежих к более старым, но т.к. это линейный список, то нельзя сделать так,
		чтобы и половинки отправлений, и половинки прибытий были так упорядочены;
		возиться со сравнением времён не хочется, кажется более продуктивным считать, что приоритеты не могут быть одинаковыми.
	*/
	if isUnknown(currentStatus.DepartureStatus, currentStatus.DepartureTimeActual) ||
		(newDepartureSource.Priority > currentDepartureSource.Priority && !isUnknown(status.DepartureStatus, status.DepartureTimeActual)) ||
		(newDepartureSource.Priority == currentDepartureSource.Priority && currentStatus.DepartureReceivedAtUtc < status.DepartureReceivedAtUtc) {
		updateDepartureFields(currentStatus, &status)
	}
	currentArrivalSource, currentOk := statusSources[currentStatus.ArrivalSourceID]
	newArrivalSource, newOk := statusSources[status.ArrivalSourceID]
	if !currentOk {
		currentArrivalSource = zeroPrioritySource
	}
	if !newOk {
		newArrivalSource = zeroPrioritySource
	}
	if isUnknown(currentStatus.ArrivalStatus, currentStatus.ArrivalTimeActual) ||
		(newArrivalSource.Priority > currentArrivalSource.Priority && !isUnknown(status.ArrivalStatus, status.ArrivalTimeActual)) ||
		(newArrivalSource.Priority == currentArrivalSource.Priority && currentStatus.ArrivalReceivedAtUtc < status.ArrivalReceivedAtUtc) {
		updateArrivalFields(currentStatus, &status)
	}
	sfsm.statuses[statusKey] = currentStatus
	return *currentStatus
}

func (sfsm *SynchronizedFlightStatusMap) GetStatus(date dtutil.IntDate) (*structs.FlightStatus, bool) {
	if sfsm == nil {
		return nil, false
	}
	sfsm.RLock()
	defer sfsm.RUnlock()
	s, ok := sfsm.statuses[date]
	return s, ok
}

func (sfsm *SynchronizedFlightStatusMap) Len() int {
	sfsm.RLock()
	defer sfsm.RUnlock()
	return len(sfsm.statuses)
}

func (sfsm *SynchronizedFlightStatusMap) Keys() chan dtutil.IntDate {
	dates := make(chan dtutil.IntDate)
	go func() {
		sfsm.RLock()
		defer sfsm.RUnlock()
		defer close(dates)
		for k := range sfsm.statuses {
			dates <- k
		}
	}()
	return dates
}

func isUnknown(statusText, timeActual string) bool {
	return statusText == "" || ((statusText == "no-data" || statusText == "unknown") && timeActual == "")
}

func updateDepartureFields(currentStatus, status *structs.FlightStatus) {
	currentStatus.DepartureTimeActual = status.DepartureTimeActual
	currentStatus.DepartureTimeScheduled = status.DepartureTimeScheduled
	currentStatus.DepartureStatus = status.DepartureStatus
	currentStatus.DepartureGate = status.DepartureGate
	currentStatus.DepartureTerminal = status.DepartureTerminal
	currentStatus.DepartureDiverted = status.DepartureDiverted
	currentStatus.DepartureDivertedAirportCode = status.DepartureDivertedAirportCode
	currentStatus.DepartureCreatedAtUtc = status.DepartureCreatedAtUtc
	currentStatus.DepartureReceivedAtUtc = status.DepartureReceivedAtUtc
	currentStatus.DepartureUpdatedAtUtc = status.DepartureUpdatedAtUtc
	currentStatus.CheckInDesks = status.CheckInDesks
	currentStatus.DepartureSourceID = status.DepartureSourceID
	if status.DepartureStation > 0 {
		currentStatus.DepartureStation = status.DepartureStation
	}
	updateTimstamp(currentStatus)
}

func updateArrivalFields(currentStatus, status *structs.FlightStatus) {
	currentStatus.ArrivalTimeActual = status.ArrivalTimeActual
	currentStatus.ArrivalTimeScheduled = status.ArrivalTimeScheduled
	currentStatus.ArrivalStatus = status.ArrivalStatus
	currentStatus.ArrivalGate = status.ArrivalGate
	currentStatus.ArrivalTerminal = status.ArrivalTerminal
	currentStatus.ArrivalDiverted = status.ArrivalDiverted
	currentStatus.ArrivalDivertedAirportCode = status.ArrivalDivertedAirportCode
	currentStatus.ArrivalCreatedAtUtc = status.ArrivalCreatedAtUtc
	currentStatus.ArrivalReceivedAtUtc = status.ArrivalReceivedAtUtc
	currentStatus.ArrivalUpdatedAtUtc = status.ArrivalUpdatedAtUtc
	currentStatus.BaggageCarousels = status.BaggageCarousels
	if status.ArrivalStation > 0 {
		currentStatus.ArrivalStation = status.ArrivalStation
	}
	currentStatus.ArrivalSourceID = status.ArrivalSourceID
	updateTimstamp(currentStatus)
}

func updateTimstamp(currentStatus *structs.FlightStatus) {
	currentStatus.UpdatedAtUtc = currentStatus.DepartureUpdatedAtUtc
	if currentStatus.UpdatedAtUtc < currentStatus.ArrivalUpdatedAtUtc {
		currentStatus.UpdatedAtUtc = currentStatus.ArrivalUpdatedAtUtc
	}
}
