package storage

import (
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/services/storage/segment"
	tzutils "a.yandex-team.ru/travel/avia/shared_flights/api/internal/services/storage/timezone"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/storage/carrier"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/storage/flight"
	popularityscores "a.yandex-team.ru/travel/avia/shared_flights/api/internal/storage/popularity_scores"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/storage/station"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/storage/status"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/storage/timezone"
	transportmodel "a.yandex-team.ru/travel/avia/shared_flights/api/internal/storage/transport_model"
	"a.yandex-team.ru/travel/avia/shared_flights/api/pkg/structs"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/dtutil"
	iatacorrector "a.yandex-team.ru/travel/avia/shared_flights/lib/go/iata_correction"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/logger"
	"a.yandex-team.ru/travel/proto/dicts/rasp"
	"a.yandex-team.ru/travel/proto/shared_flights/snapshots"
)

type Storage struct {
	flightStorage              flight.FlightStorage
	stationStorage             station.StationStorage
	carrierStorage             *carrier.CarrierStorage
	statusStorage              status.StatusStorage
	stationStatusSourceStorage *status.StationStatusSourceStorage
	blacklistRuleStorage       flight.BlacklistRuleStorage
	flightMergeRuleStorage     flight.FlightMergeRuleStorage
	iataCorrector              iatacorrector.IataCorrector
	timeZoneStorage            timezone.TimeZoneStorage
	transportModelStorage      transportmodel.TransportModelStorage
	carriersPopularityScores   *popularityscores.CarriersPopularityScores
	IsAvailable                bool
	nextFlightPatternID        int32
	lastImportedDate           string

	// Cache dependent data
	FlightBoard station.FlightBoardStorage
}

func (s *Storage) StationStatusSourceStorage() *status.StationStatusSourceStorage {
	return s.stationStatusSourceStorage
}

func (s *Storage) CarrierStorage() *carrier.CarrierStorage {
	return s.carrierStorage
}

// Copy makes a shallow-ish copy of storage. It doesnt make a copy of underlying maps or slices,
// but it calls Copy on internal storages.
func (s *Storage) Copy() *Storage {
	flightStorage := s.flightStorage.Copy()
	flightBoard := station.NewFlightBoard(flightStorage)
	return &Storage{
		flightStorage:              flightStorage, // guts swapped by delta
		stationStorage:             s.stationStorage,
		carrierStorage:             s.carrierStorage,
		statusStorage:              s.statusStorage,              // swapped by delta
		stationStatusSourceStorage: s.stationStatusSourceStorage, // swapped by delta
		blacklistRuleStorage:       s.blacklistRuleStorage,       // swapped by delta
		flightMergeRuleStorage:     s.flightMergeRuleStorage,     // swapped by delta
		iataCorrector:              s.iataCorrector,
		timeZoneStorage:            s.timeZoneStorage,
		transportModelStorage:      s.transportModelStorage,
		carriersPopularityScores:   s.carriersPopularityScores,
		IsAvailable:                s.IsAvailable,
		nextFlightPatternID:        s.nextFlightPatternID,
		lastImportedDate:           s.lastImportedDate,
		FlightBoard:                flightBoard,
	}
}

func NewStorage() *Storage {
	return NewStorageWithStartDate("")
}

func NewStorageWithStartDate(startDate string) *Storage {
	iataCorrector := iatacorrector.NewIataCorrector(nil, nil, nil)
	carrierStorage := carrier.NewCarrierStorage()
	instance := Storage{
		flightStorage:            flight.NewFlightStorage(iataCorrector, carrierStorage, startDate),
		iataCorrector:            iataCorrector,
		stationStorage:           station.NewStationStorage(),
		timeZoneStorage:          timezone.NewTimeZoneStorage(),
		transportModelStorage:    transportmodel.NewTransportModelStorage(),
		carrierStorage:           carrierStorage,
		carriersPopularityScores: popularityscores.NewCarriersPopularityScores(),
		IsAvailable:              false,
		nextFlightPatternID:      flight.MinIataCorrectedFlightPatternID,
		lastImportedDate:         "",
	}
	tzProvider := tzutils.NewTimeZoneProvider(instance.timeZoneStorage, instance.stationStorage)
	instance.statusStorage = status.NewStatusStorage(tzProvider)
	instance.blacklistRuleStorage = flight.NewBlacklistRuleStorage(instance.stationStorage)
	instance.flightMergeRuleStorage = flight.NewFlightMergeRuleStorage()
	instance.FlightBoard = station.NewFlightBoard(instance.flightStorage)
	return &instance
}

func (s *Storage) PutStation(station *snapshots.TStationWithCodes) {
	s.stationStorage.PutStation(station)
}

func (s *Storage) UpdateCacheDependentData() error {
	if err := s.FlightBoard.PutFlightPatterns(s.flightStorage.GetFlightPatterns(), s.flightStorage.GetDopFlightPatterns()); err != nil {
		return xerrors.Errorf("cannot update flight board: %w", err)
	}
	return nil
}

func (s *Storage) Timezones() timezone.TimeZoneStorage {
	return s.timeZoneStorage
}

func (s *Storage) TransportModels() transportmodel.TransportModelStorage {
	return s.transportModelStorage
}

func (s *Storage) CarriersPopularityScores() *popularityscores.CarriersPopularityScores {
	return s.carriersPopularityScores
}

func (s *Storage) Stations() station.StationStorage {
	return s.stationStorage
}

func (s *Storage) FlightStorage() flight.FlightStorage {
	return s.flightStorage
}

func (s *Storage) StatusStorage() status.StatusStorage {
	return s.statusStorage
}

func (s *Storage) SetStatusStorage(newStatusStorage status.StatusStorage) {
	s.statusStorage = newStatusStorage
}

func (s *Storage) BlacklistRuleStorage() flight.BlacklistRuleStorage {
	return s.blacklistRuleStorage
}

func (s *Storage) SetBlacklistRuleStorage(newBlacklistRuleStorage flight.BlacklistRuleStorage) {
	s.blacklistRuleStorage = newBlacklistRuleStorage
}

func (s *Storage) FlightMergeRuleStorage() flight.FlightMergeRuleStorage {
	return s.flightMergeRuleStorage
}

func (s *Storage) SetFlightMergeRuleStorage(newFlightMergeRuleStorage flight.FlightMergeRuleStorage) {
	s.flightMergeRuleStorage = newFlightMergeRuleStorage
}

func (s *Storage) PutFlightBase(flightBase structs.FlightBase) {
	if flightBase.OperatingCarrierCode != "" {
		s.carrierStorage.PutCarrier(flightBase.OperatingCarrier, flightBase.OperatingCarrierCode, "", "", "")
	} else {
		flightBase.OperatingCarrierCode = s.carrierStorage.GetCarrierCodeByID(flightBase.OperatingCarrier)
	}
	s.flightStorage.PutFlightBase(flightBase)
}

func (s *Storage) UpdateP2PCache(entry *snapshots.TP2PCacheEntry) {
	s.flightStorage.UpdateP2PCache(entry)
}

func (s *Storage) UpdateTransportModelsCache(transportModel *rasp.TTransportModel) {
	s.transportModelStorage.PutTransportModel(transportModel)
}

func (s *Storage) PutFlightPattern(flightPattern structs.FlightPattern) {
	flyingCarrierIata := ""
	fb, err := s.flightStorage.GetFlightBase(flightPattern.FlightBaseID, flightPattern.IsDop)
	if err == nil {
		flyingCarrierIata = fb.FlyingCarrierIata
	}

	if !flightPattern.IsCodeshare {
		sirenaCarrier := s.carrierStorage.GetSirenaByID(flightPattern.MarketingCarrier)
		correctedCarrierID := s.IataCorrector().FindFlightPatternCarrier(&flightPattern, flyingCarrierIata, sirenaCarrier)
		if correctedCarrierID != 0 {
			correctedCarrierIata := s.carrierStorage.GetCarrierCodeByID(correctedCarrierID)
			if correctedCarrierIata == flightPattern.MarketingCarrierCode {
				// See RASPTICKETS-17651 for details
				flightPattern.FilingCarrier = flightPattern.MarketingCarrier
				flightPattern.FilingCarrierCode = flightPattern.MarketingCarrierCode
				flightPattern.MarketingCarrier = correctedCarrierID
				if !s.updateFlightBase(flightPattern.FlightBaseID, correctedCarrierID, correctedCarrierIata) {
					return
				}
			} else {
				correctedFlightPattern := flightPattern
				s.nextFlightPatternID++
				correctedFlightPattern.ID = s.nextFlightPatternID
				correctedFlightPattern.MarketingCarrier = correctedCarrierID
				correctedFlightPattern.MarketingCarrierCode = correctedCarrierIata
				correctedFlightPattern.FilingCarrier = flightPattern.MarketingCarrier
				correctedFlightPattern.FilingCarrierCode = flightPattern.MarketingCarrierCode
				flightPattern.CorrectedCarrier = correctedCarrierID

				flightPattern.IsCodeshare = true
				flightPattern.OperatingFlightPatternID = correctedFlightPattern.ID
				if !s.updateFlightBase(flightPattern.FlightBaseID, correctedCarrierID, correctedCarrierIata) {
					return
				}
				s.flightStorage.PutFlightPattern(correctedFlightPattern)
			}
		}
	}

	if flightPattern.MarketingCarrierCode != "" {
		s.carrierStorage.PutCarrier(flightPattern.MarketingCarrier, flightPattern.MarketingCarrierCode, "", "", "")
	} else {
		flightPattern.MarketingCarrierCode = s.carrierStorage.GetCarrierCodeByID(flightPattern.MarketingCarrier)
	}
	s.flightStorage.PutFlightPattern(flightPattern)
}

func (s *Storage) updateFlightBase(flightBaseID, carrierID int32, carrierCode string) bool {
	flightBase, err := s.flightStorage.GetFlightBase(flightBaseID, false)
	if err != nil {
		logger.Logger().Error(
			"Invalid flight base ID",
			log.Int32("id", flightBaseID),
			log.Int32("carrierID", carrierID),
			log.String("carrierCode", carrierCode),
			log.Error(err),
		)
		return false
	}
	flightBase.OperatingCarrier = carrierID
	flightBase.OperatingCarrierCode = carrierCode
	s.flightStorage.PutFlightBase(flightBase)
	return true
}

func (s *Storage) IataCorrector() iatacorrector.IataCorrector {
	return s.iataCorrector
}

func (s *Storage) SetIataCorrector(newIataCorrector iatacorrector.IataCorrector) {
	s.iataCorrector = newIataCorrector
}

func (s *Storage) UpdateLastImported(lastImportedDate string) {
	// Cache the earliest synce date across all data sources
	if s.lastImportedDate == "" || lastImportedDate < s.lastImportedDate {
		s.lastImportedDate = lastImportedDate
	}
}

func (s *Storage) GetLastImported() string {
	return s.lastImportedDate
}

func (s *Storage) BuildAeroflotCache() error {
	return s.flightStorage.AeroflotCache().Rebuild()
}

func (s *Storage) SetStationStatusSourceStorage(storage *status.StationStatusSourceStorage) {
	s.stationStatusSourceStorage = storage
}

type SegmentProcessor struct {
	OnSegment func(fb structs.FlightBase, fp structs.FlightPattern)
	OnError   func(err error)
}

func (s *Storage) AttachFlightBases(
	patterns []*structs.FlightPattern,
	showBanned bool,
	showCodeshares bool,
	nationalVersion string,
) (segments []segment.Segment, err error) {
	segmentProcessor := SegmentProcessor{
		OnSegment: func(fb structs.FlightBase, fp structs.FlightPattern) {
			segments = append(segments, segment.Segment{
				FlightBase:    fb,
				FlightPattern: fp,
			})
		},
		OnError: func(e error) {
			err = e
		},
	}
	s.iterateOverFlightBases(patterns, showBanned, showCodeshares, nationalVersion, segmentProcessor)
	return
}

func (s *Storage) FetchStations(
	patterns []*structs.FlightPattern,
	showBanned bool,
	showCodeshares bool,
	nationalVersion string,
) (stations []*snapshots.TStationWithCodes, err error) {
	stationsMap := make(map[int64]bool)
	segmentProcessor := SegmentProcessor{
		OnSegment: func(fb structs.FlightBase, _ structs.FlightPattern) {
			stationsMap[fb.DepartureStation] = true
			stationsMap[fb.ArrivalStation] = true
		},
		OnError: func(e error) {
			err = e
		},
	}
	s.iterateOverFlightBases(patterns, showBanned, showCodeshares, nationalVersion, segmentProcessor)
	if err != nil {
		return
	}
	stations = make([]*snapshots.TStationWithCodes, 0, len(stationsMap))
	for stationID := range stationsMap {
		station, ok := s.stationStorage.ByID(stationID)
		if ok {
			stations = append(stations, station)
		}
	}
	return
}

func (s *Storage) iterateOverFlightBases(
	patterns []*structs.FlightPattern,
	showBanned bool,
	showCodeshares bool,
	nationalVersion string,
	segmentProcessor SegmentProcessor,
) {
	for _, pattern := range patterns {
		if pattern.IsCodeshare && !showCodeshares {
			continue
		}
		var flightBase structs.FlightBase
		var err error
		if flightBase, err = s.FlightStorage().GetFlightBase(pattern.FlightBaseID, pattern.IsDop); err != nil {
			segmentProcessor.OnError(xerrors.Errorf("cannot get flight base: %w", err))
			return
		}
		if showBanned {
			segmentProcessor.OnSegment(flightBase, *pattern)
			continue
		}

		bannedFrom, bannedUntil, isBanned := s.BlacklistRuleStorage().GetBannedDates(flightBase, pattern, nationalVersion)

		if isBanned {
			if bannedFrom == "" {
				// flight is banned regardless of dates
				continue
			}
			intervals := dtutil.Intersect(
				dtutil.NewStringDateInterval(pattern.OperatingFromDate, pattern.OperatingUntilDate),
				dtutil.NewStringDateInterval(bannedFrom, bannedUntil),
			)
			for _, interval := range intervals {
				startDate := interval.From.ToTime(time.UTC)
				endDate := interval.To.ToTime(time.UTC)
				intervalHasOperatingDay := false
				for flightDate := startDate; flightDate.Before(endDate) || flightDate.Equal(endDate); flightDate = flightDate.AddDate(0, 0, 1) {
					if dtutil.OperatesOn(pattern.OperatingOnDays, flightDate.Weekday()) {
						intervalHasOperatingDay = true
						break
					}
				}
				if intervalHasOperatingDay {
					intervalFlightPattern := *pattern
					intervalFlightPattern.OperatingFromDate = string(interval.From)
					intervalFlightPattern.OperatingUntilDate = string(interval.To)
					segmentProcessor.OnSegment(flightBase, intervalFlightPattern)
					continue
				}
			}
		} else {
			segmentProcessor.OnSegment(flightBase, *pattern)
		}
	}
}
