package timezone

import (
	"sync"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/logger"
	"a.yandex-team.ru/travel/proto/dicts/rasp"
)

type TimeZoneStorage interface {
	PutTimezone(*rasp.TTimeZone)
	GetTimeZone(station *rasp.TStation) *time.Location
}

type timeZoneStorageImpl struct {
	timeZoneCodeByTimeZoneID map[int32]string
	zoneCodeLock             sync.RWMutex
	locationByStationID      map[int32]*time.Location
	locationLock             sync.RWMutex
}

func NewTimeZoneStorage() TimeZoneStorage {
	return &timeZoneStorageImpl{
		make(map[int32]string),
		sync.RWMutex{},
		make(map[int32]*time.Location),
		sync.RWMutex{},
	}
}
func (s *timeZoneStorageImpl) PutTimezone(timezone *rasp.TTimeZone) {
	s.zoneCodeLock.Lock()
	defer s.zoneCodeLock.Unlock()
	s.timeZoneCodeByTimeZoneID[timezone.Id] = timezone.Code

}

func (s *timeZoneStorageImpl) getTimeZoneCode(timezoneID int32) (string, bool) {
	s.zoneCodeLock.RLock()
	defer s.zoneCodeLock.RUnlock()
	code, ok := s.timeZoneCodeByTimeZoneID[timezoneID]
	return code, ok

}

func (s *timeZoneStorageImpl) GetTimeZone(station *rasp.TStation) *time.Location {
	stationID := station.Id
	s.locationLock.RLock()
	location, ok := s.locationByStationID[stationID]
	s.locationLock.RUnlock()
	if ok {
		return location
	}

	// TODO(mikhailche): Проверить, можно ли использовать timezone из протобуфа самой станции
	stationTimeZoneID := station.TimeZoneId

	stationTimeZoneCode, hasTimeZone := s.getTimeZoneCode(stationTimeZoneID)

	if !hasTimeZone {
		logger.Logger().Error("unable to find departure station time zone", log.Int32("time zone id", stationTimeZoneID))
		return nil
	}

	tz, err := time.LoadLocation(stationTimeZoneCode)
	if err != nil {
		logger.Logger().Error(
			"Cannot load time.Location for station",
			log.Int64("station id", int64(stationID)),
			log.String("timezone name", stationTimeZoneCode),
		)
		return nil
	}
	s.locationLock.Lock()
	defer s.locationLock.Unlock()
	s.locationByStationID[stationID] = tz
	return tz
}

func CompareOffsets(loc1, loc2 *time.Location, instant time.Time) bool {
	year, month, day := instant.Date()
	instant1 := time.Date(year, month, day, 0, 0, 0, 0, loc1)
	instant2 := time.Date(year, month, day, 0, 0, 0, 0, loc2)
	return instant1.Equal(instant2)
}
