package structs

import (
	"fmt"

	"a.yandex-team.ru/travel/proto/shared_flights/ssim"
)

type FlightBaseSource int8

const (
	UnknownSource FlightBaseSource = 0 + iota
	InnovataSource
	SirenaSource
	ApmSource
	DopSource
	AmadeusSource
	end
)

// FlightBase is based on TFlightBase protobuf
type FlightBase struct {
	ID                     int32            `json:"Id,omitempty"`
	OperatingCarrier       int32            `json:"Carrier,omitempty"`
	OperatingCarrierCode   string           `json:"CarrierCode,omitempty"`
	OperatingFlightNumber  string           `json:"FlightNumber,omitempty"`
	LegNumber              int32            `json:"LegNumber,omitempty"`
	DepartureStation       int64            `json:"DepartureStation,omitempty"`
	DepartureStationCode   string           `json:"DepartureStationCode,omitempty"`
	DepartureTimeScheduled int32            `json:"DepartureTimeScheduled,omitempty"`
	DepartureTerminal      string           `json:"DepartureTerminal,omitempty"`
	ArrivalStation         int64            `json:"ArrivalStation,omitempty"`
	ArrivalStationCode     string           `json:"ArrivalStationCode,omitempty"`
	ArrivalTimeScheduled   int32            `json:"ArrivalTimeScheduled,omitempty"`
	ArrivalTerminal        string           `json:"ArrivalTerminal,omitempty"`
	AircraftTypeID         int64            `json:"AircraftType,omitempty"`
	FlyingCarrierIata      string           `json:"FlyingCarrierIata,omitempty"`
	IntlDomesticStatus     string           `json:"IntlDomesticStatus,omitempty"`
	DesignatedCarrier      int32            `json:"DesignatedCarrier,omitempty"`
	Source                 FlightBaseSource `json:"Source,omitempty"`
}

func (fb FlightBase) String() string {
	return fmt.Sprintf("%#v", fb)
}

func FlightBaseFromProto(source *ssim.TFlightBase, stationResolver stationCodeResolver, carrierResolver carrierCodeResolver) FlightBase {
	return FlightBase{
		ID:                     source.Id,
		OperatingCarrier:       source.OperatingCarrier,
		OperatingCarrierCode:   carrierResolver(source.OperatingCarrier),
		OperatingFlightNumber:  source.OperatingFlightNumber,
		LegNumber:              source.LegSeqNumber,
		DepartureStation:       source.DepartureStation,
		DepartureStationCode:   stationResolver(source.DepartureStation),
		DepartureTimeScheduled: source.ScheduledDepartureTime,
		DepartureTerminal:      source.DepartureTerminal,
		ArrivalStation:         source.ArrivalStation,
		ArrivalStationCode:     stationResolver(source.ArrivalStation),
		ArrivalTimeScheduled:   source.ScheduledArrivalTime,
		ArrivalTerminal:        source.ArrivalTerminal,
		AircraftTypeID:         source.AircraftTypeId,
		FlyingCarrierIata:      source.FlyingCarrierIata,
		IntlDomesticStatus:     source.IntlDomesticStatus,
		DesignatedCarrier:      source.DesignatedCarrier,
		Source:                 GetSource(source.Source),
	}
}

func GetSource(protoSource ssim.EFlightBaseSource) FlightBaseSource {
	switch protoSource {
	case ssim.EFlightBaseSource_TYPE_INNOVATA:
		return InnovataSource
	case ssim.EFlightBaseSource_TYPE_SIRENA:
		return SirenaSource
	case ssim.EFlightBaseSource_TYPE_APM:
		return ApmSource
	case ssim.EFlightBaseSource_TYPE_DOP:
		return DopSource
	case ssim.EFlightBaseSource_TYPE_AMADEUS:
		return AmadeusSource
	default:
		return UnknownSource
	}
}

func (fb FlightBase) GetBucketKey() string {
	return fmt.Sprintf("%v.%v.%v", fb.OperatingCarrier, fb.OperatingFlightNumber, fb.LegNumber)
}
