package structs

import (
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/avia/shared_flights/api/internal/utils"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/logger"
	"a.yandex-team.ru/travel/proto/shared_flights/snapshots"
)

// FlightStatus is based on FlightStatus protobuf and extends it with some fields, that should be filled separately
type FlightStatus struct {
	AirlineID                    int64  `json:"AirlineId,omitempty"`
	CarrierCode                  string `json:"CarrierCode,omitempty"`
	FlightNumber                 string `json:"FlightNumber,omitempty"`
	LegNumber                    int32  `json:"LegNumber,omitempty"`
	FlightDate                   string `json:"FlightDate,omitempty"`
	StatusSourceID               int32  `json:"StatusSourceId,omitempty"` // Deprecated: Do not use.
	CreatedAtUtc                 string `json:"CreatedAtUtc,omitempty"`
	UpdatedAtUtc                 string `json:"UpdatedAtUtc,omitempty"`
	DepartureTimeActual          string `json:"DepartureTimeActual,omitempty"`
	DepartureTimeScheduled       string `json:"DepartureTimeScheduled,omitempty"`
	DepartureStatus              string `json:"DepartureStatus,omitempty"`
	DepartureGate                string `json:"DepartureGate,omitempty"`
	DepartureTerminal            string `json:"DepartureTerminal,omitempty"`
	DepartureDiverted            bool   `json:"DepartureDiverted,omitempty"`
	DepartureDivertedAirportCode string `json:"DepartureDivertedAirportCode,omitempty"`
	DepartureDivertedAirportID   int32  `json:"DepartureDivertedAirportID,omitempty"` // use FillDivertedAirportIDs
	DepartureCreatedAtUtc        string `json:"DepartureCreatedAtUtc,omitempty"`
	DepartureReceivedAtUtc       string `json:"DepartureReceivedAtUtc,omitempty"`
	DepartureUpdatedAtUtc        string `json:"DepartureUpdatedAtUtc,omitempty"`
	ArrivalTimeActual            string `json:"ArrivalTimeActual,omitempty"`
	ArrivalTimeScheduled         string `json:"ArrivalTimeScheduled,omitempty"`
	ArrivalStatus                string `json:"ArrivalStatus,omitempty"`
	ArrivalGate                  string `json:"ArrivalGate,omitempty"`
	ArrivalTerminal              string `json:"ArrivalTerminal,omitempty"`
	ArrivalDiverted              bool   `json:"ArrivalDiverted,omitempty"`
	ArrivalDivertedAirportCode   string `json:"ArrivalDivertedAirportCode,omitempty"`
	ArrivalDivertedAirportID     int32  `json:"ArrivalDivertedAirportID,omitempty"` // use FillDivertedAirportIDs
	ArrivalCreatedAtUtc          string `json:"ArrivalCreatedAtUtc,omitempty"`
	ArrivalReceivedAtUtc         string `json:"ArrivalReceivedAtUtc,omitempty"`
	ArrivalUpdatedAtUtc          string `json:"ArrivalUpdatedAtUtc,omitempty"`
	CheckInDesks                 string `json:"CheckInDesks,omitempty"`
	BaggageCarousels             string `json:"BaggageCarousels,omitempty"`
	DepartureStation             int64  `json:"DepartureStation,omitempty"`
	ArrivalStation               int64  `json:"ArrivalStation,omitempty"`
	DepartureSourceID            int32  `json:"DepartureSourceId,omitempty"`
	DepartureSourceIsTrusted     bool
	ArrivalSourceID              int32 `json:"ArrivalSourceId,omitempty"`
	ArrivalSourceIsTrusted       bool
}

type stationID = int32
type stationDecoder func(string) (stationID, bool)

func (fs *FlightStatus) FillDivertedAirportIDs(stationDecoders ...stationDecoder) {
	var decoded bool
	if decoded = fs.fillDivertedAirportID(fs.ArrivalDivertedAirportCode, &fs.ArrivalDivertedAirportID, stationDecoders...); !decoded {
		logger.Logger().Warn(
			"Could not decode arrival flight status station",
			log.String("station", fs.ArrivalDivertedAirportCode),
			log.Reflect("status", fs),
		)
	}
	if decoded = fs.fillDivertedAirportID(fs.DepartureDivertedAirportCode, &fs.DepartureDivertedAirportID, stationDecoders...); !decoded {
		logger.Logger().Warn(
			"Could not decode departure flight status station",
			log.String("station", fs.DepartureDivertedAirportCode),
			log.Reflect("status", fs),
		)
	}
}

type trustVerifier interface {
	IsTrusted(stationID int64, statusSourceID int64) bool
}

func (fs *FlightStatus) FillSourceTrust(trustVerifier trustVerifier) {
	if fs.DepartureSourceID > 0 {
		fs.DepartureSourceIsTrusted = trustVerifier.IsTrusted(fs.DepartureStation, int64(fs.DepartureSourceID))
	}
	if fs.ArrivalSourceID > 0 {
		fs.ArrivalSourceIsTrusted = trustVerifier.IsTrusted(fs.ArrivalStation, int64(fs.ArrivalSourceID))
	}
}

func (fs *FlightStatus) fillDivertedAirportID(
	stationCode string,
	stationID *int32,
	stationDecoders ...stationDecoder,
) bool {
	if len(stationCode) == 0 {
		return true
	}
	for _, decoder := range stationDecoders {
		if decoder == nil {
			continue
		}
		id, exists := decoder(stationCode)
		if exists {
			*stationID = id
			return true
		}
	}
	return false
}

func (fs *FlightStatus) DepartureStatusExists() bool {
	return fs.DepartureCreatedAtUtc != ""
}

func (fs *FlightStatus) ArrivalStatusExists() bool {
	return fs.ArrivalCreatedAtUtc != ""
}

func (fs *FlightStatus) DepartureCopy() FlightStatus {
	return FlightStatus{
		AirlineID:                    fs.AirlineID,
		CarrierCode:                  fs.CarrierCode,
		FlightNumber:                 fs.FlightNumber,
		LegNumber:                    fs.LegNumber,
		FlightDate:                   fs.FlightDate,
		StatusSourceID:               fs.StatusSourceID,
		CreatedAtUtc:                 fs.CreatedAtUtc,
		UpdatedAtUtc:                 fs.UpdatedAtUtc,
		DepartureTimeActual:          fs.DepartureTimeActual,
		DepartureTimeScheduled:       fs.DepartureTimeScheduled,
		DepartureStatus:              fs.DepartureStatus,
		DepartureGate:                fs.DepartureGate,
		DepartureTerminal:            fs.DepartureTerminal,
		DepartureDiverted:            fs.DepartureDiverted,
		DepartureDivertedAirportCode: fs.DepartureDivertedAirportCode,
		DepartureDivertedAirportID:   fs.DepartureDivertedAirportID,
		DepartureCreatedAtUtc:        fs.DepartureCreatedAtUtc,
		DepartureReceivedAtUtc:       fs.DepartureReceivedAtUtc,
		DepartureUpdatedAtUtc:        fs.DepartureUpdatedAtUtc,
		CheckInDesks:                 fs.CheckInDesks,
		BaggageCarousels:             fs.BaggageCarousels,
		DepartureStation:             fs.DepartureStation,
		ArrivalStation:               fs.ArrivalStation,
		DepartureSourceID:            fs.DepartureSourceID,
		DepartureSourceIsTrusted:     fs.DepartureSourceIsTrusted,
	}
}

func (fs *FlightStatus) ArrivalCopy() FlightStatus {
	return FlightStatus{
		AirlineID:                  fs.AirlineID,
		CarrierCode:                fs.CarrierCode,
		FlightNumber:               fs.FlightNumber,
		LegNumber:                  fs.LegNumber,
		FlightDate:                 fs.FlightDate,
		StatusSourceID:             fs.StatusSourceID,
		CreatedAtUtc:               fs.CreatedAtUtc,
		UpdatedAtUtc:               fs.UpdatedAtUtc,
		ArrivalTimeActual:          fs.ArrivalTimeActual,
		ArrivalTimeScheduled:       fs.ArrivalTimeScheduled,
		ArrivalStatus:              fs.ArrivalStatus,
		ArrivalGate:                fs.ArrivalGate,
		ArrivalTerminal:            fs.ArrivalTerminal,
		ArrivalDiverted:            fs.ArrivalDiverted,
		ArrivalDivertedAirportCode: fs.ArrivalDivertedAirportCode,
		ArrivalDivertedAirportID:   fs.ArrivalDivertedAirportID,
		ArrivalCreatedAtUtc:        fs.ArrivalCreatedAtUtc,
		ArrivalReceivedAtUtc:       fs.ArrivalReceivedAtUtc,
		ArrivalUpdatedAtUtc:        fs.ArrivalUpdatedAtUtc,
		CheckInDesks:               fs.CheckInDesks,
		BaggageCarousels:           fs.BaggageCarousels,
		DepartureStation:           fs.DepartureStation,
		ArrivalStation:             fs.ArrivalStation,
		ArrivalSourceID:            fs.ArrivalSourceID,
		ArrivalSourceIsTrusted:     fs.ArrivalSourceIsTrusted,
	}
}
func truncateDate(date string) string {
	if len(date) > 10 {
		return date[:10]
	}
	return date
}

func FlightStatusFromProto(source *snapshots.TFlightStatus, target *FlightStatus) {
	*target = FlightStatus{
		AirlineID:                    source.AirlineId,
		CarrierCode:                  source.CarrierCode,
		FlightNumber:                 utils.TrimZeroesFromFlightNumber(source.FlightNumber),
		LegNumber:                    source.LegNumber,
		FlightDate:                   truncateDate(source.FlightDate),
		StatusSourceID:               source.StatusSourceId,
		CreatedAtUtc:                 source.CreatedAtUtc,
		UpdatedAtUtc:                 source.UpdatedAtUtc,
		DepartureTimeActual:          source.DepartureTimeActual,
		DepartureTimeScheduled:       source.DepartureTimeScheduled,
		DepartureStatus:              source.DepartureStatus,
		DepartureGate:                source.DepartureGate,
		DepartureTerminal:            source.DepartureTerminal,
		DepartureDiverted:            source.DepartureDiverted,
		DepartureDivertedAirportCode: source.DepartureDivertedAirportCode,
		DepartureCreatedAtUtc:        source.DepartureCreatedAtUtc,
		DepartureReceivedAtUtc:       source.DepartureReceivedAtUtc,
		DepartureUpdatedAtUtc:        source.DepartureUpdatedAtUtc,
		ArrivalTimeActual:            source.ArrivalTimeActual,
		ArrivalTimeScheduled:         source.ArrivalTimeScheduled,
		ArrivalStatus:                source.ArrivalStatus,
		ArrivalGate:                  source.ArrivalGate,
		ArrivalTerminal:              source.ArrivalTerminal,
		ArrivalDiverted:              source.ArrivalDiverted,
		ArrivalDivertedAirportCode:   source.ArrivalDivertedAirportCode,
		ArrivalCreatedAtUtc:          source.ArrivalCreatedAtUtc,
		ArrivalReceivedAtUtc:         source.ArrivalReceivedAtUtc,
		ArrivalUpdatedAtUtc:          source.ArrivalUpdatedAtUtc,
		CheckInDesks:                 source.CheckInDesks,
		BaggageCarousels:             source.BaggageCarousels,
		DepartureStation:             source.DepartureStation,
		ArrivalStation:               source.ArrivalStation,
		DepartureSourceID:            source.DepartureSourceId,
		ArrivalSourceID:              source.ArrivalSourceId,
	}
}
