# coding=utf-8
from __future__ import unicode_literals

import sys
import time

import sentry_sdk
from cloud_logger import default_config
from sentry_sdk.integrations.logging import LoggingIntegration
from tornado.httpserver import HTTPServer
from tornado.ioloop import IOLoop
from tornado.web import Application

import logging.config
import os
import threading
import traceback
from travel.avia.shared_flights.data_importer.flights_data_importer import FlightsDataImporter
from travel.avia.shared_flights.data_importer.handlers import PingHandler, StateHandler
from travel.avia.shared_flights.data_importer.parse_args import parse_args
from travel.avia.shared_flights.data_importer.state import AppState, State

REPEAT_PERIOD_IN_SECONDS = 1800

DEBUG = bool(os.getenv('DEBUG', False))
LOG_PATH = os.getenv('AVIA_LOG_PATH')
SENTRY_URL = os.getenv('AVIA_SENTRY_URL')
SANDBOX_OAUTH_TOKEN = os.getenv('AVIA_SANDBOX_OAUTH_TOKEN')


class DataImporterApplication(Application):

    def __init__(self, *args, **kwargs):
        super(DataImporterApplication, self).__init__(*args, **kwargs)
        self._state = AppState()
        self._args = {}

    @property
    def state(self):
        return self._state

    @state.setter
    def state(self, value):
        self._state = value

    def initialize(self, args_dict):
        self._args = args_dict
        thread = threading.Thread(target=self.initialize_in_background, args=())
        thread.daemon = True
        thread.start()

    def initialize_in_background(self):
        logger = logging.getLogger(__name__)
        try:
            logger.info('Starting data-importer application')
            while True:
                logger.info('Starting data import process')
                self.state.state = State.busy
                self.state.message = 'Data import process in progress'

                FlightsDataImporter(logger, oauth_token=SANDBOX_OAUTH_TOKEN).import_flights_data(self._args)

                self.state.state = State.available
                self.state.message = 'Data importer is idle'
                logger.info('Successfully completed data import process')
                time.sleep(REPEAT_PERIOD_IN_SECONDS)
        except:
            tb = traceback.format_exc()
            logger.exception('Unable to initialize the app')
            self.state.message = tb
            self.state.state = State.broken


def create_app():
    logging.basicConfig(
        format='%(asctime)s %(levelname)s %(message)s',
        level=logging.INFO,
        stream=sys.stdout,
    )

    state = AppState()
    handler_params = dict(
        state=state,
    )
    application = DataImporterApplication(
        [
            (r'/ping', PingHandler, handler_params),
            (r'/state', StateHandler, handler_params),
        ]
    )
    application.state = state

    return application


def main():
    if not os.getenv('AVIA_PGAAS_PASSWORD'):
        raise RuntimeError('AVIA_PGAAS_PASSWORD env var should be set.')

    if os.getenv('AVIA_SF_DI_SENTRY_DSN'):
        sentry_sdk.init(
            dsn=os.getenv('AVIA_SF_DI_SENTRY_DSN'),
            integrations=[LoggingIntegration()],
            environment=os.getenv('YANDEX_ENVIRONMENT_TYPE'),
        )
    setup_logging()

    logger = logging.getLogger(__name__)
    args = parse_args()

    # Command line version
    if not args.nodaemon:
        FlightsDataImporter(logger, oauth_token=SANDBOX_OAUTH_TOKEN).import_flights_data(args)
        return

    # Daemon version
    logger.info('About to start data-importer daemon in 5 sec. Press Ctrl+C now to abort if you are not sure')
    time.sleep(5)

    application = create_app()
    application.initialize(args)
    application.listen(8080)

    server = HTTPServer(application)
    server.start()

    def stop_handler(*args, **kw_args):
        while True:
            time.sleep(1)
            if application.state.state == State.broken:
                server.stop()
                IOLoop.current().stop()
                logger.error('Application is broken, aborting.')
                return

    thread = threading.Thread(target=stop_handler, args=())
    thread.daemon = True
    thread.start()

    IOLoop.current().start()


def setup_logging():
    if LOG_PATH:
        config = default_config(LOG_PATH)
        logging.config.dictConfig(config)
    else:
        logging.basicConfig(
            format='%(asctime)s %(levelname)s %(message)s',
            level=logging.INFO,
            stream=sys.stdout,
        )

if __name__ == '__main__':
    main()
