# coding=utf-8
from __future__ import unicode_literals

import json
import traceback

from sandbox.common import rest


class LatestReadyResource(object):

    def __init__(self, resource_types, logger):
        self._last_resource = None
        self._resource_types = resource_types
        self._logger = logger

    def __enter__(self):
        # now compare with the last time the rasp-resource has been updated
        sandbox = rest.Client()
        last_resource_meta = None
        last_resource = None
        last_resource_id = None
        for resource_type in self._resource_types:
            last_resource_meta = sandbox.resource.read(
                type=resource_type, state='READY', order='-id', limit=1)
            if not last_resource_meta or not last_resource_meta.get('items') or not last_resource_meta['items'][0]:
                continue
            resource = last_resource_meta['items'][0]
            resource_id = resource.get('id') if resource else None
            self._logger.info('Trying resource type: %s id: %s', resource_type, resource_id)
            if last_resource_id and resource_id and last_resource_id > resource_id:
                continue
            last_resource = resource
            last_resource_id = resource_id

        if not last_resource:
            return None

        self._last_resource = last_resource
        return self

    def __exit__(self, exc_type, exc_value, tb):
        if exc_type is not None:
            traceback.print_exception(exc_type, exc_value, tb)
        return False

    def __bool__(self):
        return bool(self._last_resource and self.get_id())

    def get_id(self):
        return self._last_resource.get('id') if self._last_resource else None

    def get_attributes(self):
        if not self._last_resource:
            return None
        return self._last_resource.get('attributes')

    def get_resource_proxy_url(self):
        if not self._last_resource:
            return None
        last_resource_http = self._last_resource.get('http')
        last_resource_proxy = last_resource_http.get('proxy') if last_resource_http else None
        if not last_resource_proxy:
            self.logger.warning('Unable to get http proxy URL for the latest SSIM resource. Result: %r.', self)
            return None
        return last_resource_proxy

    def __repr__(self):
        if not self._last_resource:
            return '{}'
        else:
            return json.dumps(self._last_resource)
