import json
import requests
from travel.avia.library.python.shared_dicts.rasp import get_repository, ResourceType


class CarrierStorage:
    def __init__(self, logger, sandbox, oauth_token=None):
        self.logger = logger
        self.by_code, self.by_id = self.fetch_carriers(sandbox, oauth_token)

    def fetch_carriers(self, sandbox, oauth_token):
        carrier_repository = get_repository(
            ResourceType.TRAVEL_DICT_RASP_CARRIER_PROD,
            oauth=oauth_token,
        )
        if not carrier_repository:
            self.logger.error('Unable to fetch the carriers data.')
            return None

        empty_codes = 0
        carriers = {}
        carriers_by_id = {}
        carriers_to_sort = []

        for carrier in carrier_repository.itervalues():
            if not carrier.Iata and not carrier.Icao and not carrier.SirenaId and not carrier.IcaoRu:
                empty_codes += 1
                continue

            if carrier.IsHidden:
                continue
            carriers_by_id[carrier.Id] = carrier
            carriers_to_sort.append(carrier)

        carriers_to_sort.sort(key=self.carriers_sort_key)
        for carrier in carriers_to_sort:
            if carrier.Iata and not carriers.get(carrier.Iata):
                # It's more convenient to have Aeroflot as SU, other carriers with Iata='SU' are accessible by id
                if carrier.Iata != 'SU' or carrier.Icao == 'AFL':
                    carriers[carrier.Iata] = carrier
            if carrier.Icao and not carriers.get(carrier.Icao):
                carriers[carrier.Icao] = carrier

        self.logger.info('Parsed carriers: %s', format_number(len(carriers)))
        self.logger.info('Empty codes: %s', format_number(empty_codes))

        return carriers, carriers_by_id

    # Sort by "priority descending" first, then by id
    @staticmethod
    def carriers_sort_key(carrier):
        return -carrier.Priority, carrier.Id


def format_number(num):
    return '{:,}'.format(num)
