import logging
from collections import defaultdict
from dataclasses import dataclass, field, asdict
from travel.avia.shared_flights.lib.python.db_models import (
    Carrier, FlightStatus, FlightPattern, SirenaFlightPattern, ApmFlightPattern, )
from travel.library.python.sender import TransactionalApi
from travel.library.python.sender.error import SenderError
from travel.proto.shared_flights.sirena.routes_pb2 import TRoute
from travel.proto.shared_flights.ssim.flights_pb2 import TFlightBase
from typing import Iterable, Dict, List

logger = logging.getLogger(__name__)


@dataclass
class MissingMetaInfoRouteStopPoint:
    station_code: str = ''
    city_code: str = ''
    departure_time: str = ''
    arrival_time: str = ''


@dataclass
class MissingMetaInfoRoute:
    operating_from_date: str = ''
    operating_to_date: str = ''
    operating_on_days: str = ''
    leg_seq_number: int = 1

    aircraft_model: str = ''
    points: List[MissingMetaInfoRouteStopPoint] = field(default_factory=list)


@dataclass
class MissingMetaInfo:
    carrier_code: str = ''
    flight_number: str = ''
    routes: List[MissingMetaInfoRoute] = field(default_factory=list)


def time_or_empty(inttime):
    if inttime == 9999:
        return ''
    strtime = '{:04}'.format(inttime)
    return ':'.join((strtime[:2], strtime[2:]))


def date_format(intdate):
    strdate = '{:08}'.format(intdate)
    return '-'.join((strdate[:4], strdate[4:6], strdate[6:]))


def flight_base_route(flight_base_proto: TFlightBase):
    return asdict(MissingMetaInfo(
        carrier_code=flight_base_proto.OperatingCarrierIata,
        flight_number=flight_base_proto.OperatingFlightNumber,
        routes=[
            MissingMetaInfoRoute(
                aircraft_model=flight_base_proto.AircraftModel,
                leg_seq_number=flight_base_proto.LegSeqNumber,
                points=[
                    MissingMetaInfoRouteStopPoint(
                        station_code=flight_base_proto.DepartureStationIata,
                        departure_time=time_or_empty(flight_base_proto.ScheduledDepartureTime),
                    ),
                    MissingMetaInfoRouteStopPoint(
                        station_code=flight_base_proto.ArrivalStationIata,
                        arrival_time=time_or_empty(flight_base_proto.ScheduledArrivalTime),
                    ),
                ]
            )
        ]
    ))


def sirena_route_extra(route: TRoute):
    return asdict(MissingMetaInfo(
        carrier_code=route.CarrierCode,
        flight_number=route.FlightNumber,
        routes=[
            MissingMetaInfoRoute(
                operating_from_date=date_format(fp.OperatingFromDate),
                operating_to_date=date_format(fp.OperatingUntilDate),
                operating_on_days=str(fp.OperatingOnDays),
                aircraft_model=fp.AircraftModel,
                points=[
                    MissingMetaInfoRouteStopPoint(
                        station_code=point.StationCode,
                        city_code=point.CityCode,
                        departure_time=time_or_empty(point.DepartureTime),
                        arrival_time=time_or_empty(point.ArrivalTime),
                    ) for point in fp.StopPoints
                ]
            ) for fp in route.FlightPatterns
        ]
    ))


class MissingDataStorage:
    def __init__(self):
        self.missing_carriers = dict()
        self.missing_stations = dict()

    def add_station(self, station, meta=None):
        if station not in self.missing_stations:
            logger.info('Added missing station %r', station)
            self.missing_stations[station] = meta

    def add_carrier(self, carrier, meta=None):
        if carrier not in self.missing_carriers:
            logger.info('Added missing carrier %r', carrier)
            self.missing_carriers[carrier] = meta

    def dump(self):
        return {
            'carriers': self.missing_carriers,
            'stations': self.missing_stations,
        }

    def empty(self):
        return (len(self.missing_stations) + len(self.missing_carriers)) == 0


class ExtraDataStorage:
    def __init__(self):
        self.extra_carriers = []

    def empty(self):
        return len(self.extra_carriers) == 0

    def dump(self):
        return {'extra_carriers': self.extra_carriers}

    def collect_if_needed(self, session_factory):
        session = session_factory()
        if not session:
            logger.error('Cannot start db session to collect extra carriers')
        try:
            session = session_factory()
            db_carriers: Iterable[Carrier] = session.query(Carrier).filter(Carrier.is_hidden.is_(False)).all()
            db_flight_patterns: Iterable[FlightPattern] = session.query(
                FlightPattern.marketing_carrier
            ).union(
                session.query(SirenaFlightPattern.marketing_carrier),
                session.query(ApmFlightPattern.marketing_carrier)
            ).distinct()
            db_flight_status_carrier_codes = session.query(FlightStatus.airlinecode).distinct().all()

            carriers_by_code = defaultdict(list)
            carrier_by_id: Dict[int, Carrier] = {}
            for db_carrier in db_carriers:
                carriers_by_code[db_carrier.iata].append(db_carrier.id)
                carriers_by_code[db_carrier.sirena_id].append(db_carrier.id)
                carriers_by_code[db_carrier.icao].append(db_carrier.id)
                carriers_by_code[db_carrier.icao_ru].append(db_carrier.id)
                carrier_by_id[db_carrier.id] = db_carrier
            carriers_by_code.pop(None, None)
            carriers_by_code.pop('', None)

            schedule_carriers = set(pattern.marketing_carrier for pattern in db_flight_patterns)

            flight_status_carriers = set(
                carrier_id for carrier_code in db_flight_status_carrier_codes
                for carrier_id in carriers_by_code[carrier_code.airlinecode]
                if carrier_code.airlinecode in carriers_by_code
            )

            extra_carriers = set(carrier_by_id.keys()) - (schedule_carriers | flight_status_carriers)

            extra_list = []
            for extra_carrier in extra_carriers:
                carrier = carrier_by_id[extra_carrier]
                extra_list.append(
                    {
                        'id': carrier.id,
                        'iata': carrier.iata,
                        'sirena': carrier.sirena_id,
                        'icao': carrier.icao,
                        'icao_ru': carrier.icao_ru,
                        'title': carrier.title_ru or carrier.title,
                    })
            self.extra_carriers = sorted(extra_list, key=lambda c: c['id'])
            if self.extra_carriers:
                logger.info('Found extra carriers: %s', self.extra_carriers)
            else:
                logger.info('There are %d non-hidden carriers, but no extra', len(carrier_by_id))
        finally:
            if session:
                session.close()


class InvalidDataEmailDumper:
    def __init__(
        self,
        missing_data: MissingDataStorage,
        extra_data: ExtraDataStorage,
        email_client: TransactionalApi,
        environment: str = 'UNKNOWN',
    ):
        self.missing_data: MissingDataStorage = missing_data
        self.extra_data: ExtraDataStorage = extra_data
        self.email_client: TransactionalApi = email_client
        self.environment = environment

    def send(self, send_to: str, for_testing: bool):
        missing_dump = {}
        if self.missing_data.empty():
            logger.info('Missing data wont be sent because its empty')
        else:
            missing_dump = self.missing_data.dump()

        extra_dump = {}
        if self.extra_data.empty():
            logger.info('Extra data wont be sent because its empty')
        else:
            extra_dump = self.extra_data.dump()

        if not missing_dump and not extra_dump:
            logger.info('Message with missing and extra data wont be sent, because there\'s no data')
            return

        data = {**missing_dump, **extra_dump}
        data['environment'] = self.environment
        try:
            self.email_client.send(
                to_email=send_to,
                args=data,
                is_async=False,
                for_testing=for_testing,
            )
        except SenderError:
            logger.exception('Cannot send email with missing data to %s', send_to)
