import json
import requests
from travel.avia.library.python.shared_dicts.rasp import get_repository, ResourceType
from travel.proto.dicts.rasp.code_system_pb2 import ECodeSystem
from travel.proto.dicts.rasp.station_code_pb2 import TStationCode


class StationCodes:
    def __init__(self):
        self.iata = ''
        self.icao = ''
        self.sirena = ''


class StationCodesStorage:
    def __init__(self, logger, sandbox, oauth_token=None):
        self._logger = logger
        self.by_id = self.fetch_station_codes(sandbox, oauth_token)

    def fetch_station_codes(self, sandbox, oauth_token):
        station_code_repository = get_repository(
            ResourceType.TRAVEL_DICT_RASP_STATION_CODE_PROD,
            oauth=oauth_token,
        )
        if not station_code_repository:
            self.logger.error('Unable to fetch the station codes data.')
            return None

        station_codes = {}
        empty_code = 0
        skipped_station_codes_count = 0
        known_station_codes = set()
        duplicates = set()
        for station_code in station_code_repository.values():
            if station_code.SystemId == ECodeSystem.CODE_SYSTEM_IATA:
                self.ensure_station_id(station_codes, station_code.StationId)
                station_codes[station_code.StationId].iata = station_code.Code
                continue

            if station_code.SystemId == ECodeSystem.CODE_SYSTEM_ICAO:
                self.ensure_station_id(station_codes, station_code.StationId)
                station_codes[station_code.StationId].icao = station_code.Code
                continue

            if station_code.SystemId == ECodeSystem.CODE_SYSTEM_SIRENA:
                self.ensure_station_id(station_codes, station_code.StationId)
                station_codes[station_code.StationId].sirena = station_code.Code
                continue

            skipped_station_codes_count += 1
            if not station_code.Code:
                empty_code += 1
                continue
            if station_code.Code in known_station_codes:
                duplicates.add(station_code.Code)
            known_station_codes.add(station_code.Code)

        self._logger.info('Parsed station codes: %s', format_number(len(station_codes)))
        self._logger.info('Station codes with empty code: %s', format_number(empty_code))
        self._logger.info('Station code duplicates count: %s', format_number(len(duplicates)))
        self._logger.info('Skipped station codes: %s', format_number(skipped_station_codes_count))
        return station_codes

    @staticmethod
    def ensure_station_id(station_codes: dict, station_id):
        if not station_codes.get(station_id):
            station_codes[station_id] = StationCodes()


def format_number(num):
    return '{:,}'.format(num)
