# coding=utf-8
from __future__ import unicode_literals

import os
import sys
import threading
import time
import traceback

from tornado.httpserver import HTTPServer
from tornado.ioloop import IOLoop
from tornado.web import Application

from travel.avia.shared_flights.diff_builder.diff_builder import DiffBuilder, Mode
from travel.avia.shared_flights.diff_builder.handlers import PingHandler, StateHandler
from travel.avia.shared_flights.diff_builder.mds_cleaner import MdsCleaner
from travel.avia.shared_flights.diff_builder.state import AppState, State
from travel.avia.shared_flights.diff_builder.start_app import get_solomon_push_reporter, parse_args, setup_logging, test_env_vars

REPEAT_PERIOD_IN_SECONDS = 30


class DiffBuilderApplication(Application):
    def __init__(self, *args, solomon_push_reporter=None, logger=None, **kwargs):
        super(DiffBuilderApplication, self).__init__(*args, **kwargs)
        self._state = AppState()
        self._args = {}
        self.solomon_push_reporter = solomon_push_reporter
        self._logger = logger

    @property
    def state(self):
        return self._state

    @state.setter
    def state(self, value):
        self._state = value

    def initialize(self, args_dict):
        self._args = args_dict
        threading.Thread(target=self.initialize_in_background, args=(), daemon=True).start()

    def initialize_in_background(self):
        try:
            self._logger.info('Starting diff-builder application')
            while True:
                # build delta every REPEAT_PERIOD_IN_SECONDS
                self._logger.info('Delta build is starting')
                self.state.state = State.busy
                self.state.message = 'Delta build is in progress'

                DiffBuilder(self._logger, self.solomon_push_reporter).build_diff(Mode.delta)

                self.state.state = State.available
                self.state.message = 'Diff builder is idle'
                self._logger.info('Delta has been built successfully')
                time.sleep(REPEAT_PERIOD_IN_SECONDS)
        except:
            tb = traceback.format_exc()
            self._logger.exception('Unable to initialize the app')
            self.state.message = tb
            self.state.state = State.broken


def create_app(solomon_push_reporter=None, logger=None):
    state = AppState()
    handler_params = dict(
        state=state,
    )
    application = DiffBuilderApplication(
        [
            (r'/ping', PingHandler, handler_params),
            (r'/state', StateHandler, handler_params),
        ],
        solomon_push_reporter=solomon_push_reporter,
        logger=logger,
    )
    application.state = state

    return application


def main():
    logger = setup_logging()
    test_env_vars()
    args = parse_args()
    solomon_push_reporter = get_solomon_push_reporter(logger)

    # Command line version
    if not args.nodaemon:
        mode = None
        if args.snapshot:
            mode = Mode.snapshot
        elif args.delta:
            mode = Mode.delta

        if not mode:
            logger.error("Please specify --snapshot or --delta mode")
            return
        DiffBuilder(logger, solomon_push_reporter).build_diff(mode, args.output_file, args.force)
        return

    # Daemon version
    logger.info('About to start diff-builder daemon in 5 sec. Press Ctrl+C now to abort if you are not sure')
    time.sleep(5)

    application = create_app(
        solomon_push_reporter=solomon_push_reporter,
        logger=logger,
    )
    application.initialize(args)
    application.listen(8080)

    server = HTTPServer(application)
    server.start(1)

    def stop_handler(*args, **kw_args):
        while True:
            time.sleep(1)
            if application.state.state == State.broken:
                server.stop()
                IOLoop.current().stop()
                logger.error('Application is broken, aborting.')
                time.sleep(2)
                sys.exit(-1)

    thread = threading.Thread(target=stop_handler, args=())
    thread.daemon = True
    thread.start()

    MdsCleaner(
        os.getenv('AVIA_MDS_ACCESS_KEY_ID'),
        os.getenv('AVIA_MDS_ACCESS_KEY_SECRET'),
        os.getenv('AVIA_MDS_SNAPSHOTS_FOLDER', 'dev'),
    ).start()

    IOLoop.current().start()


if __name__ == '__main__':
    main()
