# coding=utf-8
from __future__ import unicode_literals

from collections import defaultdict
from datetime import datetime, timedelta

from travel.avia.shared_flights.lib.python.consts.consts import MIN_APM_FLIGHT_PATTERN_ID, MIN_SIRENA_FLIGHT_PATTERN_ID
from travel.avia.shared_flights.lib.python.date_utils.date_index import DateIndex
from travel.avia.shared_flights.lib.python.date_utils.date_matcher import DateMatcher
from travel.proto.shared_flights.ssim.flights_pb2 import TFlightPattern


class CodesharesMap(object):
    def __init__(self, min_flight_pattern_id, logger):
        # flight key -> bucket key -> []flight patterns
        self._amadeus_flights = defaultdict(lambda: defaultdict(list))
        self._sirena_flights = defaultdict(lambda: defaultdict(list))
        self._apm_flights = defaultdict(lambda: defaultdict(list))
        self._known_derivative_flight_keys = set()
        self._processed_derivative_flight_keys = set()

        self._min_flight_pattern_id = min_flight_pattern_id
        self._start_date = datetime.now() - timedelta(days=75)
        self._date_index = DateIndex(self._start_date)
        self._date_matcher = DateMatcher(self._date_index)
        self._logger = logger

    def add_flight(self, flight_pattern: TFlightPattern) -> None:
        marketing_bucket_key = get_bucket_key(flight_pattern)
        flight_key = bucket_to_flight_key(marketing_bucket_key)  # marketing flight
        if flight_pattern.IsDerivative and not flight_pattern.IsCodeshare:
            self._known_derivative_flight_keys.add(flight_key)
        if CodesharesMap.is_sirena_flight(flight_pattern):
            self._sirena_flights[flight_key][marketing_bucket_key].append(flight_pattern)
        elif CodesharesMap.is_apm_flight(flight_pattern):
            self._apm_flights[flight_key][marketing_bucket_key].append(flight_pattern)
        else:
            self._amadeus_flights[flight_key][marketing_bucket_key].append(flight_pattern)

    def generate_flights_to_write(self):
        bucket_keys_map = {}

        # Make APM flights top priority, assume they don't have codeshares
        for flight_key, apm_value in self._apm_flights.items():
            for bucket_key, flight_patterns in apm_value.items():
                bucket_keys_map[bucket_key] = flight_patterns

        # Replace Amadeus flights with Sirena flights first
        for flight_key, amadeus_value in self._amadeus_flights.items():
            sirena_value = self._sirena_flights.get(flight_key)
            if flight_key in self._apm_flights:
                continue
            is_derivative = flight_key in self._known_derivative_flight_keys
            if is_derivative:
                self._processed_derivative_flight_keys.add(flight_key)
            if is_derivative or not sirena_value:
                for bucket_key, flight_patterns in amadeus_value.items():
                    bucket_keys_map[bucket_key] = flight_patterns

        for flight_key, sirena_value in self._sirena_flights.items():
            if flight_key in self._apm_flights:
                continue
            if not sirena_value:
                continue
            for bucket_key, flight_patterns in sirena_value.items():
                for sirena_fp in flight_patterns:
                    if sirena_fp.IsCodeshare:
                        continue
                    amadeus_value = self._amadeus_flights.get(flight_key)
                    if not amadeus_value:
                        continue
                    for amadeus_bucket, amadeus_fps in amadeus_value.items():
                        if amadeus_bucket != bucket_key:
                            continue
                        for amadeus_fp in amadeus_fps:
                            if not amadeus_fp.IsCodeshare:
                                continue
                            overlap = self._date_matcher.intersect(
                                amadeus_fp.OperatingFromDate,
                                amadeus_fp.OperatingUntilDate,
                                amadeus_fp.OperatingOnDays,
                                sirena_fp.OperatingFromDate,
                                sirena_fp.OperatingUntilDate,
                                sirena_fp.OperatingOnDays,
                            )
                            if overlap:
                                sirena_fp.IsCodeshare = True
                                sirena_fp.BucketKey = amadeus_fp.BucketKey
                                sirena_fp.OperatingFlightPatternId = amadeus_fp.Id
                                break
                # skip operating flights not fetched directly from Sirena
                if flight_key in self._processed_derivative_flight_keys:
                    filtered_flight_patterns = [fp for fp in flight_patterns if fp.IsCodeshare]
                    if filtered_flight_patterns:
                        current_value = bucket_keys_map.get(bucket_key)
                        # It is unclear what to do in this case: we have an operating flight from Amadeus
                        # and Sirena claims that it is a codeshare flight (possibly on different dates).
                        # Based on our past experience, whatever side we choose, it may be an incorrect decision.
                        # For now, let's drop the flights from Sirena and keep ones from Amadeus -
                        # mostly because it is the simplest decision to implement and we don't have an ideal
                        # algorithm anyway. If it turns out this is not good enough, then we'd implement an improvement.
                        if not current_value:
                            bucket_keys_map[bucket_key] = filtered_flight_patterns
                else:
                    bucket_keys_map[bucket_key] = flight_patterns

        missing_operating_buckets = 0
        for bucket_key, flight_patterns in bucket_keys_map.items():
            for flight_pattern in flight_patterns:
                if flight_pattern.IsCodeshare:
                    bucket_key_segments = bucket_keys_map.get(flight_pattern.BucketKey)
                    found_operating_segments = False
                    if bucket_key_segments:
                        operating_segments = [segment for segment in bucket_key_segments if not segment.IsCodeshare]
                        if operating_segments:
                            found_operating_segments = True
                            yield from self.generate_overlaps(flight_pattern, operating_segments)
                    if not found_operating_segments:
                        missing_operating_buckets += 1
                else:
                    yield flight_pattern

        self._logger.info('Missing operating buckets count: %d', missing_operating_buckets)

    def generate_overlaps(self, flight_pattern, matching_flight_patterns):
        if not matching_flight_patterns:
            return

        for matching_leg in matching_flight_patterns:
            overlap = self._date_matcher.intersect(
                flight_pattern.OperatingFromDate,
                flight_pattern.OperatingUntilDate,
                flight_pattern.OperatingOnDays,
                matching_leg.OperatingFromDate,
                matching_leg.OperatingUntilDate,
                matching_leg.OperatingOnDays,
            )
            if overlap:
                new_flight_pattern = TFlightPattern()
                new_flight_pattern.MergeFrom(flight_pattern)
                new_flight_pattern.Id = self._min_flight_pattern_id
                self._min_flight_pattern_id += 1
                new_flight_pattern.FlightId = matching_leg.FlightId
                new_flight_pattern.BucketKey = matching_leg.BucketKey
                new_flight_pattern.OperatingFromDate = overlap[0].replace('.', '-')
                new_flight_pattern.OperatingUntilDate = overlap[1].replace('.', '-')
                new_flight_pattern.OperatingOnDays = overlap[2]
                new_flight_pattern.IsCodeshare = True
                new_flight_pattern.OperatingFlightPatternId = matching_leg.Id
                yield new_flight_pattern

    def get_amadeus_flights(self, marketing_bucket_key):
        flight_key = bucket_to_flight_key(marketing_bucket_key)
        for key, amadeus_value in self._amadeus_flights[flight_key].items():
            if key == marketing_bucket_key:
                return amadeus_value
        return None

    def get_sirena_flights(self, marketing_bucket_key):
        flight_key = bucket_to_flight_key(marketing_bucket_key)
        for key, sirena_value in self._sirena_flights[flight_key].items():
            if key == marketing_bucket_key:
                return sirena_value
        return None

    @staticmethod
    def is_sirena_flight(flight_pattern: TFlightPattern) -> bool:
        return flight_pattern.Id >= MIN_SIRENA_FLIGHT_PATTERN_ID

    @staticmethod
    def is_apm_flight(flight_pattern: TFlightPattern) -> bool:
        return MIN_APM_FLIGHT_PATTERN_ID <= flight_pattern.Id < MIN_SIRENA_FLIGHT_PATTERN_ID


def bucket_to_flight_key(bucket_key: str) -> str:
    parts = bucket_key.split('.')
    return '.'.join(parts[:2])


def get_bucket_key(flight_pattern: TFlightPattern) -> str:
    return '{}.{}.{}'.format(
        flight_pattern.MarketingCarrier, flight_pattern.MarketingFlightNumber, flight_pattern.LegSeqNumber
    )
