# coding=utf-8
from __future__ import unicode_literals

import logging
import threading
from datetime import datetime, timedelta

import pytz
import time

from travel.avia.shared_flights.diff_builder.factory import make_mds_client

logger = logging.getLogger(__name__)

BUCKET = 'avia-shared-flights-dev'
REPEAT_PERIOD_IN_SECONDS = 900
DAYS_TO_KEEP = 2


class MdsCleaner:
    def __init__(self, access_key_id: str, aws_secret_access_key: str, mds_folder: str):
        self._mds_client = make_mds_client(key_id=access_key_id, key_secret=aws_secret_access_key)
        self._folder_key = 'snapshots/{}/'.format(mds_folder)

    def start(self):
        thread = threading.Thread(target=self.run, args=())
        thread.daemon = True
        thread.start()

    def run(self):
        start = True
        while True:
            if not start:
                time.sleep(REPEAT_PERIOD_IN_SECONDS)
            start = False
            try:
                threshold_date = datetime.now(pytz.UTC) - timedelta(days=DAYS_TO_KEEP)
                logger.info('Staring clean-up with threshold %s', threshold_date)
                mds_objects_response = self._mds_client.list_objects(
                    Bucket=BUCKET,
                    Prefix=self._folder_key,
                )
                mds_objects = mds_objects_response.get('Contents')
                if not mds_objects:
                    logger.info('No suitable objects in the bucket')
                    continue

                for mds_object in mds_objects:
                    last_modified = mds_object['LastModified']
                    key = mds_object['Key']
                    if key and last_modified and threshold_date > last_modified:
                        if 'latest.txt' not in key:
                            logger.info('Deleting %s', key)
                            self._mds_client.delete_object(
                                Bucket=BUCKET,
                                Key=key,
                            )

            except Exception as exc:
                logger.exception('Error while deleting old objects. %r', exc)
