# coding=utf-8
from __future__ import unicode_literals

import argparse
import logging
import logging.config
import os
import sys

import sentry_sdk
from sentry_sdk.integrations.logging import LoggingIntegration

from cloud_logger import default_config
from travel.avia.shared_flights.diff_builder.diff_builder import DiffBuilder, Mode
from travel.avia.shared_flights.tasks.monitoring.helpers import UnsupportedEnvironmentError, solomon_push_reporter_maker


def parse_args(args_list=None):
    parser = argparse.ArgumentParser()
    parser.add_argument('--force', required=False, default=False, type=bool)
    parser.add_argument('--snapshot', required=False, default=False, type=bool)
    parser.add_argument('--delta', required=False, default=False, type=bool)
    parser.add_argument('--nodaemon', required=False, action='store_false')
    parser.add_argument('--output_file', required=False, default='snapshot.zip', type=str)
    if args_list:
        return parser.parse_args(args_list)
    return parser.parse_args()


def setup_logging():
    LOG_PATH = os.environ['AVIA_LOG_PATH']
    config = default_config(LOG_PATH)
    logging.config.dictConfig(config)
    logging.basicConfig(
        format='%(asctime)s %(levelname)s %(message)s',
        level=logging.INFO,
        stream=sys.stdout,
    )
    return logging.getLogger(__name__)


def test_env_vars():
    if os.getenv('AVIA_SF_DB_SENTRY_DSN'):
        sentry_sdk.init(
            dsn=os.getenv('AVIA_SF_DB_SENTRY_DSN'),
            integrations=[LoggingIntegration()],
            environment=os.getenv('YANDEX_ENVIRONMENT_TYPE'),
        )

    if not os.getenv('AVIA_PGAAS_PASSWORD'):
        raise RuntimeError('AVIA_PGAAS_PASSWORD env var should be set.')

    if not os.getenv('AVIA_MDS_ACCESS_KEY_ID'):
        raise RuntimeError('AVIA_MDS_ACCESS_KEY_ID env var should be set.')

    if not os.getenv('AVIA_MDS_ACCESS_KEY_SECRET'):
        raise RuntimeError('AVIA_MDS_ACCESS_KEY_SECRET env var should be set.')


def get_solomon_push_reporter(logger):
    SOLOMON_TOKEN = os.getenv('SOLOMON_TOKEN', '') or None
    if not SOLOMON_TOKEN:
        logger.warning('Solomon token is not provided. Metrics might not be sent.')

    ENVIRONMENT = os.getenv('YANDEX_ENVIRONMENT_TYPE', 'UNKNOWN').lower()

    solomon_push_reporter = None
    try:
        solomon_push_reporter = solomon_push_reporter_maker(ENVIRONMENT, SOLOMON_TOKEN)
    except UnsupportedEnvironmentError:
        logger.exception('Metrics won\'t be sent in current environment')
    except:
        logger.exception('Unexpected error while creating solomon push reporter')
    return solomon_push_reporter


def run_build_snapshot(args, output_data_file):
    logger = setup_logging()
    test_env_vars()
    parsed_args = parse_args(args)
    solomon_push_reporter = get_solomon_push_reporter(logger)
    DiffBuilder(logger, solomon_push_reporter).build_diff(Mode.snapshot, output_data_file, parsed_args.force)
