package dtutil

import (
	"fmt"
	"strconv"
	"time"

	"a.yandex-team.ru/library/go/core/xerrors"
)

// Noon in HHMM format
var Noon = 1200

const TimeNotSpecified = 9999

var MinDate = 20100101
var MaxDate = 20990101
var MinYear = MinDate / 10000
var MaxYear = MaxDate / 10000

const IsoDate = "2006-01-02"
const IsoDateNoDashes = "20060102"
const IsoTime = "15:04:05"
const IsoDateTime = "2006-01-02 15:04:05"

//StringTime in "HH:MM:SS" format
type StringTime string

//IntTime in HHMM format
type IntTime int32

// StringDate formatted in "YYYY-MM-DD" (default) or "YYYYMMDD" (supported)
type StringDate string

// IntDate formatted in YYYYMMDD
type IntDate int32

// Datetime in YYYY-MM-DDTHH:MM:SS format
type StringDateTime string

// Converts "YYYY-MM-DD" into (int32)YYYYMMDD
func (date StringDate) ToIntDate() IntDate {
	if len(date) > 10 {
		date = date[:10]
	}
	result := 0
	for _, ch := range []byte(date) {
		if ch >= '0' && ch <= '9' {
			result = result*10 + int(ch-'0')
		}
	}
	if result < MinDate || result > MaxDate {
		return 0
	}
	day := result % 100
	if day == 0 || day > 31 {
		return 0
	}
	month := result / 100 % 100
	if month == 0 || month > 12 {
		return 0
	}
	return IntDate(result)
}

// ToTime converts converts string date to time.Time in defined timezone
func (date StringDate) ToTime(tz *time.Location) time.Time {
	var yearstr, monthstr, daystr string
	var year, month, day uint16

	if len(date) == 8 {
		yearstr = string(date[0:4])
		monthstr = string(date[4:6])
		daystr = string(date[6:8])
	} else if len(date) == 10 {
		yearstr = string(date[0:4])
		monthstr = string(date[5:7])
		daystr = string(date[8:10])
	}
	year = parseBase10_16(yearstr)
	month = parseBase10_16(monthstr)
	day = parseBase10_16(daystr)
	return time.Date(int(year), time.Month(month), int(day), 0, 0, 0, 0, tz)
}

// AddDaysP is a panic version of AddDays
func (date IntDate) AddDaysP(days int) IntDate {
	di := DateCache.IndexOfIntDateP(date)
	return DateCache.Date(di + days)
}

func (date IntDate) AddDays(days int) IntDate {
	di, ok := DateCache.IndexOfIntDate(date)
	if !ok {
		return 0
	}
	return DateCache.Date(di + days)
}

func (date StringDate) AddDaysP(days int) StringDate {
	if days == 0 {
		return date
	}
	di := DateCache.IndexOfStringDateP(date)
	return DateCache.Date(di + days).StringDateDashed()
}

func (date IntDate) StringDate() StringDate {
	return StringDate(strconv.Itoa(int(date)))
}

func (date IntDate) Components() (year int, month time.Month, day int) {
	day = int(date % 100)
	date /= 100
	month = time.Month(date % 100)
	date /= 100
	year = int(date % 10000)
	return year, month, day
}

func (date IntDate) StringDateDashed() StringDate {
	day := date % 100
	date /= 100
	month := date % 100
	date /= 100
	year := date % 10000

	bytes := [10]byte{}

	bytes[9] = byte('0' + (day % 10))
	day /= 10
	bytes[8] = byte('0' + (day % 10))
	bytes[7] = '-'
	bytes[6] = byte('0' + (month % 10))
	month /= 10
	bytes[5] = byte('0' + (month % 10))
	bytes[4] = '-'
	bytes[3] = byte('0' + (year % 10))
	year /= 10
	bytes[2] = byte('0' + (year % 10))
	year /= 10
	bytes[1] = byte('0' + (year % 10))
	year /= 10
	bytes[0] = byte('0' + (year % 10))

	return StringDate(bytes[:])
}

func parseBase10_16(num string) uint16 {
	var out uint16
	zeropad := true
	for _, c := range num {
		if zeropad {
			if c == '0' {
				zeropad = true
				continue
			} else {
				zeropad = false
			}
		}
		out = out*10 + uint16(c-'0')
	}
	return out
}

func (t IntTime) StringTime() StringTime {
	return StringTime(t.String())
}

func (t IntTime) IsSpecified() bool {
	return t != TimeNotSpecified
}

func (t IntTime) IsValid() bool {
	return t.IsSpecified() && t/100 < 24 && t%100 < 60
}

func (t IntTime) String() string {
	if t < 0 || t/100 >= 24 || t%100 >= 60 {
		return "<Invalid time " + strconv.Itoa(int(t)) + ">"
	}
	return fmt.Sprintf("%02d:%02d:00", t/100, t%100)
}

func (t IntTime) Duration() time.Duration {
	return time.Duration(t%100)*time.Minute + time.Duration(t/100)*time.Hour
}

func DateToInt(dt time.Time) IntDate {
	year, month, day := dt.Date()
	date := int32(year)
	date = date*100 + int32(month)
	date = date*100 + int32(day)
	return IntDate(date)
}

func TimeToIntTime(dt time.Time) IntTime {
	if dt.IsZero() {
		return TimeNotSpecified
	}
	hour, minute := dt.Hour(), dt.Minute()
	result := int32(hour)*100 + int32(minute)
	return IntTime(result)
}

// Converts (int32)YYYYMMDD в time.Date
func IntToTime(date IntDate, hhmm IntTime, tz *time.Location) time.Time {
	if !hhmm.IsValid() || tz == nil {
		return time.Time{}
	}
	return time.Date(int(date/10000), time.Month(date/100%100), int(date%100), int(hhmm/100), int(hhmm%100), 0, 0, tz)
}

func FormatDateTimeISO(t time.Time) string {
	if t.IsZero() {
		return ""
	}
	buf := []byte("2006-01-02 15:04:05")
	year, month, day := t.Date()
	hour, min, sec := t.Clock()

	buf[3] = byte(year%10) + '0'
	year /= 10
	buf[2] = byte(year%10) + '0'
	year /= 10
	buf[1] = byte(year%10) + '0'
	year /= 10
	buf[0] = byte(year%10) + '0'

	buf[6] = byte(month%10) + '0'
	month /= 10
	buf[5] = byte(month%10) + '0'

	buf[9] = byte(day%10) + '0'
	day /= 10
	buf[8] = byte(day%10) + '0'

	buf[12] = byte(hour%10) + '0'
	hour /= 10
	buf[11] = byte(hour%10) + '0'

	buf[15] = byte(min%10) + '0'
	min /= 10
	buf[14] = byte(min%10) + '0'

	buf[18] = byte(sec%10) + '0'
	sec /= 10
	buf[17] = byte(sec%10) + '0'

	return string(buf)
}

func ParseDateTimeISO(strDate string, tz *time.Location) (time.Time, error) {
	return time.ParseInLocation(IsoDateTime, strDate, tz)
}

func FormatTimeHHMM(t IntTime) string {
	if !t.IsValid() {
		return ""
	}
	buf := [5]byte{'0', '0', ':', '0', '0'}
	buf[0] = byte((t/1000)%10) + '0'
	buf[1] = byte((t/100)%10) + '0'
	buf[3] = byte((t/10)%10) + '0'
	buf[4] = byte(t%10) + '0'
	return string(buf[:])
}

func FormatTimezone(t time.Time) string {
	return t.Format("-0700")
}

//FormatDateIso formats time.Time value to 2006-01-02 format
func FormatDateIso(t time.Time) StringDate {
	buf := [10]byte{'0', '0', '0', '0', '-', '0', '0', '-', '0', '0'}
	year, month, day := t.Date()
	buf[0] = byte((year/1000)%10) + '0'
	buf[1] = byte((year/100)%10) + '0'
	buf[2] = byte((year/10)%10) + '0'
	buf[3] = byte(year%10) + '0'

	buf[5] = byte((month/10)%10) + '0'
	buf[6] = byte((month)%10) + '0'

	buf[8] = byte((day/10)%10) + '0'
	buf[9] = byte((day)%10) + '0'

	return StringDate(buf[:])
}

func (datetime StringDateTime) ToTime(loc *time.Location) (time.Time, error) {
	if len(datetime) == 0 {
		return time.Time{}, nil
	}
	result, err := time.ParseInLocation("2006-01-02T15:04:05", string(datetime), loc)
	if err != nil {
		return result, err
	}
	if result.Year() < MinYear || result.Year() >= MaxYear {
		return time.Time{}, xerrors.Errorf("Date value %v is not within the allowed scope", result)
	}
	return result, nil
}

// FormatTimeIso formats time.Time value to 23:59:59 format
func FormatTimeIso(t time.Time) string {
	buf := [8]byte{'0', '0', ':', '0', '0', ':', '0', '0'}
	hour, minute, second := t.Hour(), t.Minute(), t.Second()
	buf[0] = byte((hour/10)%10) + '0'
	buf[1] = byte(hour%10) + '0'

	buf[3] = byte((minute/10)%10) + '0'
	buf[4] = byte((minute)%10) + '0'

	buf[6] = byte((second/10)%10) + '0'
	buf[7] = byte((second)%10) + '0'

	return string(buf[:])
}

func DateTimeStringToTimeUTC(dt string) (time.Time, error) {
	return StringDateTime(dt).ToTime(time.UTC)
}

func TruncateToDate(dt time.Time) time.Time {
	year, month, day := dt.Date()
	return time.Date(year, month, day, 0, 0, 0, 0, dt.Location())
}

func TruncateToDateUTC(dt time.Time) time.Time {
	year, month, day := dt.Date()
	return time.Date(year, month, day, 0, 0, 0, 0, time.UTC)
}

func GetMonthAgoString() string {
	return string(FormatDateIso(time.Now().Add(-24 * 32 * time.Hour)))
}
