package dtutil

import (
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/logger"
)

const BitsPerElem = 63
const MaxDaysInSchedule = 366

func GetAllOneBits(numBits int) int64 {
	result := int64(0)
	for pos := 0; pos < numBits; pos++ {
		result = result | (int64(1) << pos)
	}
	return result
}

var AllOneBits = GetAllOneBits(BitsPerElem)

// Efficiently stores dates as bits
type BitsMask interface {
	AddPos(pos int)
	RemovePos(pos int)
	GetPos(pos int) bool
	IsEmpty() bool
	GetDates(startDateIndex int) []IntDate
	GetDateIndexes(startDateIndex int) []int
	GetDateIndexesFromDate(startDateIndex int, minDate IntDate) []int
	GetFirstDate(startDateIndex int) IntDate
	GetMask() []int64
	AddMask(mask BitsMask)
	RemoveMask(mask BitsMask)
	IntersectWithMask(mask BitsMask)
	ShiftDays(days int)
	GetSingleRange(startDateIndex int) (operatingFrom, operatingUntil StringDate, operatingOnDays int32)
}

// Efficiently manages mask objects for dates
type DateMask interface {
	AddIntDate(date IntDate)
	AddStringDate(date StringDate)
	RemoveIntDate(date IntDate)
	RemoveStringDate(date StringDate)
	AddRange(operatingFrom, operatingUntil StringDate, operatingOn int32)
	RemoveRange(operatingFrom, operatingUntil StringDate, operatingOn int32)
	AddMask(mask DateMask)
	RemoveMask(mask DateMask)
	IntersectWithMask(mask DateMask)
	GetBits() BitsMask
	IsEmpty() bool
	GetDateIndexes() []int
	GetDateIndexesFromDate(date IntDate) []int
	GetDates() []IntDate
	GetFirstDate() IntDate
	ShiftDays(days int)
	GetSingleRange() (operatingFrom, operatingUntil StringDate, operatingOnDays int32)
}

type bitsMaskImpl struct {
	mask []int64
}

type dateMaskImpl struct {
	startDateIndex int
	maxDays        int
	bitsMask       BitsMask
}

func NewBitsMask(maxDays int) BitsMask {
	return &bitsMaskImpl{
		mask: make([]int64, maxDays/BitsPerElem+1),
	}
}

func (bm *bitsMaskImpl) AddPos(pos int) {
	bm.mask[pos/BitsPerElem] = bm.mask[pos/BitsPerElem] | (1 << (pos % BitsPerElem))
}

func (bm *bitsMaskImpl) RemovePos(pos int) {
	bm.mask[pos/BitsPerElem] = bm.mask[pos/BitsPerElem] & (AllOneBits ^ (1 << (pos % BitsPerElem)))
}

func (bm *bitsMaskImpl) GetPos(pos int) bool {
	return (bm.mask[pos/BitsPerElem]>>(pos%BitsPerElem))%2 == 1
}

func (bm *bitsMaskImpl) IsEmpty() bool {
	for _, elem := range bm.mask {
		if elem > 0 {
			return false
		}
	}
	return true
}

func (bm *bitsMaskImpl) GetMask() []int64 {
	return bm.mask
}

func (bm *bitsMaskImpl) AddMask(mask BitsMask) {
	size := len(bm.mask)
	sourceMask := mask.GetMask()
	if size > len(sourceMask) {
		size = len(sourceMask)
	}
	for index := 0; index < size; index++ {
		bm.mask[index] = bm.mask[index] | sourceMask[index]
	}
}

func (bm *bitsMaskImpl) RemoveMask(mask BitsMask) {
	size := len(bm.mask)
	sourceMask := mask.GetMask()
	if size > len(sourceMask) {
		size = len(sourceMask)
	}
	for index := 0; index < size; index++ {
		bm.mask[index] = bm.mask[index] & (AllOneBits ^ sourceMask[index])
	}
}

func (bm *bitsMaskImpl) IntersectWithMask(mask BitsMask) {
	size := len(bm.mask)
	otherMask := mask.GetMask()
	if size > len(otherMask) {
		size = len(otherMask)
	}
	for index := 0; index < size; index++ {
		bm.mask[index] = bm.mask[index] & otherMask[index]
	}
}

func (bm *bitsMaskImpl) ShiftDays(days int) {
	maxPos := BitsPerElem*len(bm.mask) - 1
	if days > 0 {
		for pos := maxPos - days; pos >= 0; pos-- {
			if bm.GetPos(pos) {
				bm.AddPos(pos + days)
			} else {
				bm.RemovePos(pos + days)
			}
		}
		for pos := days - 1; pos >= 0; pos-- {
			bm.RemovePos(pos)
		}
	}
	if days < 0 {
		days = -days
		for pos := days; pos <= maxPos; pos++ {
			if bm.GetPos(pos) {
				bm.AddPos(pos - days)
			} else {
				bm.RemovePos(pos - days)
			}
		}
		for pos := maxPos - days + 1; pos <= maxPos; pos++ {
			bm.RemovePos(pos)
		}
	}
}

func (bm *bitsMaskImpl) GetDates(startDateIndex int) []IntDate {
	result := []IntDate{}
	for _, elem := range bm.mask {
		shift := 0
		for ; elem > 0; elem = elem >> 1 {
			if elem%2 == 1 {
				result = append(result, DateCache.Date(startDateIndex+shift))
			}
			shift++
		}
		startDateIndex += BitsPerElem
	}
	return result
}

func (bm *bitsMaskImpl) GetFirstDate(startDateIndex int) IntDate {
	for _, elem := range bm.mask {
		shift := 0
		for ; elem > 0; elem = elem >> 1 {
			if elem%2 == 1 {
				return DateCache.Date(startDateIndex + shift)
			}
			shift++
		}
		startDateIndex += BitsPerElem
	}
	return 0
}

func (bm *bitsMaskImpl) GetSingleRange(startDateIndex int) (operatingFrom, operatingUntil StringDate, operatingOnDays int32) {
	od := OperatingDays(0)
	for _, elem := range bm.mask {
		shift := 0
		for ; elem > 0; elem = elem >> 1 {
			if elem%2 == 1 {
				currentDate := DateCache.Date(startDateIndex + shift).StringDateDashed()
				if operatingFrom == "" {
					operatingFrom = currentDate
				}
				operatingUntil = currentDate
				od = od.AddDay(DateCache.WeekDay(startDateIndex + shift))
			}
			shift++
		}
		startDateIndex += BitsPerElem
	}
	operatingOnDays = int32(od)
	return
}

func (bm *bitsMaskImpl) GetDateIndexes(startDateIndex int) []int {
	return bm.GetDateIndexesFromDate(startDateIndex, IntDate(-1))
}

func (bm *bitsMaskImpl) GetDateIndexesFromDate(startDateIndex int, minDate IntDate) []int {
	minDateIndex, ok := DateCache.IndexOfIntDate(minDate)
	if !ok {
		minDateIndex = -1
	}
	result := []int{}
	for _, elem := range bm.mask {
		shift := 0
		for ; elem > 0; elem = elem >> 1 {
			if elem%2 == 1 && (minDate < 0 || startDateIndex+shift >= minDateIndex) {
				result = append(result, startDateIndex+shift)
			}
			shift++
		}
		startDateIndex += BitsPerElem
	}
	return result
}

func NewDateMask(startDateIndex, maxDays int) DateMask {
	return &dateMaskImpl{
		startDateIndex: startDateIndex,
		maxDays:        maxDays,
		bitsMask:       NewBitsMask(maxDays),
	}
}

func CloneDateMask(origin DateMask) DateMask {
	oldMask := origin.(*dateMaskImpl)
	newMask := dateMaskImpl{
		startDateIndex: oldMask.startDateIndex,
		maxDays:        oldMask.maxDays,
		bitsMask:       NewBitsMask(oldMask.maxDays),
	}
	newMask.AddMask(origin)
	return &newMask
}

func (dm *dateMaskImpl) AddIntDate(date IntDate) {
	pos := DateCache.IndexOfIntDateP(date) - dm.startDateIndex
	if pos < 0 || pos > dm.maxDays {
		logger.Logger().Warn("Invalid index for the date", log.Int("date", int(date)), log.Int("index", pos))
		return
	}
	dm.bitsMask.AddPos(pos)
}

func (dm *dateMaskImpl) AddStringDate(date StringDate) {
	pos := DateCache.IndexOfStringDateP(date) - dm.startDateIndex
	if pos < 0 || pos > dm.maxDays {
		logger.Logger().Warn("Invalid index for the date", log.String("date", string(date)), log.Int("index", pos))
		return
	}
	dm.bitsMask.AddPos(pos)
}

func (dm *dateMaskImpl) RemoveIntDate(date IntDate) {
	pos := DateCache.IndexOfIntDateP(date) - dm.startDateIndex
	if pos < 0 || pos > dm.maxDays {
		logger.Logger().Warn("Invalid index for the date", log.Int("date", int(date)), log.Int("index", pos))
		return
	}
	dm.bitsMask.RemovePos(pos)
}

func (dm *dateMaskImpl) RemoveStringDate(date StringDate) {
	pos := DateCache.IndexOfStringDateP(date) - dm.startDateIndex
	if pos < 0 || pos > dm.maxDays {
		logger.Logger().Warn("Invalid index for the date", log.String("date", string(date)), log.Int("index", pos))
		return
	}
	dm.bitsMask.RemovePos(pos)
}

func (dm *dateMaskImpl) AddRange(operatingFrom, operatingUntil StringDate, operatingOn int32) {
	indexFrom, okFrom := DateCache.IndexOfStringDate(operatingFrom)
	indexUntil, okUntil := DateCache.IndexOfStringDate(operatingUntil)
	if !okFrom || indexFrom < dm.startDateIndex {
		indexFrom = dm.startDateIndex
	}
	if !okUntil || indexUntil > dm.startDateIndex+dm.maxDays {
		indexUntil = dm.startDateIndex + dm.maxDays
	}
	operatingDays := OperatingDays(operatingOn)
	for index := indexFrom; index <= indexUntil; index++ {
		if operatingDays.OperatesOn(DateCache.WeekDay(index)) {
			dm.bitsMask.AddPos(index - dm.startDateIndex)
		}
	}
}

func (dm *dateMaskImpl) RemoveRange(operatingFrom, operatingUntil StringDate, operatingOn int32) {
	indexFrom, okFrom := DateCache.IndexOfStringDate(operatingFrom)
	indexUntil, okUntil := DateCache.IndexOfStringDate(operatingUntil)
	if !okFrom || indexFrom < dm.startDateIndex {
		indexFrom = dm.startDateIndex
	}
	if !okUntil || indexUntil > dm.startDateIndex+dm.maxDays {
		indexUntil = dm.startDateIndex + dm.maxDays
	}
	operatingDays := OperatingDays(operatingOn)
	for index := indexFrom; index <= indexUntil; index++ {
		if operatingDays.OperatesOn(DateCache.WeekDay(index)) {
			dm.bitsMask.RemovePos(index - dm.startDateIndex)
		}
	}
}

func (dm *dateMaskImpl) GetDates() []IntDate {
	return dm.bitsMask.GetDates(dm.startDateIndex)
}

func (dm *dateMaskImpl) GetFirstDate() IntDate {
	return dm.bitsMask.GetFirstDate(dm.startDateIndex)
}

// Only works for date masks originally created from a single flight pattern
func (dm *dateMaskImpl) GetSingleRange() (operatingFrom, operatingUntil StringDate, operatingOnDays int32) {
	return dm.bitsMask.GetSingleRange(dm.startDateIndex)
}

func (dm *dateMaskImpl) GetBits() BitsMask {
	return dm.bitsMask
}

func (dm *dateMaskImpl) AddMask(mask DateMask) {
	dm.bitsMask.AddMask(mask.GetBits())
}

func (dm *dateMaskImpl) RemoveMask(mask DateMask) {
	dm.bitsMask.RemoveMask(mask.GetBits())
}

func (dm *dateMaskImpl) IntersectWithMask(mask DateMask) {
	dm.bitsMask.IntersectWithMask(mask.GetBits())
}

func (dm *dateMaskImpl) GetDateIndexes() []int {
	return dm.bitsMask.GetDateIndexes(dm.startDateIndex)
}

func (dm *dateMaskImpl) GetDateIndexesFromDate(startDate IntDate) []int {
	return dm.bitsMask.GetDateIndexesFromDate(dm.startDateIndex, startDate)
}

func (dm *dateMaskImpl) IsEmpty() bool {
	return dm.bitsMask.IsEmpty()
}

func (dm *dateMaskImpl) ShiftDays(days int) {
	if days == 0 {
		return
	}
	dm.bitsMask.ShiftDays(days)
}
