package iatacorrection

import (
	"regexp"
	"sort"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/avia/shared_flights/api/pkg/structs"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/logger"
	"a.yandex-team.ru/travel/proto/shared_flights/snapshots"
)

type IataCorrector interface {
	FindFlightBaseCarrier(flightBase *structs.FlightBase, carrierSirena string) int32
	FindFlightPatternCarrier(flightPattern *structs.FlightPattern, flyingCarrierIata, carrierSirena string) int32
	FindCarrier(iataCode, sirenaCode, flightNumber, flyingCarrierIata string, designatedCarrierID int32) int32
	// Rule must be one of the rules given to the constructor of the IataCorrector object
	ApplyRule(rule *snapshots.TIataCorrectionRule, iataCode, sirenaCode, flightNumber, flyingCarrierIata string, designatedCarrierID int32) int32
	GetRules() []*snapshots.TIataCorrectionRule
}

type iataCorrectorImpl struct {
	Rules              []*snapshots.TIataCorrectionRule
	DesignatedCarriers map[int32]string
	Carriers           map[string]int32
	Regexps            map[int32]*regexp.Regexp
}

func NewIataCorrector(rules []*snapshots.TIataCorrectionRule, designatedCarriers map[int32]string, carriers map[string]int32) IataCorrector {
	filteredRules := make([]*snapshots.TIataCorrectionRule, 0)
	regexps := make(map[int32]*regexp.Regexp)
	for _, rule := range rules {
		if rule.CarrierId == 0 {
			logger.Logger().Error("Invalid IATA correction rule, no CarrierId", log.Reflect("rule", rule))
			continue
		}
		filteredRules = append(filteredRules, rule)

		if len(rule.FlightNumberRegex) > 0 {
			re, err := regexp.Compile(rule.FlightNumberRegex)
			if err != nil {
				logger.Logger().Error(
					"Invalid flight regexp in the rule",
					log.Reflect("rule", rule),
					log.Error(err),
				)
			} else {
				regexps[rule.Id] = re
			}
		}
	}
	sort.Slice(filteredRules, func(i, j int) bool {
		return filteredRules[i].Priority >= filteredRules[j].Priority
	})
	if designatedCarriers == nil {
		designatedCarriers = make(map[int32]string)
	}
	if carriers == nil {
		carriers = make(map[string]int32)
	}
	return &iataCorrectorImpl{
		Rules:              filteredRules,
		DesignatedCarriers: designatedCarriers,
		Carriers:           carriers,
		Regexps:            regexps,
	}
}

func (s *iataCorrectorImpl) FindFlightBaseCarrier(flightBase *structs.FlightBase, carrierSirena string) int32 {
	result := s.FindCarrier(
		flightBase.OperatingCarrierCode,
		carrierSirena,
		flightBase.OperatingFlightNumber,
		flightBase.FlyingCarrierIata,
		flightBase.DesignatedCarrier,
	)
	if result == flightBase.OperatingCarrier {
		return 0
	}
	return result
}

func (s *iataCorrectorImpl) FindFlightPatternCarrier(flightPattern *structs.FlightPattern, flyingCarrierIata, carrierSirena string) int32 {
	result := s.FindCarrier(
		flightPattern.MarketingCarrierCode,
		carrierSirena,
		flightPattern.MarketingFlightNumber,
		flyingCarrierIata,
		flightPattern.DesignatedCarrier,
	)
	if result == flightPattern.MarketingCarrier {
		return 0
	}
	return result
}

func (s *iataCorrectorImpl) FindCarrier(iataCode, sirenaCode, flightNumber, flyingCarrierIata string, designatedCarrierID int32) int32 {
	for _, rule := range s.Rules {
		carrierID := s.ApplyRule(rule, iataCode, sirenaCode, flightNumber, flyingCarrierIata, designatedCarrierID)
		if carrierID != 0 {
			return carrierID
		}
	}
	return 0
}

// Rule must be one of the rules given to the constructor of the IataCorrector object
func (s *iataCorrectorImpl) ApplyRule(
	rule *snapshots.TIataCorrectionRule,
	iataCode, sirenaCode, flightNumber, flyingCarrierIata string,
	designatedCarrierID int32) int32 {
	// After RASPTICKETS-18657 we heavily depend here on the fact that Iata, Sirena and Icao codes are non-intersecting sets of codes
	if len(rule.MarketingCarrierIata) > 0 && rule.MarketingCarrierIata != iataCode {
		return 0
	}

	if len(rule.CarrierSirena) > 0 && rule.CarrierSirena != sirenaCode {
		return 0
	}

	hasRegexp := false
	if len(rule.FlightNumberRegex) > 0 {
		hasRegexp = true
		re, ok := s.Regexps[rule.Id]
		if !ok {
			return 0
		}
		if !re.MatchString(flightNumber) {
			return 0
		}
	}

	if len(rule.DesignatedCarrier) > 0 {
		designatedCarrier, ok := s.DesignatedCarriers[designatedCarrierID]
		if !ok || designatedCarrier != rule.DesignatedCarrier {
			return 0
		}
	}

	if len(rule.FlyingCarrierIata) > 0 {
		if rule.FlyingCarrierIata == "*" {
			flyingCarrierID, ok := s.Carriers[flyingCarrierIata]
			if !ok {
				return 0
			}
			return flyingCarrierID
		} else if rule.FlyingCarrierIata != flyingCarrierIata {
			return 0
		}
	}

	if len(rule.MarketingCarrierIata) > 0 || len(rule.CarrierSirena) > 0 || hasRegexp || len(rule.DesignatedCarrier) > 0 || len(rule.FlyingCarrierIata) > 0 {
		return rule.CarrierId
	}

	return 0
}

func (s *iataCorrectorImpl) GetRules() []*snapshots.TIataCorrectionRule {
	return s.Rules
}
