# coding=utf-8
from __future__ import unicode_literals

from datetime import datetime


# Matches dates with day masks like 12357, used for finding out whether flight leg operates on a certain day
class DateMatcher(object):

    def __init__(self, date_index):
        self._date_index = date_index

    def operates_on_date(self, date, bits):
        return DateMatcher.operates_on(self._date_index.get_weekday_by_date(date), bits)

    def operates_on_datestr(self, date_str, bits):
        if '.' in date_str:
            date_str = date_str.replace('.', '-')
        return self.operates_on_date(datetime.strptime(date_str, '%Y-%m-%d').date(), bits)

    # Here index is a difference in days between the base date for the self._date_index and the date to test operations on
    def operates_on_index(self, index, bits):
        return DateMatcher.operates_on(self._date_index.get_weekday_by_index(index), bits)

    # Returns the maximum possible (from_date_str, until_date_str, days_mask) that matches both input triples
    def intersect(self, from_date_str1, until_date_str1, days_mask1, from_date_str2, until_date_str2, days_mask2):
        index_from1 = self._date_index.get_index_for_date_str(from_date_str1)
        index_until1 = self._date_index.get_index_for_date_str(until_date_str1)
        bits1 = DateMatcher.get_bits(days_mask1)

        index_from2 = self._date_index.get_index_for_date_str(from_date_str2)
        index_until2 = self._date_index.get_index_for_date_str(until_date_str2)
        bits2 = DateMatcher.get_bits(days_mask2)

        if index_until1 < index_from2 or index_from1 > index_until2:
            return None

        common_bits = bits1 & bits2
        min_index = index_until1 + 1
        max_index = index_from1 - 1
        for day_index in range(index_from1, index_until1+1):
            if day_index < index_from2 or day_index > index_until2:
                continue
            if self.operates_on_index(day_index, common_bits):
                if min_index > day_index:
                    min_index = day_index
                if max_index < day_index:
                    max_index = day_index

        if min_index > max_index:
            return None
        days_mask = DateMatcher.get_days_mask(common_bits)
        if not days_mask:
            return None
        return (self._date_index.get_date_str(min_index), self._date_index.get_date_str(max_index), days_mask)

    @staticmethod
    def get_bits(days_mask):
        result = 0
        while days_mask > 0:
            result = result | (1 << (days_mask % 10))
            days_mask = days_mask // 10
        return result

    @staticmethod
    def get_bits_from_string(days_mask_string):
        days_mask_string = days_mask_string.replace('0', '')
        if not days_mask_string:
            return 0
        return DateMatcher.get_bits(int(days_mask_string))

    @staticmethod
    def get_days_mask(bits):
        bits = bits >> 1
        result = 0
        count = 0
        while bits > 0:
            count += 1
            if bits % 2:
                result = result * 10 + count
            bits = bits >> 1
        return result

    # weekday is 1-based
    @staticmethod
    def operates_on(weekday, bits):
        return weekday and (bits >> weekday) % 2 == 1

    # weekday is 1-based
    @staticmethod
    def operates_on_days_mask(weekday, days_mask):
        return DateMatcher.operates_on(weekday, DateMatcher.get_bits(days_mask))
