# coding=utf-8
from __future__ import unicode_literals

from datetime import datetime, timedelta
import json
import retrying
import traceback
import uuid

from sqlalchemy import and_

from travel.avia.shared_flights.lib.python.db_models.base import CustomEncoder
from travel.avia.shared_flights.lib.python.db_models.db_lock import DbLock


class DbLockHandleException(Exception):
    pass


class DbLockHandle(object):

    # timeout is in seconds
    def __init__(self, session, lock_type, logger, force_lock=False, timeout=30, retry_timeout=0):
        self._session = session
        self._lock_type = lock_type
        self._lock_token = str(uuid.uuid4())
        self._logger = logger
        self._force_lock = force_lock
        self._timeout = timeout
        self._lock_expires_at = None
        if not retry_timeout:
            retry_timeout = timeout
        retryable = retrying.retry(stop_max_delay=retry_timeout*1000, wait_fixed=10000)
        self._get_lock_with_retry = retryable(self._get_lock)

    def __enter__(self):
        self._session.expunge_all()
        self._logger.info('Obtaining a db lock for {}'.format(self._lock_type.value))
        db_lock = self._get_lock_with_retry()
        self._lock_expires_at = datetime.now() + timedelta(seconds=self._timeout)
        db_lock.lock_token = self._lock_token
        db_lock.expires_at = self._lock_expires_at
        self._session.commit()
        self._logger.info('Obtained a new lock: {}.'.format(self))

        return self

    def _get_lock(self):
        try:
            db_lock = self._session.query(DbLock).filter_by(lock_type=self._lock_type.value).with_for_update(skip_locked=True).one()
        except:
            raise DbLockHandleException('Unable to obtain a new lock.')

        if db_lock.expires_at and db_lock.expires_at > datetime.now() and not self._force_lock:
            self._session.commit()  # to release the lock
            error_text = 'Unable to obtain a new lock due to already existing one: {}.'.format(
                json.dumps(db_lock, cls=CustomEncoder)
            )
            self._session.expunge_all()
            raise DbLockHandleException(error_text)
        return db_lock

    def __exit__(self, exc_type, exc_value, tb):
        if not self._lock_expires_at:
            self._logger.info('No lock of type {} to discard.'.format(self._lock_type.value))
        elif self._lock_expires_at < datetime.now():
            self._logger.info('Lock {} has been already discarded.'.format(self))
        else:
            self._session.execute('SET LOCAL lock_timeout = 30000')
            db_lock = self._session.query(DbLock).filter(and_(
                DbLock.lock_type == self._lock_type.value,
                DbLock.lock_token == self._lock_token,
            )).with_for_update(skip_locked=False).one_or_none()

            if db_lock:
                db_lock.expires_at = datetime.now()
                self._session.commit()
                self._logger.info('Discarded lock: {}.'.format(self))
                self._session.expunge_all()
            else:
                self._logger.info('No lock has been found for a discard')

        if exc_type is not None:
            traceback.print_exception(exc_type, exc_value, tb)
            return False

        return True

    def __bool__(self):
        return self._lock_expires_at and self._lock_expires_at < datetime.now()

    def __repr__(self):
        return '<Lock {} expires_at={} token={} timeout={} />'.format(
            self._lock_type.value, self._lock_expires_at, self._lock_token, self._timeout
        )
