# coding: utf8
from __future__ import absolute_import, unicode_literals

import datetime
from enum import Enum

import sqlalchemy as sa
import sqlalchemy.orm as sorm

from travel.avia.library.python.enum import Language, NationalVersion
from travel.avia.shared_flights.lib.python.db_models.base import Base
from travel.avia.shared_flights.lib.python.db_models.carrier import Carrier
from travel.avia.shared_flights.lib.python.db_models.station import Station


class ForceEnable(Enum):
    AS_SCHEDULED = 'As scheduled'
    FORCE_BAN = 'Force ban'
    DISABLED = 'Disabled'

    def __str__(self):
        return self.name

    def __repr__(self):
        return self.name + ' ' + self.value


class FlightBaseSource(Enum):
    UNKNOWN = 'None'
    INNOVATA = 'Innovata'
    SIRENA = 'Sirena'
    APM = 'APM'
    DOP = 'Dop'
    AMADEUS = 'Amadeus'

    def __str__(self):
        return self.name

    def __repr__(self):
        return self.name + ' ' + self.value


class Blacklist(Base):
    __versioned__ = {}
    __tablename__ = 'blacklist'

    id = sa.Column(sa.BigInteger, primary_key=True, autoincrement=True)

    country_from_id = sa.Column(sa.Integer)

    settlement_from_id = sa.Column(sa.Integer)

    station_from_id = sa.Column(sa.Integer, sa.ForeignKey(Station.id))
    station_from = sorm.relationship(
        Station, foreign_keys=[station_from_id],
    )

    country_to_id = sa.Column(sa.Integer)

    settlement_to_id = sa.Column(sa.Integer)

    station_to_id = sa.Column(sa.Integer, sa.ForeignKey(Station.id))
    station_to = sorm.relationship(
        Station, foreign_keys=[station_to_id],
    )

    carrier_id = sa.Column(sa.Integer, sa.ForeignKey(Carrier.id))
    carrier = sorm.relationship(
        Carrier, foreign_keys=[carrier_id],
    )

    flight_number = sa.Column(sa.String)

    flight_date_from = sa.Column(sa.Date)

    flight_date_to = sa.Column(sa.Date)

    national_version = sa.Column(sa.Enum(NationalVersion))
    language = sa.Column(sa.Enum(Language))

    reason = sa.Column(sa.String, nullable=False)

    force = sa.Column(
        sa.Enum(ForceEnable),
        nullable=False,
        default=ForceEnable.AS_SCHEDULED,
        server_default=ForceEnable.AS_SCHEDULED.name,
    )

    active_since = sa.Column(sa.DateTime)
    active_until = sa.Column(sa.DateTime)

    created_at = sa.Column(sa.DateTime, server_default=sa.func.now(), default=datetime.datetime.utcnow, nullable=False)
    updated_at = sa.Column(sa.DateTime, onupdate=datetime.datetime.utcnow, nullable=True)

    source = sa.Column(
        sa.Enum(FlightBaseSource),
        nullable=False,
        default=FlightBaseSource.UNKNOWN,
        server_default=FlightBaseSource.UNKNOWN.name,
    )
