package db

import (
	"context"
	"crypto/tls"
	"fmt"
	"os"
	"strings"

	"github.com/jackc/pgconn"
	pgx "github.com/jackc/pgx/v4"
	"github.com/jackc/pgx/v4/log/zapadapter"
	"github.com/jackc/pgx/v4/pgxpool"
	"go.uber.org/zap"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/logger"
	"a.yandex-team.ru/travel/avia/shared_flights/status_importer/internal/settings"
)

func newConnectionPool(hosts string, config *settings.PG, rw bool) (conn *pgxpool.Pool, err error) {
	logLevel := logLevelFromEnvOrDefault(pgx.LogLevelError)
	var targetSessionAttrs string
	if rw {
		targetSessionAttrs = "read-write"
	} else {
		targetSessionAttrs = "any"
	}
	var pgxLogger pgx.Logger
	if config.Logging {
		pgxLogger = zapadapter.NewLogger(logger.Logger().L.WithOptions(zap.AddCallerSkip(0)))
	}

	poolConfig, err := pgxpool.ParseConfig(
		strings.Join([]string{
			"user=" + config.User,
			"password=" + config.Password,
			"host=" + hosts,
			"port=" + fmt.Sprintf("%v", config.Port),
			"dbname=" + config.DataBase,
			"sslmode=verify-ca",
			"pool_max_conns=20",
			"connect_timeout=10",
			"keepalives_interval=300",
			"target_session_attrs=" + targetSessionAttrs,
			"statement_cache_mode=prepare",
		}, " "))
	if err != nil {
		return nil, err
	}
	poolConfig.ConnConfig.TLSConfig = &tls.Config{
		InsecureSkipVerify: true,
	}
	poolConfig.ConnConfig.PreferSimpleProtocol = strings.ToLower(os.Getenv("PGX_PREFER_SIMPLE_PROTOCOL")) != "false"
	poolConfig.ConnConfig.Logger = pgxLogger
	poolConfig.ConnConfig.LogLevel = logLevel
	poolConfig.ConnConfig.AfterConnect = func(ctx context.Context, conn *pgconn.PgConn) error {
		logger.Logger().Info("Connection to database established", log.Reflect("connection", conn))
		return nil
	}
	logger.Logger().Info("Connecting to database pool")
	return pgxpool.ConnectConfig(context.Background(), poolConfig)
}
