package handlers

import (
	"net/http"

	"github.com/jackc/pgx/v4/pgxpool"

	"a.yandex-team.ru/travel/avia/shared_flights/status_importer/internal/db"
	"a.yandex-team.ru/travel/avia/shared_flights/status_importer/internal/shutdown"
)

func NewPingHandler(config PingHandlerConfig) *pingHandler {
	handler := &pingHandler{Cluster: config.Cluster}
	config.ShutdownRegistrar.AddShutdownable(shutdown.ShutdownFunc(func() { handler.ShuttingDown = true }))
	return handler
}

type shutdownRegistrar interface {
	AddShutdownable(shutdownable shutdown.Shutdownable)
}

type PingHandlerConfig struct {
	Cluster           db.Cluster
	ShutdownRegistrar shutdownRegistrar
}

type PingResponse struct {
	PGConnPoolStatRW interface{}
	PGConnPoolStatRO interface{}
	ShuttingDown     bool
}

type pingHandler struct {
	Cluster      db.Cluster
	ShuttingDown bool
}

func stat(s *pgxpool.Stat) map[string]interface{} {
	return map[string]interface{}{
		"AcquireCount":         s.AcquireCount(),
		"AcquiredConns":        s.AcquiredConns(),
		"CanceledAcquireCount": s.CanceledAcquireCount(),
		"ConstructingConns":    s.ConstructingConns(),
		"EmptyAcquireCount":    s.EmptyAcquireCount(),
		"IdleConns":            s.IdleConns(),
		"MaxConns":             s.MaxConns(),
		"TotalConns":           s.TotalConns(),
	}
}

func (h *pingHandler) ServeHTTP(w http.ResponseWriter, _ *http.Request) {
	var response Response
	defer response.WriteJSON(w)
	response.Message = PingResponse{
		PGConnPoolStatRW: stat(h.Cluster.RW().Stat()),
		PGConnPoolStatRO: stat(h.Cluster.RO().Stat()),
		ShuttingDown:     h.ShuttingDown,
	}
	if h.ShuttingDown {
		response.Status = http.StatusServiceUnavailable
	}
}
