package handlers

import (
	"bytes"
	"encoding/json"
	"net/http"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/logger"
	"a.yandex-team.ru/travel/avia/shared_flights/status_importer/internal/status"
)

type StatusHandlerConfig struct {
	Updater status.Updater
}

type StatusHandler struct {
	updater status.Updater
}

func NewStatusHandler(config *StatusHandlerConfig) *StatusHandler {
	return &StatusHandler{updater: config.Updater}
}

func (h *StatusHandler) UpdateStatus(w http.ResponseWriter, r *http.Request) {
	logger.Logger().Info("Got an update status request")
	var (
		statusesToParse status.StatusesToParse
		response        Response
	)
	defer response.WriteJSON(w)

	buf := new(bytes.Buffer)
	_, _ = buf.ReadFrom(r.Body)

	if err := json.Unmarshal(buf.Bytes(), &statusesToParse); err != nil {
		logger.Logger().Error("StatusHandler: could not unmarshal status.",
			log.ByteString("buf", buf.Bytes()),
			log.Error(err),
		)
		response.Error = xerrors.Errorf("statusHandler: could not unmarshal status: %w", err)
		response.Status = http.StatusBadRequest
		return
	}
	var statuses status.Statuses
	for index := 0; index < len(statusesToParse); index++ {
		statuses = append(statuses, &statusesToParse[index])
	}
	statuses.InitReceivedAt()
	processing := status.ProcessingUnit{
		Statuses: statuses,
		Finished: make(chan error, 2),
	}
	if err := h.updater.Update(processing, true); err != nil {
		logger.Logger().Error("StatusHandler: error updating statuses",
			log.Error(err),
		)
		response.Error = xerrors.Errorf("statusHandler: error updating statuses: %w", err)
		response.Status = http.StatusInternalServerError
		return
	}
	response.Message = "OK"
}
