package logbroker

import (
	"context"
	"encoding/json"
	"time"

	"github.com/cenkalti/backoff/v4"

	"a.yandex-team.ru/kikimr/public/sdk/go/persqueue"
	"a.yandex-team.ru/kikimr/public/sdk/go/ydb"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/avia/library/go/logbroker"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/logger"
	appMetrics "a.yandex-team.ru/travel/avia/shared_flights/status_importer/internal/metrics"
	"a.yandex-team.ru/travel/avia/shared_flights/status_importer/internal/settings"
	"a.yandex-team.ru/travel/avia/shared_flights/status_importer/internal/shutdown"
	"a.yandex-team.ru/travel/avia/shared_flights/status_importer/internal/status"
	"a.yandex-team.ru/travel/library/go/metrics"
)

type StatusHandlerConfig struct {
	Updater   status.Updater
	Logbroker settings.Logbroker
}

type StatusesHandler struct {
	config       StatusHandlerConfig
	statusesChan chan status.Statuses
}

func NewStatusesHandler(config StatusHandlerConfig) *StatusesHandler {
	return &StatusesHandler{
		config:       config,
		statusesChan: make(chan status.Statuses),
	}
}

// Run is a blocking function that start logbroker endpoint and runs status updater loop
func (h *StatusesHandler) Run(shutdownEventDispatcher shutdown.ShutdownRegistrar) {
	if h.config.Logbroker.StatusUpdater.Disable {
		logger.Logger().Warn("Logbroker status updater is disabled by environment")
		return
	}
	ctx, cancelContext := context.WithCancel(context.Background())
	for _, endpoint := range h.config.Logbroker.ReadEndpoints {
		go h.maintainSingleEndpoint(ctx, endpoint)
	}
	shutdownEventDispatcher.AddShutdownable(shutdown.ShutdownFunc(cancelContext))

	for statusBatch := range h.statusesChan {
		unit := status.ProcessingUnit{
			Statuses: statusBatch,
			Finished: make(chan error, 2),
		}

		_ = h.config.Updater.Update(unit, false)
		go func(errorResult chan error) {
			if err := <-errorResult; err != nil {
				logFn := logger.Logger().Error
				if xerrors.As(err, &status.NotImportantError{}) {
					logFn = logger.Logger().Warn
				}
				logFn(
					"Cannot update statuses: "+err.Error(),
					log.Error(err),
				)
			}
		}(unit.Finished)

	}
	logger.Logger().Fatal("Someone closed status channel. This should not have happened")
}

func (h *StatusesHandler) maintainSingleEndpoint(ctx context.Context, endpoint string) {
	defer func() {
		if e := recover(); e != nil {
			logger.Logger().Error("Unexpected logbroker panic")
		}
	}()
	logbrokerConfig := persqueue.ReaderOptions{
		Endpoint:              endpoint,
		Credentials:           ydb.AuthTokenCredentials{AuthToken: h.config.Logbroker.Token},
		Consumer:              h.config.Logbroker.StatusUpdater.Consumer,
		Topics:                []persqueue.TopicInfo{{Topic: h.config.Logbroker.StatusUpdater.Topic}},
		MaxReadSize:           8 * 1024, // bytes
		MaxReadMessagesCount:  1,
		DecompressionDisabled: true,
		RetryOnFailure:        true,
	}
	cancelledExternally := false
	for {
		backoffErr := backoff.RetryNotify(
			func() error {
				var maintainErr error
				var reader *logbroker.Reader

				select {
				case <-ctx.Done():
					cancelledExternally = true
					return backoff.Permanent(
						xerrors.Errorf(
							"maintainSingleEndpoint %s: cancel: %w",
							endpoint,
							ctx.Err(),
						),
					)
				default:
				}
				if maintainErr = backoff.RetryNotify(
					func() error {
						var lbErr error
						reader, lbErr = logbroker.NewReader(
							ctx,
							logbrokerConfig,
							logbroker.WithOnBatchReceived(h.onBatchReceived),
							logbroker.WithOnMessage(h.onMessage),
						)
						return lbErr
					},
					backoff.NewConstantBackOff(10*time.Second),
					func(err error, delay time.Duration) {
						logger.Logger().Error(
							"Unable to start a new reader. Trying to reconnect",
							log.Any("endpoint", endpoint),
							log.Duration("delay", delay),
							log.Error(err),
						)
					},
				); maintainErr != nil {
					return maintainErr
				}
				logger.Logger().Info(
					"Connection to logbroker established",
					log.String("endpoint", endpoint),
				)
				maintainErr = reader.Read()
				return xerrors.Errorf("maintain: %w", maintainErr)
			},
			backoff.NewConstantBackOff(10*time.Second),
			func(err error, delay time.Duration) {
				select {
				case <-ctx.Done():
					cancelledExternally = true
				default:
				}
				if cancelledExternally {
					return
				}
				logger.Logger().Error(
					"Error maintaning logbroker endpoint. Trying to reconnect",
					log.String("endpoint", endpoint),
					log.Duration("delay", delay),
					log.Error(err),
				)
			},
		)
		if cancelledExternally {
			logger.Logger().Info("Logbroker worker is cancelled from the outside")
			return
		}
		logger.Logger().Error("Backoff failed", log.Error(backoffErr))
		time.Sleep(time.Minute)
	}
}

func (h *StatusesHandler) onBatchReceived(batch persqueue.MessageBatch) {
	logger.Logger().Debug(
		"Received batch",
		log.Any("topic", batch.Topic),
		log.Any("partition", batch.Partition),
		log.Any("messages", len(batch.Messages)),
	)
}

func (h *StatusesHandler) onMessage(message persqueue.ReadMessage) {
	decompressedMessage, err := h.decompressMessage(message)
	var statuses status.Statuses
	if err == nil {
		statuses, err = h.unmarshalStatuses(decompressedMessage.Msg.Data)
	}
	if err == nil {
		h.onStatuses(decompressedMessage, statuses)
		return
	}
	logger.Logger().Error(
		"Cannot read or handle status message",
		log.Error(err),
		log.Any("message", message),
	)
	metrics.GlobalAppMetrics().GetOrCreateCounter(
		appMetrics.LogbrokerReceive,
		nil,
		appMetrics.FailureRate,
	).Inc()
}

func (h *StatusesHandler) decompressMessage(message persqueue.ReadMessage) (*persqueue.ReadMessageOrError, error) {
	decompressorInput := make(chan persqueue.ReadMessage, 1)
	decompressorOutput := persqueue.Decompress(decompressorInput)
	decompressorInput <- message
	decompressedMessage := <-decompressorOutput

	if decompressedMessage.Err != nil {
		return nil, xerrors.Errorf("cannot read decompressed message: %w", decompressedMessage.Err)
	}
	logger.Logger().Debug(
		"Received message",
		log.Any("seqNo", decompressedMessage.Msg.SeqNo),
		log.Any("source-id", string(decompressedMessage.Msg.SourceID)),
		log.Any("created", decompressedMessage.Msg.CreateTime),
		log.Any("written", decompressedMessage.Msg.WriteTime),
	)
	return &decompressedMessage, nil
}

func (h *StatusesHandler) unmarshalStatuses(data []byte) (status.Statuses, error) {
	var statuses status.Statuses
	err := json.Unmarshal(data, &statuses)
	if err != nil {
		return nil, xerrors.Errorf("cannot unmarshal status: %s: %w", string(data), err)
	}
	return statuses, nil
}

func (h *StatusesHandler) onStatuses(decompressedMessage *persqueue.ReadMessageOrError, statuses status.Statuses) {
	if len(statuses) == 0 {
		logger.Logger().Warn("no statuses", log.Any("data", string(decompressedMessage.Msg.Data)))
	} else {
		h.statusesChan <- statuses
	}
	metrics.GlobalAppMetrics().GetOrCreateCounter(
		appMetrics.LogbrokerReceive,
		nil,
		appMetrics.SuccessRate,
	).Inc()
}
