package carriercache

import (
	"context"
	"sort"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/logger"
	"a.yandex-team.ru/travel/avia/shared_flights/status_importer/internal/objects/cache"
	"a.yandex-team.ru/travel/avia/shared_flights/status_importer/internal/objects/model"
)

type tCarrier struct {
	*innerCache
	config *Config
}

type innerCache struct {
	byID   map[int64]model.Carrier
	byCode map[string][]model.Carrier
}

type CarrierChanProvider interface {
	All() (chan *model.Carrier, error)
}

type Config struct {
	CarrierProvider CarrierChanProvider
	UpdateInterval  time.Duration
}

func New(config *Config) *tCarrier {
	c := &tCarrier{
		config: config,
	}
	cache.RunUpdater(&cache.UpdaterConfig{
		Name:          "carrier",
		Renewer:       c.updater,
		RenewInterval: c.config.UpdateInterval,
		Ctx:           context.Background(),
	})
	return c
}

func (c *tCarrier) updater() error {
	newCache := &innerCache{
		make(map[int64]model.Carrier),
		make(map[string][]model.Carrier),
	}
	carrierChan, err := c.config.CarrierProvider.All()
	if err != nil {
		return xerrors.Errorf("carrier cache updater: %w", err)
	}
	carriersCount := 0
	for carrier := range carrierChan {
		carriersCount++
		newCache.addByID(*carrier)
		newCache.addByCode(*carrier)
	}
	if carriersCount == 0 {
		return xerrors.Errorf("unable to fetch carriers from the database")
	} else {
		logger.Logger().Info("fetched carriers", log.Int("count", carriersCount))
	}
	newCache.sortCodesByPriority()
	c.innerCache = newCache
	return nil
}

func (c *innerCache) ByID(key int64) model.Carrier {
	return c.byID[key]
}

func (c *innerCache) ByCode(key string) []model.Carrier {
	return c.byCode[key]
}

func (c *innerCache) addByID(carrier model.Carrier) {
	c.byID[carrier.ID] = carrier
}

func (c *innerCache) addByCode(carrier model.Carrier) {
	c.appendCodeIfNotExist(carrier, carrier.Iata)
	c.appendCodeIfNotExist(carrier, carrier.Sirena)
	c.appendCodeIfNotExist(carrier, carrier.Icao)
	c.appendCodeIfNotExist(carrier, carrier.IcaoRU)
}

func (c *innerCache) appendCodeIfNotExist(carrier model.Carrier, code string) {
	if len(code) == 0 {
		return
	}
	var carriers = c.byCode[code]
	for _, existingCarrier := range carriers {
		if existingCarrier.ID == carrier.ID {
			return
		}
	}
	c.byCode[code] = append(carriers, carrier)
}

func (c *innerCache) sortCodesByPriority() {
	for _, carriers := range c.byCode {
		sort.SliceStable(carriers, func(i, j int) bool { return carriers[i].Priority > carriers[j].Priority })
	}
}
