package legcache

import (
	"sync"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/direction"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/dtutil"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/logger"
	"a.yandex-team.ru/travel/avia/shared_flights/status_importer/internal/objects"
	"a.yandex-team.ru/travel/avia/shared_flights/status_importer/internal/objects/model"
)

func (c *innerCache) fillCacheWindow(dateWindow time.Duration) cacheWindow {
	now := time.Now().UTC()
	yy, mm, dd := now.Add(-dateWindow).Date()
	leftBound := time.Date(yy, mm, dd, 0, 0, 0, 0, time.UTC)
	yy, mm, dd = now.Add(dateWindow).Date()
	rightBound := time.Date(yy, mm, dd, 0, 0, 0, 0, time.UTC)
	c.cacheWindow = cacheWindow{leftBound, rightBound}
	return c.cacheWindow
}

func (c *innerCache) loadFlightsInDateWindow(dateWindow time.Duration, flights objects.FlightProvider) error {
	cacheWindow := c.fillCacheWindow(dateWindow)
	flightChan := make(chan *model.FlightPatternRecord)
	var wg sync.WaitGroup
	wg.Add(1)
	var counter int64
	go func() {
		defer wg.Done()
		var lastPrinterCounter int64 = 0
		for flight := range flightChan {
			counter += c.loadFlightBasePatternRecordToCache(flight, cacheWindow.leftBound, cacheWindow.rightBound)
			if counter-lastPrinterCounter > 100000 {
				logger.Logger().Info("Loading flights", log.Int64("counter", counter))
				lastPrinterCounter = counter
			}
		}
	}()

	err := flights.Flights(flightChan, cacheWindow.leftBound, cacheWindow.rightBound)
	if err != nil {
		return xerrors.Errorf("cannot load flight bulk: %w", err)
	}
	wg.Wait()
	logger.Logger().Info(
		"Done loading cache for flight legs",
		log.Time("from", cacheWindow.leftBound),
		log.Time("to", cacheWindow.rightBound),
		log.Int64("records total", counter),
	)
	return nil
}

func (c *innerCache) loadFlightBasePatternRecordToCache(
	flight *model.FlightPatternRecord, leftBound time.Time, rightBound time.Time,
) int64 {
	var cacheItemsLoaded int64
	for dt := leftBound; !dt.After(rightBound); dt = dt.Add(time.Hour * 24) {
		if !flight.IsOperatingAt(dt) {
			continue
		}
		leg := flight.LegNumber
		arrivalDate := dt.AddDate(0, 0, flight.ArrivalDayShift)
		item := cacheItem{
			leg:           leg,
			departureDate: dtutil.FormatDateIso(dt),
		}
		c.putFlightLegBatch(
			flight.CarrierID,
			flight.Number,
			dtutil.FormatDateIso(dt), // departure date as key for departure
			direction.DEPARTURE,
			flight.DepartureStation,
			item,
		)
		c.putFlightLegBatch(
			flight.CarrierID,
			flight.Number,
			dtutil.FormatDateIso(arrivalDate), //arrival date as key for arrival
			direction.ARRIVAL,
			flight.ArrivalStation,
			item,
		)
		cacheItemsLoaded += 2
	}
	return cacheItemsLoaded
}
