package legcache

import (
	"sync"
	"time"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/direction"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/dtutil"
	"a.yandex-team.ru/travel/avia/shared_flights/status_importer/internal/objects/model"
)

func (c *flightLegCache) shouldBeInCache(flightDay dtutil.StringDate) bool {
	flightDayParsed, err := time.Parse("2006-01-02", string(flightDay))
	if err != nil || c.leftBound.After(flightDayParsed) || c.rightBound.Before(flightDayParsed) {
		return false
	}
	return true
}

func (c *flightLegCache) FlightLeg(carrierID int64, flightNumber string, flightDay dtutil.StringDate, direction direction.Direction, stationID model.StationID) (leg int16, departureDate dtutil.StringDate, err error) {
	var (
		ok   bool
		item cacheItem
	)
	if carrierID == 0 {
		return 0, "", nil
	}
	item, ok = c.flightLeg(carrierID, flightNumber, flightDay, direction, stationID)
	if !ok {
		if c.shouldBeInCache(flightDay) { // if it should be, but it's not - it's not in DB either
			return 0, "", nil
		}
		leg, departureDate, err = c.config.FlightLegProvider.FlightLeg(carrierID, flightNumber, flightDay, direction, stationID)
		if err != nil {
			return leg, departureDate, xerrors.Errorf("flight-leg: cache miss: %w", err)
		}
		item = cacheItem{leg: leg, departureDate: departureDate}
		c.putFlightLegOverlay(
			carrierID, flightNumber, flightDay, direction, stationID,
			item,
		)
	}
	return item.leg, item.departureDate, err

}

type flightLegCache struct {
	*innerCache
	config *Config
}

type innerCache struct {
	db           map[cacheKey]cacheItem
	overlay      map[cacheKey]cacheItem
	overlayMutex sync.RWMutex
	cacheWindow
}

type cacheWindow struct {
	leftBound  time.Time
	rightBound time.Time
}

type cacheKey struct {
	carrierID    int64
	flightNumber string
	Date         dtutil.StringDate
	direction    direction.Direction
	stationID    model.StationID
}

type cacheItem struct {
	leg           int16
	departureDate dtutil.StringDate
}

func (c *innerCache) flightLeg(
	carrierID int64,
	flightNumber string,
	flightDay dtutil.StringDate,
	direction direction.Direction,
	stationID model.StationID,
) (cacheItem, bool) {
	var (
		ok bool
		v  cacheItem
	)
	k := cacheKey{
		carrierID:    carrierID,
		flightNumber: flightNumber,
		Date:         flightDay,
		direction:    direction,
		stationID:    stationID,
	}
	v, ok = c.db[k]
	if !ok {
		c.overlayMutex.RLock()
		defer c.overlayMutex.RUnlock()
		v, ok = c.overlay[k]
	}
	return v, ok
}

func (c *innerCache) putFlightLegBatch(
	carrierID int64, flightNumber string, Date dtutil.StringDate,
	direction direction.Direction, stationID model.StationID,
	item cacheItem,
) {
	c.db[cacheKey{
		carrierID:    carrierID,
		flightNumber: flightNumber,
		Date:         Date,
		direction:    direction,
		stationID:    stationID,
	}] = item
}

func (c *innerCache) putFlightLegOverlay(
	carrierID int64, flightNumber string, Date dtutil.StringDate,
	direction direction.Direction, stationID model.StationID,
	item cacheItem,
) {
	c.overlayMutex.Lock()
	defer c.overlayMutex.Unlock()
	c.overlay[cacheKey{
		carrierID:    carrierID,
		flightNumber: flightNumber,
		Date:         Date,
		direction:    direction,
		stationID:    stationID,
	}] = item
}

func (c *innerCache) copyOverlayDeleteOld(cache *innerCache) {
	c.overlay = make(map[cacheKey]cacheItem)
	if cache == nil || cache.overlay == nil {
		return
	}

	for k, v := range cache.overlay {
		if time.Since(k.Date.ToTime(time.UTC)) > 5*24*time.Hour {
			continue
		}
		c.overlay[k] = v
	}
}
