package cache

import (
	"context"
	"time"

	"github.com/cenkalti/backoff/v4"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/logger"
)

type UpdaterConfig struct {
	Name          string
	Renewer       cacheRenewer
	RenewInterval time.Duration
	Ctx           context.Context
}

type cacheRenewer func() error

func RunUpdater(rcf *UpdaterConfig) {
	logger.Logger().Info("Start: Updating", log.String("cache", rcf.Name))
	err := backoff.Retry(
		backoff.Operation(rcf.Renewer),
		backoff.WithMaxRetries(backoff.NewExponentialBackOff(), 10),
	)
	if err != nil {
		logger.Logger().Fatal(
			"Cannot initialize cache",
			log.String("cache", rcf.Name),
			log.Error(err),
		)
	}

	logger.Logger().Info("Stop: Updating", log.String("cache", rcf.Name))
	go func() {
		for {
			time.Sleep(rcf.RenewInterval)
			select {
			case <-rcf.Ctx.Done():
				logger.Logger().Info(
					"Updater is cancelled",
					log.String("name", rcf.Name),
					log.Error(rcf.Ctx.Err()),
				)
				return
			default:
			}
			logger.Logger().Info("Start: Reloading cache", log.String("name", rcf.Name))
			err = backoff.Retry(
				backoff.Operation(rcf.Renewer),
				backoff.WithMaxRetries(backoff.NewExponentialBackOff(), 10),
			)
			if err != nil {
				logger.Logger().Error(
					"Error reloading cache",
					log.String("cache", rcf.Name),
					log.Error(err),
				)
				continue
			}
			logger.Logger().Info("Stop: Reloading cache", log.String("name", rcf.Name))
		}
	}()
}
