package stationcache

import (
	"context"
	"sync"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/logger"
	"a.yandex-team.ru/travel/avia/shared_flights/status_importer/internal/objects/cache"
	"a.yandex-team.ru/travel/avia/shared_flights/status_importer/internal/objects/model"
)

type tStation struct {
	byCode sync.Map
	byID   sync.Map
	config *Config
}

type StationChanProvider interface {
	All() (chan *model.Station, error)
}

type Config struct {
	StationProvider StationChanProvider
	UpdateInterval  time.Duration
}

func New(config *Config) *tStation {
	c := &tStation{
		config: config,
	}
	cache.RunUpdater(&cache.UpdaterConfig{
		Name:          "station",
		Renewer:       c.updater,
		RenewInterval: c.config.UpdateInterval,
		Ctx:           context.Background(),
	})
	return c
}

func (c *tStation) updater() error {
	stationChan, err := c.config.StationProvider.All()
	if err != nil {
		return xerrors.Errorf("station cache updater: %w", err)
	}
	stationsCount := 0
	for station := range stationChan {
		if len(station.Iata) > 0 || len(station.Sirena) > 0 {
			stationsCount++
			if len(station.Iata) > 0 {
				c.byCode.Store(station.Iata, station)
			}
			if len(station.Sirena) > 0 {
				c.byCode.Store(station.Sirena, station)
			}
			c.byID.Store(station.ID, station)
		}
	}
	logger.Logger().Info("fetched stations", log.Int("count", stationsCount))
	return nil
}

func (c *tStation) ByCode(key string) *model.Station {
	v, ok := c.byCode.Load(key)
	if !ok {
		return nil
	}
	if v == nil {
		return nil
	}
	result, typeCheck := v.(*model.Station)
	if !typeCheck {
		return nil
	}
	return result
}

func (c *tStation) ByID(key model.StationID) *model.Station {
	v, ok := c.byID.Load(key)
	if !ok {
		return nil
	}
	if v == nil {
		return nil
	}
	result, typeCheck := v.(*model.Station)
	if !typeCheck {
		return nil
	}
	return result
}
