package stoppointcache

import (
	"context"
	"sync"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/avia/shared_flights/lib/go/logger"
	"a.yandex-team.ru/travel/avia/shared_flights/status_importer/internal/objects/cache"
	"a.yandex-team.ru/travel/avia/shared_flights/status_importer/internal/objects/model"
)

type tStopPoint struct {
	byCode sync.Map
	config *Config
}

type StopPointChanProvider interface {
	All() (chan *model.StopPoint, error)
}

type Config struct {
	StopPointProvider StopPointChanProvider
	UpdateInterval    time.Duration
}

func New(config *Config) *tStopPoint {
	c := &tStopPoint{
		config: config,
	}
	cache.RunUpdater(&cache.UpdaterConfig{
		Name:          "stop-point",
		Renewer:       c.updater,
		RenewInterval: c.config.UpdateInterval,
		Ctx:           context.Background(),
	})
	return c
}

func (c *tStopPoint) updater() error {
	stopPointChan, err := c.config.StopPointProvider.All()
	if err != nil {
		return xerrors.Errorf("stop point cache updater: %w", err)
	}
	stopPointsCount := 0
	for stopPoint := range stopPointChan {
		if len(stopPoint.StationCode) > 0 {
			stopPointsCount++
			c.byCode.Store(stopPoint.StationCode, stopPoint)
		}
		if len(stopPoint.CityCode) > 0 {
			if _, exists := c.byCode.Load(stopPoint.CityCode); !exists {
				c.byCode.Store(stopPoint.CityCode, stopPoint)
			}
		}
	}
	logger.Logger().Info("fetched stop points", log.Int("count", stopPointsCount))
	return nil
}

func (c *tStopPoint) ByCode(key string) *model.StopPoint {
	v, ok := c.byCode.Load(key)
	if !ok {
		return nil
	}
	if v == nil {
		return nil
	}
	result, typeCheck := v.(*model.StopPoint)
	if !typeCheck {
		return nil
	}
	return result
}
